"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TargetGroupMonitoringAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alarms_1 = require("../alarms");
const widgets_1 = require("../widgets");
/**
 * The TargetGroupMonitoringAspect iterates over the target groups and adds monitoring widgets and alarms.
 */
class TargetGroupMonitoringAspect {
    constructor(monitoringFacade) {
        this.monitoringFacade = monitoringFacade;
        this.overriddenConfig = {};
        this.defaultConfig = {
            minHealthyHostsThreshold: 1,
        };
    }
    visit(node) {
        if (!(node instanceof aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationTargetGroup)) {
            return;
        }
        const config = this.readConfig(node);
        const metrics = this.metrics(node);
        this.monitoringFacade.dashboard.addWidgets(...this.widgets(node, config, metrics));
        this.alarms(node, config, metrics).forEach((a) => this.monitoringFacade.addAlarm(a));
    }
    /**
     * Overrides the default configuration for a specific target group.
     * @param node The target group to monitor.
     * @param config The configuration to apply.
     */
    overrideConfig(node, config) {
        this.overriddenConfig[node.node.path] = config;
    }
    readConfig(node) {
        return {
            ...this.defaultConfig,
            ...(this.overriddenConfig[node.node.path] || {}),
        };
    }
    widgets(node, config, metrics) {
        return [
            (0, widgets_1.dashboardSectionTitle)(`TargetGroup ${node.targetGroupName}`),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Response Time',
                left: [metrics.responseTime],
                leftYAxis: widgets_1.dashboardSecondsAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.responseTimeThreshold?.toSeconds({ integral: false }) }]),
                width: 12,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Healthy Hosts',
                left: [metrics.minHealthyHosts],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.minHealthyHostsThreshold }]),
                width: 12,
            }),
        ];
    }
    alarms(node, config, metrics) {
        return (0, alarms_1.buildAlarms)({
            node,
            nodeIdentifier: node.targetGroupName,
            alarms: [
                {
                    alarmId: 'TargetGroup-ResponseTimeAlarm',
                    metric: metrics.responseTime,
                    evaluationPeriods: 5,
                    threshold: config.responseTimeThreshold?.toSeconds({ integral: false }),
                    alarmDescription: `Response time is too high on ${node.targetGroupName}`,
                },
                {
                    alarmId: 'TargetGroup-MinHealthyHostsAlarm',
                    metric: metrics.minHealthyHosts,
                    evaluationPeriods: 5,
                    threshold: config.minHealthyHostsThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `Not enough healthy hosts on ${node.targetGroupName}`,
                },
            ],
        });
    }
    metrics(node) {
        return {
            responseTime: node.metrics.targetResponseTime({
                period: cdk.Duration.minutes(1),
                statistic: 'tm99',
            }),
            minHealthyHosts: node.metrics.healthyHostCount({
                period: cdk.Duration.minutes(1),
            }),
        };
    }
}
exports.TargetGroupMonitoringAspect = TargetGroupMonitoringAspect;
_a = JSII_RTTI_SYMBOL_1;
TargetGroupMonitoringAspect[_a] = { fqn: "@condensetech/cdk-constructs.TargetGroupMonitoringAspect", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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