"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RdsInstanceMonitoringAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alarms_1 = require("../alarms");
const widgets_1 = require("../widgets");
/**
 * The RdsInstanceMonitoringAspect iterates over the RDS instances and adds monitoring widgets and alarms.
 */
class RdsInstanceMonitoringAspect {
    constructor(monitoringFacade) {
        this.monitoringFacade = monitoringFacade;
        this.overriddenConfig = {};
        this.defaultConfig = {
            cpuUtilizationThreshold: 90,
            maxConnectionsThreshold: 50,
            ebsByteBalanceThreshold: 10,
            ebsIoBalanceThreshold: 10,
            freeableMemoryThreshold: cdk.Size.mebibytes(100),
            freeStorageSpaceThreshold: cdk.Size.mebibytes(100),
            readLatencyThreshold: 20,
        };
    }
    visit(node) {
        if (!(node instanceof aws_cdk_lib_1.aws_rds.DatabaseInstance)) {
            return;
        }
        const config = this.readConfig(node);
        const metrics = this.metrics(node);
        this.monitoringFacade.dashboard.addWidgets(...this.widgets(node, config, metrics));
        this.alarms(node, config, metrics).forEach((a) => this.monitoringFacade.addAlarm(a));
    }
    /**
     * Overrides the default configuration for the RDS instance.
     * @param node The RDS instance to monitor.
     * @param config The configuration to apply.
     */
    overrideConfig(node, config) {
        this.overriddenConfig[node.node.path] = config;
    }
    readConfig(node) {
        return {
            ...this.defaultConfig,
            ...(this.overriddenConfig[node.node.path] || {}),
        };
    }
    widgets(node, config, metrics) {
        return [
            (0, widgets_1.dashboardSectionTitle)(`RDS ${node.node.path}`),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'CPU Utilization',
                left: [metrics.cpuUtilization],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.cpuUtilizationThreshold }]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Connections',
                left: [metrics.maxConnections],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.maxConnectionsThreshold }]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Storage and Memory',
                left: [metrics.freeStorageSpace, metrics.freeableMemory],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([
                    { value: config.freeStorageSpaceThreshold?.toBytes(), label: 'Free Storage Space' },
                    { value: config.freeableMemoryThreshold?.toBytes(), label: 'Freeable Memory' },
                ]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'EBS Byte & IO Balance',
                left: [metrics.ebsByteBalance, metrics.ebsIOBalance],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([
                    { value: config.ebsByteBalanceThreshold, label: 'EBS Byte Balance' },
                    { value: config.ebsIoBalanceThreshold, label: 'EBS IO Balance' },
                ]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Read Latency',
                left: [metrics.readLatency],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.readLatencyThreshold }]),
            }),
        ];
    }
    alarms(node, config, metrics) {
        return (0, alarms_1.buildAlarms)({
            node,
            nodeIdentifier: node.instanceIdentifier,
            alarms: [
                {
                    alarmId: 'RdsInstance-CpuUsageAlarm',
                    metric: metrics.cpuUtilization,
                    evaluationPeriods: 5,
                    threshold: config.cpuUtilizationThreshold,
                    alarmDescription: `CPU Utilization high on ${node.instanceIdentifier}`,
                },
                {
                    alarmId: 'RdsInstance-MaxConnectionsAlarm',
                    metric: metrics.maxConnections,
                    evaluationPeriods: 5,
                    threshold: config.maxConnectionsThreshold,
                    alarmDescription: `Max Connections high on ${node.instanceIdentifier}`, // FIXME: This depends on the database
                },
                {
                    alarmId: 'RdsInstance-EBSByteBalanceAlarm',
                    metric: metrics.ebsByteBalance,
                    evaluationPeriods: 3,
                    threshold: config.ebsByteBalanceThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `EBS Byte Balance too low on ${node.instanceIdentifier}`,
                },
                {
                    alarmId: 'RdsInstance-EBSIOBalanceAlarm',
                    metric: metrics.ebsIOBalance,
                    evaluationPeriods: 3,
                    threshold: config.ebsIoBalanceThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `EBS IO Balance too low on ${node.instanceIdentifier}`,
                },
                {
                    alarmId: 'RdsInstance-FreeableMemoryAlarm',
                    metric: metrics.freeableMemory,
                    evaluationPeriods: 15,
                    threshold: config.freeableMemoryThreshold?.toBytes(),
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `Freeable Memory low on ${node.instanceIdentifier}`,
                },
                {
                    alarmId: 'RdsInstance-FreeStorageSpaceAlarm',
                    metric: metrics.freeStorageSpace,
                    evaluationPeriods: 5,
                    threshold: config.freeStorageSpaceThreshold?.toBytes(),
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `Storage Space low on ${node.instanceIdentifier}`,
                },
                {
                    alarmId: 'RdsInstance-ReadLatencyAlarm',
                    metric: metrics.readLatency,
                    evaluationPeriods: 5,
                    threshold: config.readLatencyThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                    alarmDescription: `Read Latency high on ${node.instanceIdentifier}`,
                },
            ],
        });
    }
    metrics(node) {
        return {
            cpuUtilization: node.metricCPUUtilization({
                period: cdk.Duration.minutes(1),
            }),
            maxConnections: node.metricDatabaseConnections({
                period: cdk.Duration.minutes(1),
            }),
            freeableMemory: node.metricFreeableMemory({
                period: cdk.Duration.minutes(1),
            }),
            freeStorageSpace: node.metricFreeStorageSpace({
                period: cdk.Duration.minutes(1),
                statistic: 'Minimum',
            }),
            readLatency: node.metric('ReadLatency', {
                period: cdk.Duration.minutes(1),
                statistic: 'p90',
            }),
            ebsIOBalance: node.metric('EBSIOBalance%', {
                period: cdk.Duration.minutes(1),
            }),
            ebsByteBalance: node.metric('EBSByteBalance%', {
                period: cdk.Duration.minutes(1),
            }),
        };
    }
}
exports.RdsInstanceMonitoringAspect = RdsInstanceMonitoringAspect;
_a = JSII_RTTI_SYMBOL_1;
RdsInstanceMonitoringAspect[_a] = { fqn: "@condensetech/cdk-constructs.RdsInstanceMonitoringAspect", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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