"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateServiceMonitoringAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alarms_1 = require("../alarms");
const widgets_1 = require("../widgets");
/**
 * The FargateServiceMonitoringAspect iterates over the Fargate services and adds monitoring widgets and alarms.
 */
class FargateServiceMonitoringAspect {
    constructor(monitoringFacade) {
        this.monitoringFacade = monitoringFacade;
        this.overriddenConfig = {};
        this.defaultConfig = {
            cpuUtilizationThreshold: 90,
            memoryUtilization: 90,
        };
    }
    visit(node) {
        if (!(node instanceof aws_cdk_lib_1.aws_ecs.FargateService)) {
            return;
        }
        const config = this.readConfig(node);
        const metrics = this.metrics(node);
        this.monitoringFacade.dashboard.addWidgets(...this.widgets(node, config, metrics));
        this.alarms(node, config, metrics).forEach((a) => this.monitoringFacade.addAlarm(a));
    }
    /**
     * Overrides the default configuration for a specific Fargate service.
     * @param node The Fargate service to monitor.
     * @param config The configuration to apply.
     */
    overrideConfig(node, config) {
        this.overriddenConfig[node.node.path] = config;
    }
    readConfig(node) {
        return {
            ...this.defaultConfig,
            ...(this.overriddenConfig[node.node.path] || {}),
        };
    }
    widgets(node, config, metrics) {
        return [
            (0, widgets_1.dashboardSectionTitle)(`ECS Fargate Service ${node.serviceName}`),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'CPU Utilization',
                left: [metrics.cpuUtilization],
                leftYAxis: widgets_1.dashboardSecondsAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.cpuUtilizationThreshold }]),
                width: 12,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Memory Utilization',
                left: [metrics.memoryUtilization],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.memoryUtilization }]),
                width: 12,
            }),
        ];
    }
    alarms(node, config, metrics) {
        return (0, alarms_1.buildAlarms)({
            node,
            nodeIdentifier: node.serviceName,
            alarms: [
                {
                    alarmId: 'FargateService-CPUUtilizationAlarm',
                    metric: metrics.cpuUtilization,
                    evaluationPeriods: 5,
                    threshold: config.cpuUtilizationThreshold,
                    alarmDescription: `CPU Utilization high on ${node.serviceName}`,
                },
                {
                    alarmId: 'FargateService-MemoryUtilizationAlarm',
                    metric: metrics.memoryUtilization,
                    evaluationPeriods: 5,
                    threshold: config.memoryUtilization,
                    alarmDescription: `Memory Utilization high on ${node.serviceName}`,
                },
            ],
        });
    }
    metrics(node) {
        return {
            cpuUtilization: node.metricCpuUtilization({
                period: cdk.Duration.minutes(1),
            }),
            memoryUtilization: node.metricMemoryUtilization({
                period: cdk.Duration.minutes(1),
            }),
        };
    }
}
exports.FargateServiceMonitoringAspect = FargateServiceMonitoringAspect;
_a = JSII_RTTI_SYMBOL_1;
FargateServiceMonitoringAspect[_a] = { fqn: "@condensetech/cdk-constructs.FargateServiceMonitoringAspect", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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