"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Entrypoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_route53_targets_1 = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const application_listener_priority_allocator_1 = require("./application-listener-priority-allocator");
class EntrypointBase extends constructs_1.Construct {
    referenceListener(scope, id) {
        return aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListener.fromApplicationListenerAttributes(scope, id, {
            listenerArn: this.listener.listenerArn,
            securityGroup: aws_cdk_lib_1.aws_ec2.SecurityGroup.fromSecurityGroupId(scope, `${id}-SG`, this.securityGroup.securityGroupId),
        });
    }
    allocateListenerRule(scope, id, props) {
        const listener = this.referenceListener(scope, `${id}-Listener`);
        const priority = this.priorityAllocator.allocatePriority(scope, `${id}-Priority`, {
            priority: props.priority,
        });
        return new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListenerRule(scope, id, {
            ...props,
            listener,
            priority,
        });
    }
}
/**
 * The Entrypoint construct creates an Application Load Balancer (ALB) that serves as the centralized entry point for all applications.
 * This ALB is shared across multiple applications, primarily to optimize infrastructure costs by reducing the need for multiple load balancers.
 * It implements the IEntrypoint interface so that it can be used in other constructs and stacks without requiring to access to the underlying construct.
 *
 * It creates an ALB with:
 * - an HTTP listener that redirects all traffic to HTTPS.
 * - an HTTPS listener that returns a 403 Forbidden response by default.
 * - a custom security group. This allows to expose the security group as a property of the entrypoint construct, making it easier to reference it in other constructs.
 * Finally, it creates the Route 53 A and AAAA record that point to the ALB.
 *
 * When hostedZoneProps is provided, by default this construct creates an HTTPS certificate, bound to the domain name and all subdomains (unless wildcardCertificate is set to false).
 * You can also provide an existing certificate ARN through certificate.certificateArn.
 *
 * When an `entrypointName` is provided, this is used as the name of the ALB and as the prefix for the security group.
 * It is also used to add an additional "Name" tag to the load balancer.
 * This helps to use [ApplicationLoadBalancer#lookup](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_elasticloadbalancingv2.ApplicationLoadBalancer.html#static-fromwbrlookupscope-id-options) to find the load balancer by name.
 */
class Entrypoint extends EntrypointBase {
    static fromLookup(scope, id, props) {
        class Import extends EntrypointBase {
            constructor() {
                super(scope, id);
                if (!props.vpc && !props.vpcLookup) {
                    throw new Error('Either vpc or vpcLookup must be provided.');
                }
                const vpc = props.vpc || aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'Vpc', props.vpcLookup);
                this.securityGroup = aws_cdk_lib_1.aws_ec2.SecurityGroup.fromLookupByName(this, 'SecurityGroup', `${props.entrypointName}-sg`, vpc);
                this.alb = aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer.fromLookup(this, 'Alb', {
                    loadBalancerTags: { Name: props.entrypointName },
                });
                this.domainName = props.domainName ?? this.alb.loadBalancerDnsName;
                this.listener = aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListener.fromLookup(this, 'Listener', {
                    loadBalancerArn: this.alb.loadBalancerArn,
                    listenerProtocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
                });
                this.priorityAllocator = application_listener_priority_allocator_1.ApplicationListenerPriorityAllocator.fromPriorityAllocatorName(this, 'PriorityAllocator', props.entrypointName);
            }
        }
        return new Import();
    }
    static fromAttributes(scope, id, props) {
        if (!props.priorityAllocatorServiceToken && !props.entrypointName) {
            throw new Error('Either entrypointName or priorityAllocatorServiceToken must be provided.');
        }
        class Import extends EntrypointBase {
            constructor() {
                super(scope, id);
                this.securityGroup = aws_cdk_lib_1.aws_ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId);
                this.alb = aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer.fromApplicationLoadBalancerAttributes(this, 'Alb', {
                    loadBalancerArn: props.loadBalancerArn,
                    securityGroupId: props.securityGroupId,
                });
                this.domainName = props.domainName ?? this.alb.loadBalancerDnsName;
                this.listener = aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListener.fromApplicationListenerAttributes(this, 'Listener', {
                    listenerArn: props.listenerArn,
                    securityGroup: this.securityGroup,
                });
                this.priorityAllocator = props.entrypointName
                    ? application_listener_priority_allocator_1.ApplicationListenerPriorityAllocator.fromPriorityAllocatorName(this, 'PriorityAllocator', props.entrypointName)
                    : application_listener_priority_allocator_1.ApplicationListenerPriorityAllocator.fromServiceToken(this, 'PriorityAllocator', props.priorityAllocatorServiceToken);
            }
        }
        return new Import();
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.hostedZone = props.hostedZoneProps
            ? aws_cdk_lib_1.aws_route53.HostedZone.fromHostedZoneAttributes(this, 'HostedZone', props.hostedZoneProps)
            : undefined;
        this.domainName = props.domainName;
        const albCertificates = (props.certificates || [props.certificate ?? {}]).map((cert) => this.createCertificate(cert));
        this.securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.networking.vpc,
            allowAllOutbound: true,
            securityGroupName: props.securityGroupName ??
                props.entrypointSecurityGroupName ??
                (props.entrypointName ? `${props.entrypointName}-sg` : undefined),
        });
        const alb = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, 'Alb', {
            vpc: props.networking.vpc,
            internetFacing: true,
            securityGroup: this.securityGroup,
            loadBalancerName: props.entrypointName,
        });
        if (props.entrypointName) {
            cdk.Tags.of(alb).add('Name', props.entrypointName);
        }
        if (props.logsBucket) {
            alb.logAccessLogs(props.logsBucket);
        }
        this.alb = alb;
        this.alb.addListener('HTTP', {
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            defaultAction: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.redirect({
                port: '443',
                protocol: 'HTTPS',
            }),
        });
        this.listener = this.alb.addListener('HTTPS', {
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
            certificates: albCertificates,
            defaultAction: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.fixedResponse(403, {
                messageBody: 'Forbidden',
            }),
        });
        if (this.hostedZone) {
            new aws_cdk_lib_1.aws_route53.ARecord(this, 'AlbRecord', {
                target: aws_cdk_lib_1.aws_route53.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(this.alb)),
                zone: this.hostedZone,
                recordName: this.domainName.replace(`.${this.hostedZone.zoneName}`, ''),
            });
            new aws_cdk_lib_1.aws_route53.AaaaRecord(this, 'AlbRecordIPv6', {
                target: aws_cdk_lib_1.aws_route53.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(this.alb)),
                zone: this.hostedZone,
                recordName: this.domainName.replace(`.${this.hostedZone.zoneName}`, ''),
            });
        }
        this.priorityAllocator = new application_listener_priority_allocator_1.ApplicationListenerPriorityAllocator(this, 'PriorityAllocator', {
            listener: this.listener,
            priorityAllocatorName: props.entrypointName,
            ...props.priorityAllocator,
        });
    }
    // FIXME: This implementation allows the caller to build an instance which will issue multiple certificates for the same hosted zone.
    createCertificate(props) {
        if (props.certificate) {
            if (props.certificateArn) {
                cdk.Annotations.of(this).addError(`Both certificate (${props.certificate}) and certificateArn (${props.certificateArn}) are provided. Choose one.`);
            }
            if (props.wildcardCertificate) {
                cdk.Annotations.of(this).addError(`wildcardCertificate cannot be set when a certificate (${props.certificate}) is provided.`);
            }
            return props.certificate;
        }
        if (props.certificateArn) {
            if (props.wildcardCertificate) {
                cdk.Annotations.of(this).addError(`wildcardCertificate cannot be set when a certificateArn (${props.certificateArn}) is provided.`);
            }
            return aws_cdk_lib_1.aws_certificatemanager.Certificate.fromCertificateArn(this, 'Certificate', props.certificateArn);
        }
        if (this.hostedZone) {
            const subjectAlternativeNames = props.wildcardCertificate === false ? undefined : [`*.${this.hostedZone.zoneName}`];
            return new aws_cdk_lib_1.aws_certificatemanager.Certificate(this, 'Certificate', {
                domainName: this.domainName,
                subjectAlternativeNames,
                validation: aws_cdk_lib_1.aws_certificatemanager.CertificateValidation.fromDns(this.hostedZone),
            });
        }
        throw new Error('Hosted Zone Props are required when certificate must be automatically provisioned.');
    }
}
exports.Entrypoint = Entrypoint;
_a = JSII_RTTI_SYMBOL_1;
Entrypoint[_a] = { fqn: "@condensetech/cdk-constructs.Entrypoint", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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