"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The DatabaseInstance construct creates an RDS database instance.
 * Under the hood, it creates a [rds.DatabaseInstance](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#starting-an-instance-database) construct.
 * It implements the IDatabase interface so that it can be used in other constructs and stacks without requiring to access to the underlying construct.
 *
 * It also applies the following changes to the default behavior:
 * - A [rds.ParameterGroup](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#parameter-groups) specific for the cluster is always defined.
 *   By using a custom parameter group instead of relying on the default one, a later change in the parameter group's parameters wouldn't require a replace of the cluster.
 * - The credentials secret name is created after the construct's path. This way, the secret name is more readable and, when working with multiple stacks, can be easily inferred without having to rely on Cloudformation exports.
 * - It defaults the storage type to GP3 when not specified.
 * - It defaults the allocated storage to the minimum storage of 20 GB when not specified.
 * - The default instance type is set to t3.small.
 * - The storage is always encrypted.
 * - If the networking configuration includes a bastion host, the database allows connections from the bastion host.
 * - The default security group name is `${construct.node.path}-sg`. This allows for easier lookups when working with multiple stacks.
 */
class DatabaseInstance extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const removalPolicy = props.removalPolicy ?? cdk.RemovalPolicy.RETAIN;
        const parameterGroup = new aws_cdk_lib_1.aws_rds.ParameterGroup(this, 'ParameterGroup', {
            engine: props.engine,
            description: this.node.path,
            removalPolicy: [cdk.RemovalPolicy.DESTROY, cdk.RemovalPolicy.RETAIN].includes(removalPolicy)
                ? removalPolicy
                : cdk.RemovalPolicy.DESTROY,
        });
        const instanceType = props.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE3, aws_cdk_lib_1.aws_ec2.InstanceSize.SMALL);
        this.credentialsSecretName = props.credentialsSecretName ?? `${this.node.path}/secret`;
        const credentials = aws_cdk_lib_1.aws_rds.Credentials.fromUsername(props.credentialsUsername ?? 'db_user', {
            secretName: this.credentialsSecretName,
        });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.networking.vpc,
            allowAllOutbound: true,
            securityGroupName: props.securityGroupName ?? `${this.node.path.replace(/\//g, '-')}-sg`,
        });
        this.resource = new aws_cdk_lib_1.aws_rds.DatabaseInstance(this, 'DB', {
            instanceIdentifier: props.instanceIdentifier,
            vpc: props.networking.vpc,
            vpcSubnets: props.networking.isolatedSubnets,
            engine: props.engine,
            databaseName: props.databaseName,
            credentials,
            parameterGroup: parameterGroup,
            instanceType,
            allocatedStorage: props.allocatedStorage ?? 20,
            storageType: props.storageType ?? aws_cdk_lib_1.aws_rds.StorageType.GP3,
            multiAz: props.multiAz ?? false,
            securityGroups: [securityGroup],
            storageEncrypted: true,
            backupRetention: props.backupRetention,
            removalPolicy,
        });
        if (props.networking.bastionHost) {
            this.resource.connections.allowDefaultPortFrom(props.networking.bastionHost);
        }
        this.endpoint = this.resource.instanceEndpoint;
    }
    get connections() {
        return this.resource.connections;
    }
    fetchSecret(scope, id = 'DatabaseSecret') {
        return aws_cdk_lib_1.aws_secretsmanager.Secret.fromSecretNameV2(scope, id, this.credentialsSecretName);
    }
}
exports.DatabaseInstance = DatabaseInstance;
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_a] = { fqn: "@condensetech/cdk-constructs.DatabaseInstance", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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