"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuroraCluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The AuroraCluster Construct creates an opinionated Aurora Cluster. Under the hood, it creates a [rds.DatabaseCluster](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#starting-a-clustered-database) construct.
 * It implements the IDatabase interface so that it can be used in other constructs and stacks without requiring to access to the underlying construct.
 *
 * It also applies the following changes to the default behavior:
 * - A [rds.ParameterGroup](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#parameter-groups) specific for the cluster is always defined.
 *   By using a custom parameter group instead of relying on the default one, a later change in the parameter group's parameters wouldn't require a replace of the cluster.
 * - The credentials secret name is created after the construct's path. This way, the secret name is more readable and, when working with multiple stacks, can be easily inferred without having to rely on Cloudformation exports.
 * - The default instance type for the writer instance is set to a minimum instance type based on the engine type.
 * - The storage is always encrypted.
 * - If the networking configuration includes a bastion host, the cluster allows connections from the bastion host.
 * - The default security group name is `${construct.node.path}-sg`. This allows for easier lookups when working with multiple stacks.
 */
class AuroraCluster extends constructs_1.Construct {
    /**
     * Returns the minimum instance type supported by the Aurora cluster based on the engine type.
     * This method is used to set the default instance type for the writer instance if not otherwise specified.
     *
     * @param engine The engine type of the Aurora cluster.
     * @returns The minimum instance type supported by the Aurora cluster based on the engine type.
     */
    static minimumInstanceType(engine) {
        return engine.engineType === 'aurora-postgresql'
            ? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE3, aws_cdk_lib_1.aws_ec2.InstanceSize.MEDIUM)
            : aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE3, aws_cdk_lib_1.aws_ec2.InstanceSize.SMALL);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const removalPolicy = props.removalPolicy ?? cdk.RemovalPolicy.RETAIN;
        this.parameterGroup = new aws_cdk_lib_1.aws_rds.ParameterGroup(this, 'ParameterGroup', {
            engine: props.engine,
            description: this.node.path,
            removalPolicy: [cdk.RemovalPolicy.DESTROY, cdk.RemovalPolicy.RETAIN].includes(removalPolicy)
                ? removalPolicy
                : cdk.RemovalPolicy.DESTROY,
            parameters: props.parameters,
        });
        const backup = props.backupRetention ? { retention: props.backupRetention } : undefined;
        const credentials = aws_cdk_lib_1.aws_rds.Credentials.fromUsername(props.credentialsUsername ?? 'db_user', {
            secretName: props.credentialsSecretName ?? `${this.node.path}/secret`,
        });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.networking.vpc,
            allowAllOutbound: true,
            securityGroupName: props.securityGroupName ?? `${this.node.path.replace(/\//g, '-')}-sg`,
        });
        this.resource = new aws_cdk_lib_1.aws_rds.DatabaseCluster(this, 'DB', {
            clusterIdentifier: props.clusterIdentifier,
            engine: props.engine,
            credentials,
            writer: props.writer ??
                aws_cdk_lib_1.aws_rds.ClusterInstance.provisioned('ClusterInstance', {
                    instanceType: AuroraCluster.minimumInstanceType(props.engine),
                }),
            readers: props.readers,
            vpc: props.networking.vpc,
            vpcSubnets: props.networking.isolatedSubnets,
            defaultDatabaseName: props.databaseName,
            parameterGroup: this.parameterGroup,
            storageEncrypted: true,
            securityGroups: [securityGroup],
            removalPolicy,
            backup,
        });
        if (props.networking.bastionHost) {
            this.resource.connections.allowDefaultPortFrom(props.networking.bastionHost);
        }
        this.endpoint = this.resource.clusterEndpoint;
    }
    get connections() {
        return this.resource.connections;
    }
    fetchSecret(scope, id = 'DatabaseSecret') {
        return aws_cdk_lib_1.aws_secretsmanager.Secret.fromSecretNameV2(scope, id, `${this.node.path}/secret`);
    }
}
exports.AuroraCluster = AuroraCluster;
_a = JSII_RTTI_SYMBOL_1;
AuroraCluster[_a] = { fqn: "@condensetech/cdk-constructs.AuroraCluster", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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