"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListenerPriorityAllocator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class ApplicationListenerPriorityAllocatorBase extends constructs_1.Construct {
    /**
     * Allocates the priority of an application listener rule
     * @param scope The scope of the construct.
     * @param id The ID of the listener rule to allocate the priority to.
     * @param props
     * @returns The allocated priority.
     */
    allocatePriority(scope, id, props) {
        const cr = new cdk.CustomResource(scope, id, {
            serviceToken: this.serviceToken,
            properties: {
                rulePath: `${cdk.Stack.of(scope).region}/${cdk.Stack.of(scope).stackName}/${scope.node.path}/${id}`,
                priority: props.priority,
            },
        });
        return cdk.Token.asNumber(cr.getAtt('priority'));
    }
}
/**
 * This custom resource allows to generate unique priorities for application listener rules.
 *
 * Consumers can allocate a priority to a listener rule by calling the `allocatePriority` method, ensuring that:
 * - if no priority is set, one will be generated
 * - if a priority is set, an error will be thrown if the priority is already taken
 */
class ApplicationListenerPriorityAllocator extends ApplicationListenerPriorityAllocatorBase {
    static fromPriorityAllocatorName(scope, id, priorityAllocatorName) {
        const serviceToken = aws_cdk_lib_1.aws_ssm.StringParameter.valueForStringParameter(scope, `/service-tokens/priority-allocators/${priorityAllocatorName}`);
        return this.fromServiceToken(scope, id, serviceToken);
    }
    static fromServiceToken(scope, id, serviceToken) {
        class Imported extends ApplicationListenerPriorityAllocatorBase {
            constructor() {
                super(...arguments);
                this.serviceToken = serviceToken;
            }
        }
        return new Imported(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const table = new aws_cdk_lib_1.aws_dynamodb.Table(this, 'Table', {
            tableName: `app_listener_priorities_${cdk.Names.uniqueResourceName(props.listener, { maxLength: 220 })}`,
            partitionKey: { name: 'pk', type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            removalPolicy: props.removalPolicy ?? cdk.RemovalPolicy.DESTROY,
        });
        const onEventHandler = new aws_cdk_lib_1.aws_lambda_nodejs.NodejsFunction(this, 'handler', {
            functionName: `app_listener_priority_alloc_${cdk.Names.uniqueResourceName(props.listener, { maxLength: 36 })}`,
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_20_X,
            environment: {
                LISTENER_ARN: props.listener.listenerArn,
                TABLE_NAME: table.tableName,
                PRIORITY_INIT: props?.priorityInitialValue?.toString() ?? '1',
            },
        });
        table.grantReadWriteData(onEventHandler);
        const provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'Provider', {
            onEventHandler,
        });
        this.serviceToken = provider.serviceToken;
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'ServiceTokenParameter', {
            stringValue: this.serviceToken,
            parameterName: `/service-tokens/priority-allocators/${props.priorityAllocatorName ?? this.node.addr}`,
        });
        new cdk.CfnOutput(this, 'ServiceToken', { value: this.serviceToken });
    }
}
exports.ApplicationListenerPriorityAllocator = ApplicationListenerPriorityAllocator;
_a = JSII_RTTI_SYMBOL_1;
ApplicationListenerPriorityAllocator[_a] = { fqn: "@condensetech/cdk-constructs.ApplicationListenerPriorityAllocator", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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