#
# Copyright (c) nexB Inc. and others. All rights reserved.
# ScanCode is a trademark of nexB Inc.
# SPDX-License-Identifier: Apache-2.0
# See http://www.apache.org/licenses/LICENSE-2.0 for the license text.
# See https://github.com/nexB/scancode-toolkit for support or download.
# See https://aboutcode.org for more information about nexB OSS projects.
#

from itertools import islice

from commoncode.datautils import Boolean
from commoncode.text import toascii
from plugincode.scan import ScanPlugin
from plugincode.scan import scan_impl
from commoncode.cliutils import PluggableCommandLineOption
from commoncode.cliutils import OTHER_SCAN_GROUP
import typecode.contenttype

"""
Tag files as generated code based on conspicuous strings.
"""

# Tracing flag
TRACE = False


def logger_debug(*args):
    pass


if TRACE:
    import logging
    import sys

    logger = logging.getLogger(__name__)
    logging.basicConfig(stream=sys.stdout)
    logger.setLevel(logging.DEBUG)

    def logger_debug(*args):
        return logger.debug(' '.join(isinstance(a, str) and a or repr(a) for a in args))


@scan_impl
class GeneratedCodeDetector(ScanPlugin):
    """
    Tag a file as generated.
    """
    resource_attributes = dict(is_generated=Boolean(
        help='True if this file is likely an automatically generated file.'))

    sort_order = 50

    options = [
        PluggableCommandLineOption(('--generated',),
            is_flag=True, default=False,
            help='Classify automatically generated code files with a flag.',
            help_group=OTHER_SCAN_GROUP,
            sort_order=50,
        )
    ]

    def is_enabled(self, generated, **kwargs):
        return generated

    def get_scanner(self, **kwargs):
        return generated_scanner


def generated_scanner(location, **kwargs):
    is_generated = False
    for _clue in get_generated_code_hint(location):
        # TODO: consider returning the "clue"
        is_generated = True
        break
    return dict(is_generated=is_generated)


GENERATED_KEYWORDS_LOWERED = tuple(g.lower() for g in (
    'generated by',
    'auto-generated',
    'automatically generated',
    # Apache Axis
    'auto-generated from wsdl',
    # jni javahl and others
    'do not edit this file',
    # jni javahl
    'it is machine generated',
    'by hibernate tools',
    'generated from idl',

    # castor generated files
    'following schema fragment specifies the',

    # Tomcat JSPC
    'automatically created by',

    # in GNU Classpath
    'this file was automatically generated by gnu.localegen from cldr',
    'this document is automatically generated by gnu.supplementgen',

    # linux kernel/u-boot
    'this was automagically generated from',

    # angular
    'this code is generated',
    'this code is generated - do not modify',

    # cython
    'generated by cython',
    # sqlite amalgamation
    'this file is an amalgamation of many separate c source files from sqlite',

    # various generated or last generated:
    'generated on',
    'last generated on',

    # in freepascal unicode
    "this is an automatically created file",

    # generated by Postgres ECPG sql to c preprocessor
    'processed by ecpg (regression mode)',
    'these include files are added by the preprocessor',

    'this readme is generated, please do not update',

    'this file was automatically generated by',
    'any changes will be lost if this file is regenerated',

    # yarn lock files
    'this is an autogenerated file. do not edit this file directly'

    'this file is generated by',
    'microsoft visual c++ generated include file',

    # OpenJDK:
    'generated by the idl-to-java compiler',
    'this file was mechanically generated: do not edit!',
    'generated by mc.java version 1.0, do not edit by hand!',
    'generated from input file',
    'generated content - do not edit',
    'generators: org.graalvm',

    'Generated by GNU Autoconf',

    'This file has been generated by the GUI designer. Do not modify',

    'file is generated by gopy gen. do not edit.',

    # https://github.com/Microsoft/azure-devops-python-api/blob/0d9537016bd45cf7f2140433c0ec54b44768f726/vsts/vsts/licensing/v4_1/licensing_client.py
    'generated file, do not edit',
    'changes may cause incorrect behavior and will be lost if the code is regenerated.',
    'this file is generated. best not to modify it, as it will likely be overwritten.',

    # in Ogg/Vorbis
    'function: static codebooks autogenerated by',

    # yarn lock
    'this is an autogenerated file. do not edit this file directly.',
    'this file has been automatically created by',
    'please do not edit this file, all changes will be lost',

    # thrift
    'this is an automatically-generated file.',
    'it could get re-generated if the allow_idl_generation flag is on',

    # aws sdk ruby
    'warning about generated code',
    'this file is generated. see the contributing guide for more information',
    'this file is generated',

    # seen in Go files:
    'code generated file. do not edit',
    'code generated by running "go generate". do not edit.'

    # protoc
    'generated by the protocol buffer compiler. do not edit!',
    'generated by the protocol buffer compiler.',
))


def get_generated_code_hint(
    location,
    max_lines=150,
    generated_keywords=GENERATED_KEYWORDS_LOWERED
):
    """
    Return a line of extracted text from a file if that file is likely
    generated source code.

    for each of the first few lines of a source code file
      if generated keywords are found in the line as lowercase
         yield the line text as a 'potentially_ generated' annotation
    """
    T = typecode.contenttype.get_type(location)
    if not T.is_text:
        return
    with open(location, 'rb') as filein:
        for line in islice(filein, max_lines):
            text = toascii(line.strip()).lower()
            if any(kw in text.lower() for kw in generated_keywords):
                # yield only the first 100 chars..
                yield text[:100]

