"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// CDK parameters
const STACK_NAME = 'aws-cdk-dynamodb';
// DynamoDB table parameters
const TABLE = 'Table';
const TABLE_WITH_GLOBAL_AND_LOCAL_SECONDARY_INDEX = 'TableWithGlobalAndLocalSecondaryIndex';
const TABLE_WITH_GLOBAL_SECONDARY_INDEX = 'TableWithGlobalSecondaryIndex';
const TABLE_WITH_LOCAL_SECONDARY_INDEX = 'TableWithLocalSecondaryIndex';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_TEST_CASE_1 = 'GSI-PartitionKeyOnly';
const GSI_TEST_CASE_2 = 'GSI-PartitionAndSortKeyWithReadAndWriteCapacity';
const GSI_TEST_CASE_3 = 'GSI-ProjectionTypeKeysOnly';
const GSI_TEST_CASE_4 = 'GSI-ProjectionTypeInclude';
const GSI_TEST_CASE_5 = 'GSI-InverseTableKeySchema';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.NUMBER };
const GSI_NON_KEY = [];
for (let i = 0; i < 10; i++) { // 'A' to 'J'
    GSI_NON_KEY.push(String.fromCharCode(65 + i));
}
// DynamoDB local secondary index parameters
const LSI_TEST_CASE_1 = 'LSI-PartitionAndSortKey';
const LSI_TEST_CASE_2 = 'LSI-PartitionAndTableSortKey';
const LSI_TEST_CASE_3 = 'LSI-ProjectionTypeKeysOnly';
const LSI_TEST_CASE_4 = 'LSI-ProjectionTypeInclude';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = [];
for (let i = 0; i < 10; i++) { // 'K' to 'T'
    LSI_NON_KEY.push(String.fromCharCode(75 + i));
}
const app = new core_1.App();
const stack = new core_1.Stack(app, STACK_NAME);
const table = new lib_1.Table(stack, TABLE, {
    partitionKey: TABLE_PARTITION_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
});
const tableWithGlobalAndLocalSecondaryIndex = new lib_1.Table(stack, TABLE_WITH_GLOBAL_AND_LOCAL_SECONDARY_INDEX, {
    pointInTimeRecovery: true,
    encryption: lib_1.TableEncryption.AWS_MANAGED,
    stream: lib_1.StreamViewType.KEYS_ONLY,
    timeToLiveAttribute: 'timeToLive',
    partitionKey: TABLE_PARTITION_KEY,
    sortKey: TABLE_SORT_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
tableWithGlobalAndLocalSecondaryIndex.node.applyAspect(new core_1.Tag('Environment', 'Production'));
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_1,
    partitionKey: GSI_PARTITION_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_2,
    partitionKey: GSI_PARTITION_KEY,
    sortKey: GSI_SORT_KEY,
    readCapacity: 10,
    writeCapacity: 10,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_3,
    partitionKey: GSI_PARTITION_KEY,
    sortKey: GSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.KEYS_ONLY,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_4,
    partitionKey: GSI_PARTITION_KEY,
    sortKey: GSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.INCLUDE,
    nonKeyAttributes: GSI_NON_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_5,
    partitionKey: TABLE_SORT_KEY,
    sortKey: TABLE_PARTITION_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_2,
    sortKey: LSI_SORT_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_1,
    sortKey: TABLE_SORT_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_3,
    sortKey: LSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.KEYS_ONLY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_4,
    sortKey: LSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.INCLUDE,
    nonKeyAttributes: LSI_NON_KEY,
});
const encryptionKey = new kms.Key(stack, 'Key', {
    enableKeyRotation: true,
});
const tableWithGlobalSecondaryIndex = new lib_1.Table(stack, TABLE_WITH_GLOBAL_SECONDARY_INDEX, {
    partitionKey: TABLE_PARTITION_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryptionKey,
});
tableWithGlobalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_1,
    partitionKey: GSI_PARTITION_KEY,
});
const tableWithLocalSecondaryIndex = new lib_1.Table(stack, TABLE_WITH_LOCAL_SECONDARY_INDEX, {
    partitionKey: TABLE_PARTITION_KEY,
    sortKey: TABLE_SORT_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryption: lib_1.TableEncryption.DEFAULT,
});
tableWithLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_1,
    sortKey: LSI_SORT_KEY,
});
const role = new iam.Role(stack, 'Role', {
    assumedBy: new iam.ServicePrincipal('sqs.amazonaws.com'),
});
table.grantReadData(role);
tableWithGlobalAndLocalSecondaryIndex.grantReadData(role);
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZHluYW1vZGIuc3NlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW50ZWcuZHluYW1vZGIuc3NlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBK0Q7QUFDL0QsZ0NBQTBHO0FBRTFHLGlCQUFpQjtBQUNqQixNQUFNLFVBQVUsR0FBRyxrQkFBa0IsQ0FBQztBQUV0Qyw0QkFBNEI7QUFDNUIsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDO0FBQ3RCLE1BQU0sMkNBQTJDLEdBQUcsdUNBQXVDLENBQUM7QUFDNUYsTUFBTSxpQ0FBaUMsR0FBRywrQkFBK0IsQ0FBQztBQUMxRSxNQUFNLGdDQUFnQyxHQUFHLDhCQUE4QixDQUFDO0FBQ3hFLE1BQU0sbUJBQW1CLEdBQWMsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDO0FBQ3ZGLE1BQU0sY0FBYyxHQUFjLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztBQUVsRiw2Q0FBNkM7QUFDN0MsTUFBTSxlQUFlLEdBQUcsc0JBQXNCLENBQUM7QUFDL0MsTUFBTSxlQUFlLEdBQUcsaURBQWlELENBQUM7QUFDMUUsTUFBTSxlQUFlLEdBQUcsNEJBQTRCLENBQUM7QUFDckQsTUFBTSxlQUFlLEdBQUcsMkJBQTJCLENBQUM7QUFDcEQsTUFBTSxlQUFlLEdBQUcsMkJBQTJCLENBQUM7QUFDcEQsTUFBTSxpQkFBaUIsR0FBYyxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFLENBQUM7QUFDeEYsTUFBTSxZQUFZLEdBQWMsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDO0FBQ25GLE1BQU0sV0FBVyxHQUFhLEVBQUUsQ0FBQztBQUNqQyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsYUFBYTtJQUMxQyxXQUFXLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FDL0M7QUFFRCw0Q0FBNEM7QUFDNUMsTUFBTSxlQUFlLEdBQUcseUJBQXlCLENBQUM7QUFDbEQsTUFBTSxlQUFlLEdBQUcsOEJBQThCLENBQUM7QUFDdkQsTUFBTSxlQUFlLEdBQUcsNEJBQTRCLENBQUM7QUFDckQsTUFBTSxlQUFlLEdBQUcsMkJBQTJCLENBQUM7QUFDcEQsTUFBTSxZQUFZLEdBQWMsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDO0FBQ25GLE1BQU0sV0FBVyxHQUFhLEVBQUUsQ0FBQztBQUNqQyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsYUFBYTtJQUMxQyxXQUFXLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FDL0M7QUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0FBRXRCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztBQUV6QyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO0lBQ3BDLFlBQVksRUFBRSxtQkFBbUI7SUFDakMsYUFBYSxFQUFFLG9CQUFhLENBQUMsT0FBTztJQUNwQyxVQUFVLEVBQUUscUJBQWUsQ0FBQyxnQkFBZ0I7Q0FDN0MsQ0FBQyxDQUFDO0FBRUgsTUFBTSxxQ0FBcUMsR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsMkNBQTJDLEVBQUU7SUFDMUcsbUJBQW1CLEVBQUUsSUFBSTtJQUN6QixVQUFVLEVBQUUscUJBQWUsQ0FBQyxXQUFXO0lBQ3ZDLE1BQU0sRUFBRSxvQkFBYyxDQUFDLFNBQVM7SUFDaEMsbUJBQW1CLEVBQUUsWUFBWTtJQUNqQyxZQUFZLEVBQUUsbUJBQW1CO0lBQ2pDLE9BQU8sRUFBRSxjQUFjO0lBQ3ZCLGFBQWEsRUFBRSxvQkFBYSxDQUFDLE9BQU87Q0FDckMsQ0FBQyxDQUFDO0FBRUgscUNBQXFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFVBQUcsQ0FBQyxhQUFhLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztBQUM3RixxQ0FBcUMsQ0FBQyx1QkFBdUIsQ0FBQztJQUM1RCxTQUFTLEVBQUUsZUFBZTtJQUMxQixZQUFZLEVBQUUsaUJBQWlCO0NBQ2hDLENBQUMsQ0FBQztBQUNILHFDQUFxQyxDQUFDLHVCQUF1QixDQUFDO0lBQzVELFNBQVMsRUFBRSxlQUFlO0lBQzFCLFlBQVksRUFBRSxpQkFBaUI7SUFDL0IsT0FBTyxFQUFFLFlBQVk7SUFDckIsWUFBWSxFQUFFLEVBQUU7SUFDaEIsYUFBYSxFQUFFLEVBQUU7Q0FDbEIsQ0FBQyxDQUFDO0FBQ0gscUNBQXFDLENBQUMsdUJBQXVCLENBQUM7SUFDNUQsU0FBUyxFQUFFLGVBQWU7SUFDMUIsWUFBWSxFQUFFLGlCQUFpQjtJQUMvQixPQUFPLEVBQUUsWUFBWTtJQUNyQixjQUFjLEVBQUUsb0JBQWMsQ0FBQyxTQUFTO0NBQ3pDLENBQUMsQ0FBQztBQUNILHFDQUFxQyxDQUFDLHVCQUF1QixDQUFDO0lBQzVELFNBQVMsRUFBRSxlQUFlO0lBQzFCLFlBQVksRUFBRSxpQkFBaUI7SUFDL0IsT0FBTyxFQUFFLFlBQVk7SUFDckIsY0FBYyxFQUFFLG9CQUFjLENBQUMsT0FBTztJQUN0QyxnQkFBZ0IsRUFBRSxXQUFXO0NBQzlCLENBQUMsQ0FBQztBQUNILHFDQUFxQyxDQUFDLHVCQUF1QixDQUFDO0lBQzVELFNBQVMsRUFBRSxlQUFlO0lBQzFCLFlBQVksRUFBRSxjQUFjO0lBQzVCLE9BQU8sRUFBRSxtQkFBbUI7Q0FDN0IsQ0FBQyxDQUFDO0FBRUgscUNBQXFDLENBQUMsc0JBQXNCLENBQUM7SUFDM0QsU0FBUyxFQUFFLGVBQWU7SUFDMUIsT0FBTyxFQUFFLFlBQVk7Q0FDdEIsQ0FBQyxDQUFDO0FBQ0gscUNBQXFDLENBQUMsc0JBQXNCLENBQUM7SUFDM0QsU0FBUyxFQUFFLGVBQWU7SUFDMUIsT0FBTyxFQUFFLGNBQWM7Q0FDeEIsQ0FBQyxDQUFDO0FBQ0gscUNBQXFDLENBQUMsc0JBQXNCLENBQUM7SUFDM0QsU0FBUyxFQUFFLGVBQWU7SUFDMUIsT0FBTyxFQUFFLFlBQVk7SUFDckIsY0FBYyxFQUFFLG9CQUFjLENBQUMsU0FBUztDQUN6QyxDQUFDLENBQUM7QUFDSCxxQ0FBcUMsQ0FBQyxzQkFBc0IsQ0FBQztJQUMzRCxTQUFTLEVBQUUsZUFBZTtJQUMxQixPQUFPLEVBQUUsWUFBWTtJQUNyQixjQUFjLEVBQUUsb0JBQWMsQ0FBQyxPQUFPO0lBQ3RDLGdCQUFnQixFQUFFLFdBQVc7Q0FDOUIsQ0FBQyxDQUFDO0FBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7SUFDOUMsaUJBQWlCLEVBQUUsSUFBSTtDQUN4QixDQUFDLENBQUM7QUFFSCxNQUFNLDZCQUE2QixHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxpQ0FBaUMsRUFBRTtJQUN4RixZQUFZLEVBQUUsbUJBQW1CO0lBQ2pDLGFBQWEsRUFBRSxvQkFBYSxDQUFDLE9BQU87SUFDcEMsYUFBYTtDQUNkLENBQUMsQ0FBQztBQUNILDZCQUE2QixDQUFDLHVCQUF1QixDQUFDO0lBQ3BELFNBQVMsRUFBRSxlQUFlO0lBQzFCLFlBQVksRUFBRSxpQkFBaUI7Q0FDaEMsQ0FBQyxDQUFDO0FBRUgsTUFBTSw0QkFBNEIsR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsZ0NBQWdDLEVBQUU7SUFDdEYsWUFBWSxFQUFFLG1CQUFtQjtJQUNqQyxPQUFPLEVBQUUsY0FBYztJQUN2QixhQUFhLEVBQUUsb0JBQWEsQ0FBQyxPQUFPO0lBQ3BDLFVBQVUsRUFBRSxxQkFBZSxDQUFDLE9BQU87Q0FDcEMsQ0FBQyxDQUFDO0FBRUgsNEJBQTRCLENBQUMsc0JBQXNCLENBQUM7SUFDbEQsU0FBUyxFQUFFLGVBQWU7SUFDMUIsT0FBTyxFQUFFLFlBQVk7Q0FDdEIsQ0FBQyxDQUFDO0FBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7SUFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO0NBQ3pELENBQUMsQ0FBQztBQUNILEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDMUIscUNBQXFDLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBRTFELEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCB7IEFwcCwgUmVtb3ZhbFBvbGljeSwgU3RhY2ssIFRhZyB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQXR0cmlidXRlLCBBdHRyaWJ1dGVUeXBlLCBQcm9qZWN0aW9uVHlwZSwgU3RyZWFtVmlld1R5cGUsIFRhYmxlLCBUYWJsZUVuY3J5cHRpb24gfSBmcm9tICcuLi9saWInO1xuXG4vLyBDREsgcGFyYW1ldGVyc1xuY29uc3QgU1RBQ0tfTkFNRSA9ICdhd3MtY2RrLWR5bmFtb2RiJztcblxuLy8gRHluYW1vREIgdGFibGUgcGFyYW1ldGVyc1xuY29uc3QgVEFCTEUgPSAnVGFibGUnO1xuY29uc3QgVEFCTEVfV0lUSF9HTE9CQUxfQU5EX0xPQ0FMX1NFQ09OREFSWV9JTkRFWCA9ICdUYWJsZVdpdGhHbG9iYWxBbmRMb2NhbFNlY29uZGFyeUluZGV4JztcbmNvbnN0IFRBQkxFX1dJVEhfR0xPQkFMX1NFQ09OREFSWV9JTkRFWCA9ICdUYWJsZVdpdGhHbG9iYWxTZWNvbmRhcnlJbmRleCc7XG5jb25zdCBUQUJMRV9XSVRIX0xPQ0FMX1NFQ09OREFSWV9JTkRFWCA9ICdUYWJsZVdpdGhMb2NhbFNlY29uZGFyeUluZGV4JztcbmNvbnN0IFRBQkxFX1BBUlRJVElPTl9LRVk6IEF0dHJpYnV0ZSA9IHsgbmFtZTogJ2hhc2hLZXknLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9O1xuY29uc3QgVEFCTEVfU09SVF9LRVk6IEF0dHJpYnV0ZSA9IHsgbmFtZTogJ3NvcnRLZXknLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLk5VTUJFUiB9O1xuXG4vLyBEeW5hbW9EQiBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4IHBhcmFtZXRlcnNcbmNvbnN0IEdTSV9URVNUX0NBU0VfMSA9ICdHU0ktUGFydGl0aW9uS2V5T25seSc7XG5jb25zdCBHU0lfVEVTVF9DQVNFXzIgPSAnR1NJLVBhcnRpdGlvbkFuZFNvcnRLZXlXaXRoUmVhZEFuZFdyaXRlQ2FwYWNpdHknO1xuY29uc3QgR1NJX1RFU1RfQ0FTRV8zID0gJ0dTSS1Qcm9qZWN0aW9uVHlwZUtleXNPbmx5JztcbmNvbnN0IEdTSV9URVNUX0NBU0VfNCA9ICdHU0ktUHJvamVjdGlvblR5cGVJbmNsdWRlJztcbmNvbnN0IEdTSV9URVNUX0NBU0VfNSA9ICdHU0ktSW52ZXJzZVRhYmxlS2V5U2NoZW1hJztcbmNvbnN0IEdTSV9QQVJUSVRJT05fS0VZOiBBdHRyaWJ1dGUgPSB7IG5hbWU6ICdnc2lIYXNoS2V5JywgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkcgfTtcbmNvbnN0IEdTSV9TT1JUX0tFWTogQXR0cmlidXRlID0geyBuYW1lOiAnZ3NpU29ydEtleScsIHR5cGU6IEF0dHJpYnV0ZVR5cGUuTlVNQkVSIH07XG5jb25zdCBHU0lfTk9OX0tFWTogc3RyaW5nW10gPSBbXTtcbmZvciAobGV0IGkgPSAwOyBpIDwgMTA7IGkrKykgeyAvLyAnQScgdG8gJ0onXG4gIEdTSV9OT05fS0VZLnB1c2goU3RyaW5nLmZyb21DaGFyQ29kZSg2NSArIGkpKTtcbn1cblxuLy8gRHluYW1vREIgbG9jYWwgc2Vjb25kYXJ5IGluZGV4IHBhcmFtZXRlcnNcbmNvbnN0IExTSV9URVNUX0NBU0VfMSA9ICdMU0ktUGFydGl0aW9uQW5kU29ydEtleSc7XG5jb25zdCBMU0lfVEVTVF9DQVNFXzIgPSAnTFNJLVBhcnRpdGlvbkFuZFRhYmxlU29ydEtleSc7XG5jb25zdCBMU0lfVEVTVF9DQVNFXzMgPSAnTFNJLVByb2plY3Rpb25UeXBlS2V5c09ubHknO1xuY29uc3QgTFNJX1RFU1RfQ0FTRV80ID0gJ0xTSS1Qcm9qZWN0aW9uVHlwZUluY2x1ZGUnO1xuY29uc3QgTFNJX1NPUlRfS0VZOiBBdHRyaWJ1dGUgPSB7IG5hbWU6ICdsc2lTb3J0S2V5JywgdHlwZTogQXR0cmlidXRlVHlwZS5OVU1CRVIgfTtcbmNvbnN0IExTSV9OT05fS0VZOiBzdHJpbmdbXSA9IFtdO1xuZm9yIChsZXQgaSA9IDA7IGkgPCAxMDsgaSsrKSB7IC8vICdLJyB0byAnVCdcbiAgTFNJX05PTl9LRVkucHVzaChTdHJpbmcuZnJvbUNoYXJDb2RlKDc1ICsgaSkpO1xufVxuXG5jb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbmNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgU1RBQ0tfTkFNRSk7XG5cbmNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBUQUJMRSwge1xuICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgZW5jcnlwdGlvbjogVGFibGVFbmNyeXB0aW9uLkNVU1RPTUVSX01BTkFHRUQsXG59KTtcblxuY29uc3QgdGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleCA9IG5ldyBUYWJsZShzdGFjaywgVEFCTEVfV0lUSF9HTE9CQUxfQU5EX0xPQ0FMX1NFQ09OREFSWV9JTkRFWCwge1xuICBwb2ludEluVGltZVJlY292ZXJ5OiB0cnVlLFxuICBlbmNyeXB0aW9uOiBUYWJsZUVuY3J5cHRpb24uQVdTX01BTkFHRUQsXG4gIHN0cmVhbTogU3RyZWFtVmlld1R5cGUuS0VZU19PTkxZLFxuICB0aW1lVG9MaXZlQXR0cmlidXRlOiAndGltZVRvTGl2ZScsXG4gIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgc29ydEtleTogVEFCTEVfU09SVF9LRVksXG4gIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbn0pO1xuXG50YWJsZVdpdGhHbG9iYWxBbmRMb2NhbFNlY29uZGFyeUluZGV4Lm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnRW52aXJvbm1lbnQnLCAnUHJvZHVjdGlvbicpKTtcbnRhYmxlV2l0aEdsb2JhbEFuZExvY2FsU2Vjb25kYXJ5SW5kZXguYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICBpbmRleE5hbWU6IEdTSV9URVNUX0NBU0VfMSxcbiAgcGFydGl0aW9uS2V5OiBHU0lfUEFSVElUSU9OX0tFWSxcbn0pO1xudGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleC5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gIGluZGV4TmFtZTogR1NJX1RFU1RfQ0FTRV8yLFxuICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZLFxuICBzb3J0S2V5OiBHU0lfU09SVF9LRVksXG4gIHJlYWRDYXBhY2l0eTogMTAsXG4gIHdyaXRlQ2FwYWNpdHk6IDEwLFxufSk7XG50YWJsZVdpdGhHbG9iYWxBbmRMb2NhbFNlY29uZGFyeUluZGV4LmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgaW5kZXhOYW1lOiBHU0lfVEVTVF9DQVNFXzMsXG4gIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gIHNvcnRLZXk6IEdTSV9TT1JUX0tFWSxcbiAgcHJvamVjdGlvblR5cGU6IFByb2plY3Rpb25UeXBlLktFWVNfT05MWSxcbn0pO1xudGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleC5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gIGluZGV4TmFtZTogR1NJX1RFU1RfQ0FTRV80LFxuICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZLFxuICBzb3J0S2V5OiBHU0lfU09SVF9LRVksXG4gIHByb2plY3Rpb25UeXBlOiBQcm9qZWN0aW9uVHlwZS5JTkNMVURFLFxuICBub25LZXlBdHRyaWJ1dGVzOiBHU0lfTk9OX0tFWSxcbn0pO1xudGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleC5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gIGluZGV4TmFtZTogR1NJX1RFU1RfQ0FTRV81LFxuICBwYXJ0aXRpb25LZXk6IFRBQkxFX1NPUlRfS0VZLFxuICBzb3J0S2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLFxufSk7XG5cbnRhYmxlV2l0aEdsb2JhbEFuZExvY2FsU2Vjb25kYXJ5SW5kZXguYWRkTG9jYWxTZWNvbmRhcnlJbmRleCh7XG4gIGluZGV4TmFtZTogTFNJX1RFU1RfQ0FTRV8yLFxuICBzb3J0S2V5OiBMU0lfU09SVF9LRVksXG59KTtcbnRhYmxlV2l0aEdsb2JhbEFuZExvY2FsU2Vjb25kYXJ5SW5kZXguYWRkTG9jYWxTZWNvbmRhcnlJbmRleCh7XG4gIGluZGV4TmFtZTogTFNJX1RFU1RfQ0FTRV8xLFxuICBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSxcbn0pO1xudGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleC5hZGRMb2NhbFNlY29uZGFyeUluZGV4KHtcbiAgaW5kZXhOYW1lOiBMU0lfVEVTVF9DQVNFXzMsXG4gIHNvcnRLZXk6IExTSV9TT1JUX0tFWSxcbiAgcHJvamVjdGlvblR5cGU6IFByb2plY3Rpb25UeXBlLktFWVNfT05MWSxcbn0pO1xudGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleC5hZGRMb2NhbFNlY29uZGFyeUluZGV4KHtcbiAgaW5kZXhOYW1lOiBMU0lfVEVTVF9DQVNFXzQsXG4gIHNvcnRLZXk6IExTSV9TT1JUX0tFWSxcbiAgcHJvamVjdGlvblR5cGU6IFByb2plY3Rpb25UeXBlLklOQ0xVREUsXG4gIG5vbktleUF0dHJpYnV0ZXM6IExTSV9OT05fS0VZLFxufSk7XG5cbmNvbnN0IGVuY3J5cHRpb25LZXkgPSBuZXcga21zLktleShzdGFjaywgJ0tleScsIHtcbiAgZW5hYmxlS2V5Um90YXRpb246IHRydWUsXG59KTtcblxuY29uc3QgdGFibGVXaXRoR2xvYmFsU2Vjb25kYXJ5SW5kZXggPSBuZXcgVGFibGUoc3RhY2ssIFRBQkxFX1dJVEhfR0xPQkFMX1NFQ09OREFSWV9JTkRFWCwge1xuICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgZW5jcnlwdGlvbktleSxcbn0pO1xudGFibGVXaXRoR2xvYmFsU2Vjb25kYXJ5SW5kZXguYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICBpbmRleE5hbWU6IEdTSV9URVNUX0NBU0VfMSxcbiAgcGFydGl0aW9uS2V5OiBHU0lfUEFSVElUSU9OX0tFWSxcbn0pO1xuXG5jb25zdCB0YWJsZVdpdGhMb2NhbFNlY29uZGFyeUluZGV4ID0gbmV3IFRhYmxlKHN0YWNrLCBUQUJMRV9XSVRIX0xPQ0FMX1NFQ09OREFSWV9JTkRFWCwge1xuICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZLFxuICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gIGVuY3J5cHRpb246IFRhYmxlRW5jcnlwdGlvbi5ERUZBVUxULFxufSk7XG5cbnRhYmxlV2l0aExvY2FsU2Vjb25kYXJ5SW5kZXguYWRkTG9jYWxTZWNvbmRhcnlJbmRleCh7XG4gIGluZGV4TmFtZTogTFNJX1RFU1RfQ0FTRV8xLFxuICBzb3J0S2V5OiBMU0lfU09SVF9LRVksXG59KTtcblxuY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzcXMuYW1hem9uYXdzLmNvbScpLFxufSk7XG50YWJsZS5ncmFudFJlYWREYXRhKHJvbGUpO1xudGFibGVXaXRoR2xvYmFsQW5kTG9jYWxTZWNvbmRhcnlJbmRleC5ncmFudFJlYWREYXRhKHJvbGUpO1xuXG5hcHAuc3ludGgoKTtcbiJdfQ==