"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
// CDK parameters
const CONSTRUCT_NAME = 'MyTable';
// DynamoDB table parameters
const TABLE_NAME = 'MyTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_NAME = 'MyGSI';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.BINARY };
const GSI_NON_KEY = 'gsiNonKey';
function* GSI_GENERATOR() {
    let n = 0;
    while (true) {
        const globalSecondaryIndexProps = {
            indexName: `${GSI_NAME}${n}`,
            partitionKey: { name: `${GSI_PARTITION_KEY.name}${n}`, type: GSI_PARTITION_KEY.type },
        };
        yield globalSecondaryIndexProps;
        n++;
    }
}
function* NON_KEY_ATTRIBUTE_GENERATOR(nonKeyPrefix) {
    let n = 0;
    while (true) {
        yield `${nonKeyPrefix}${n}`;
        n++;
    }
}
// DynamoDB local secondary index parameters
const LSI_NAME = 'MyLSI';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = 'lsiNonKey';
function* LSI_GENERATOR() {
    let n = 0;
    while (true) {
        const localSecondaryIndexProps = {
            indexName: `${LSI_NAME}${n}`,
            sortKey: { name: `${LSI_SORT_KEY.name}${n}`, type: LSI_SORT_KEY.type },
        };
        yield localSecondaryIndexProps;
        n++;
    }
}
describe('default properties', () => {
    let stack;
    beforeEach(() => {
        stack = new core_1.Stack();
    });
    test('hash key only', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [{ AttributeName: 'hashKey', AttributeType: 'S' }],
            KeySchema: [{ AttributeName: 'hashKey', KeyType: 'HASH' }],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.RETAIN }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('removalPolicy is DESTROY', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, removalPolicy: core_1.RemovalPolicy.DESTROY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.DELETE }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('hash + range key', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('hash + range key can also be specified in props', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('point-in-time recovery is not enabled', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('server-side encryption is not enabled', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('stream is not enabled', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('ttl is not enabled', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('can specify new and old images', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_AND_OLD_IMAGES,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_AND_OLD_IMAGES' },
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            TableName: 'MyTable',
        });
    });
    test('can specify new images only', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_IMAGE' },
            TableName: 'MyTable',
        });
    });
    test('can specify old images only', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.OLD_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'OLD_IMAGE' },
            TableName: 'MyTable',
        });
    });
    test('can use PhysicalName.GENERATE_IF_NEEDED as the Table name', () => {
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            partitionKey: TABLE_PARTITION_KEY,
        });
        // since the resource has not been used in a cross-environment manner,
        // so the name should not be filled
        expect(stack).toHaveResourceLike('AWS::DynamoDB::Table', {
            TableName: assert_1.ABSENT,
        });
    });
});
test('when specifying every property', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        readCapacity: 42,
        writeCapacity: 1337,
        pointInTimeRecovery: true,
        serverSideEncryption: true,
        billingMode: lib_1.BillingMode.PROVISIONED,
        stream: lib_1.StreamViewType.KEYS_ONLY,
        timeToLiveAttribute: 'timeToLive',
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 42,
            WriteCapacityUnits: 1337,
        },
        PointInTimeRecoverySpecification: { PointInTimeRecoveryEnabled: true },
        SSESpecification: { SSEEnabled: true },
        StreamSpecification: { StreamViewType: 'KEYS_ONLY' },
        TableName: 'MyTable',
        Tags: [{ Key: 'Environment', Value: 'Production' }],
        TimeToLiveSpecification: { AttributeName: 'timeToLive', Enabled: true },
    });
});
test('when specifying sse with customer managed CMK', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'MyTableKey8597C7A6',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('when specifying only encryptionKey', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryptionKey,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('when specifying sse with customer managed CMK with encryptionKey provided by user', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
        encryptionKey,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('fails if encryption key is used with AWS managed CMK', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.AWS_MANAGED,
        encryptionKey,
    })).toThrow('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
});
test('fails if encryption key is used with default encryption', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.DEFAULT,
        encryptionKey,
    })).toThrow('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
});
test('fails if encryption key is used with serverSideEncryption', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        serverSideEncryption: true,
        encryptionKey,
    })).toThrow(/encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead/);
});
test('fails if both encryption and serverSideEncryption is specified', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.DEFAULT,
        serverSideEncryption: true,
    })).toThrow(/Only one of encryption and serverSideEncryption can be specified, but both were provided/);
});
test('fails if both replication regions used with customer managed CMK', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        replicationRegions: ['us-east-1', 'us-east-2', 'us-west-2'],
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
    })).toThrow('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
});
test('if an encryption key is included, decrypt permissions are also added for grantStream', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryptionKey,
        stream: lib_1.StreamViewType.NEW_IMAGE,
    });
    const user = new iam.User(stack, 'MyUser');
    table.grantStreamRead(user);
    expect(stack).toMatchTemplate({
        'Resources': {
            'Key961B73FD': {
                'Type': 'AWS::KMS::Key',
                'Properties': {
                    'KeyPolicy': {
                        'Statement': [
                            {
                                'Action': [
                                    'kms:Create*',
                                    'kms:Describe*',
                                    'kms:Enable*',
                                    'kms:List*',
                                    'kms:Put*',
                                    'kms:Update*',
                                    'kms:Revoke*',
                                    'kms:Disable*',
                                    'kms:Get*',
                                    'kms:Delete*',
                                    'kms:ScheduleKeyDeletion',
                                    'kms:CancelKeyDeletion',
                                    'kms:GenerateDataKey',
                                    'kms:TagResource',
                                    'kms:UntagResource',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':iam::',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':root',
                                            ],
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'EnableKeyRotation': true,
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'TableA3D7B5AFA': {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'ProvisionedThroughput': {
                        'ReadCapacityUnits': 5,
                        'WriteCapacityUnits': 5,
                    },
                    'SSESpecification': {
                        'KMSMasterKeyId': {
                            'Fn::GetAtt': [
                                'Key961B73FD',
                                'Arn',
                            ],
                        },
                        'SSEEnabled': true,
                        'SSEType': 'KMS',
                    },
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_IMAGE',
                    },
                    'TableName': 'MyTable',
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'MyUserDC45028B': {
                'Type': 'AWS::IAM::User',
            },
            'MyUserDefaultPolicy7B897426': {
                'Type': 'AWS::IAM::Policy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'dynamodb:ListStreams',
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            {
                                                'Fn::GetAtt': [
                                                    'TableA3D7B5AFA',
                                                    'Arn',
                                                ],
                                            },
                                            '/stream/*',
                                        ],
                                    ],
                                },
                            },
                            {
                                'Action': [
                                    'dynamodb:DescribeStream',
                                    'dynamodb:GetRecords',
                                    'dynamodb:GetShardIterator',
                                ],
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'TableA3D7B5AFA',
                                        'StreamArn',
                                    ],
                                },
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'PolicyName': 'MyUserDefaultPolicy7B897426',
                    'Users': [
                        {
                            'Ref': 'MyUserDC45028B',
                        },
                    ],
                },
            },
        },
    });
});
test('if an encryption key is included, encrypt/decrypt permissions are also added both ways', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
    });
    const user = new iam.User(stack, 'MyUser');
    table.grantReadWriteData(user);
    expect(stack).toMatchTemplate({
        'Resources': {
            'TableAKey07CC09EC': {
                'Type': 'AWS::KMS::Key',
                'Properties': {
                    'KeyPolicy': {
                        'Statement': [
                            {
                                'Action': [
                                    'kms:Create*',
                                    'kms:Describe*',
                                    'kms:Enable*',
                                    'kms:List*',
                                    'kms:Put*',
                                    'kms:Update*',
                                    'kms:Revoke*',
                                    'kms:Disable*',
                                    'kms:Get*',
                                    'kms:Delete*',
                                    'kms:ScheduleKeyDeletion',
                                    'kms:CancelKeyDeletion',
                                    'kms:GenerateDataKey',
                                    'kms:TagResource',
                                    'kms:UntagResource',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':iam::',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':root',
                                            ],
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                            {
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::GetAtt': [
                                            'MyUserDC45028B',
                                            'Arn',
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'Description': 'Customer-managed key auto-created for encrypting DynamoDB table at Default/Table A',
                    'EnableKeyRotation': true,
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'TableA3D7B5AFA': {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'ProvisionedThroughput': {
                        'ReadCapacityUnits': 5,
                        'WriteCapacityUnits': 5,
                    },
                    'SSESpecification': {
                        'KMSMasterKeyId': {
                            'Fn::GetAtt': [
                                'TableAKey07CC09EC',
                                'Arn',
                            ],
                        },
                        'SSEEnabled': true,
                        'SSEType': 'KMS',
                    },
                    'TableName': 'MyTable',
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'MyUserDC45028B': {
                'Type': 'AWS::IAM::User',
            },
            'MyUserDefaultPolicy7B897426': {
                'Type': 'AWS::IAM::Policy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': [
                                    'dynamodb:BatchGetItem',
                                    'dynamodb:GetRecords',
                                    'dynamodb:GetShardIterator',
                                    'dynamodb:Query',
                                    'dynamodb:GetItem',
                                    'dynamodb:Scan',
                                    'dynamodb:BatchWriteItem',
                                    'dynamodb:PutItem',
                                    'dynamodb:UpdateItem',
                                    'dynamodb:DeleteItem',
                                ],
                                'Effect': 'Allow',
                                'Resource': [
                                    {
                                        'Fn::GetAtt': [
                                            'TableA3D7B5AFA',
                                            'Arn',
                                        ],
                                    },
                                    {
                                        'Ref': 'AWS::NoValue',
                                    },
                                ],
                            },
                            {
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'TableAKey07CC09EC',
                                        'Arn',
                                    ],
                                },
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'PolicyName': 'MyUserDefaultPolicy7B897426',
                    'Users': [
                        {
                            'Ref': 'MyUserDC45028B',
                        },
                    ],
                },
            },
        },
    });
});
test('when specifying PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
        ],
        TableName: 'MyTable',
    });
});
test('error when specifying read or write capacity with a PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table B', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table C', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding a global secondary index with hash key only', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.ALL,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value, gsiNonKeyAttributeGenerator.next().value],
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['gsiNonKey0', 'gsiNonKey1'], ProjectionType: 'INCLUDE' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index on a table with PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    }).addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('error when adding a global secondary index with projection type INCLUDE, but without specifying non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
    })).toThrow(/non-key attributes should be specified when using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type ALL, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type KEYS_ONLY, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type INCLUDE, but with more than 20 non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    const gsiNonKeyAttributes = [];
    for (let i = 0; i < 101; i++) {
        gsiNonKeyAttributes.push(gsiNonKeyAttributeGenerator.next().value);
    }
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: gsiNonKeyAttributes,
    })).toThrow(/a maximum number of nonKeyAttributes across all of secondary indexes is 100/);
});
test('error when adding a global secondary index with read or write capacity on a PAY_PER_REQUEST table', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
    });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding multiple global secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiGenerator = GSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addGlobalSecondaryIndex(gsiGenerator.next().value);
    }
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey0', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey1', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey2', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey3', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey4', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI0',
                KeySchema: [
                    { AttributeName: 'gsiHashKey0', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI1',
                KeySchema: [
                    { AttributeName: 'gsiHashKey1', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI2',
                KeySchema: [
                    { AttributeName: 'gsiHashKey2', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI3',
                KeySchema: [
                    { AttributeName: 'gsiHashKey3', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI4',
                KeySchema: [
                    { AttributeName: 'gsiHashKey4', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index without specifying read and write capacity', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a local secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(LSI_NON_KEY);
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [lsiNonKeyAttributeGenerator.next().value, lsiNonKeyAttributeGenerator.next().value],
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['lsiNonKey0', 'lsiNonKey1'], ProjectionType: 'INCLUDE' },
            },
        ],
    });
});
test('error when adding more than 5 local secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiGenerator = LSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addLocalSecondaryIndex(lsiGenerator.next().value);
    }
    expect(() => table.addLocalSecondaryIndex(lsiGenerator.next().value))
        .toThrow(/a maximum number of local secondary index per table is 5/);
});
test('error when adding a local secondary index with the name of a global secondary index', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: 'SecondaryIndex',
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(() => table.addLocalSecondaryIndex({
        indexName: 'SecondaryIndex',
        sortKey: LSI_SORT_KEY,
    })).toThrow(/a duplicate index name, SecondaryIndex, is not allowed/);
});
test('error when validating construct if a local secondary index exists without a sort key of the table', () => {
    var _a;
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    const errors = core_1.ConstructNode.validate(table.node);
    expect(errors.length).toBe(1);
    expect((_a = errors[0]) === null || _a === void 0 ? void 0 : _a.message).toBe('a sort key of the table must be specified to add local secondary indexes');
});
test('can enable Read AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:ReadCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBReadCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('can enable Write AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:WriteCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBWriteCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('cannot enable AutoScaling twice on the same property', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/Read AutoScaling already enabled for this table/);
});
test('error when enabling AutoScaling on the PAY_PER_REQUEST table', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { billingMode: lib_1.BillingMode.PAY_PER_REQUEST, partitionKey: TABLE_PARTITION_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => {
        table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.autoScaleGlobalSecondaryIndexReadCapacity(GSI_NAME, {
        minCapacity: 1,
        maxCapacity: 5,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('error when specifying Read Auto Scaling with invalid scalingTargetValue < 10', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 5 });
    }).toThrow(/targetUtilizationPercent for DynamoDB scaling must be between 10 and 90 percent, got: 5/);
});
test('error when specifying Read Auto Scaling with invalid minimumCapacity', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => table.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 5 }))
        .toThrow(/minCapacity \(10\) should be lower than maxCapacity \(5\)/);
});
test('can autoscale on a schedule', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        readCapacity: 42,
        writeCapacity: 1337,
        partitionKey: { name: 'Hash', type: lib_1.AttributeType.STRING },
    });
    // WHEN
    const scaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 100 });
    scaling.scaleOnSchedule('SaveMoneyByNotScalingUp', {
        schedule: appscaling.Schedule.cron({}),
        maxCapacity: 10,
    });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        ScheduledActions: [
            {
                ScalableTargetAction: { 'MaxCapacity': 10 },
                Schedule: 'cron(* * * * ? *)',
                ScheduledActionName: 'SaveMoneyByNotScalingUp',
            },
        ],
    });
});
describe('metrics', () => {
    test('Can use metricConsumedReadCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedReadCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedReadCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricConsumedWriteCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedWriteCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedWriteCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricSystemErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSystemErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SystemErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricUserErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricUserErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'UserErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricConditionalCheckFailedRequests on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConditionalCheckFailedRequests())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConditionalCheckFailedRequests',
            statistic: 'Sum',
        });
    });
    test('Can use metricSuccessfulRequestLatency on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSuccessfulRequestLatency())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SuccessfulRequestLatency',
            statistic: 'Average',
        });
    });
});
describe('grants', () => {
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grant(user, 'dynamodb:action1', 'dynamodb:action2');
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:action1',
                            'dynamodb:action2',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'mytable0324D45C',
                                    'Arn',
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'userDefaultPolicy083DF682',
            'Users': [
                {
                    'Ref': 'user2C2B57AE',
                },
            ],
        });
    });
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        testGrant(['action1', 'action2'], (p, t) => t.grant(p, 'dynamodb:action1', 'dynamodb:action2'));
    });
    test('"grantReadData" allows the principal to read data from the table', () => {
        testGrant(['BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan'], (p, t) => t.grantReadData(p));
    });
    test('"grantWriteData" allows the principal to write data to the table', () => {
        testGrant(['BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'], (p, t) => t.grantWriteData(p));
    });
    test('"grantReadWriteData" allows the principal to read/write data', () => {
        testGrant([
            'BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan',
            'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
        ], (p, t) => t.grantReadWriteData(p));
    });
    test('"grantFullAccess" allows the principal to perform any action on the table ("*")', () => {
        testGrant(['*'], (p, t) => t.grantFullAccess(p));
    });
    test('"Table.grantListStreams" allows principal to list all streams', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const user = new iam.User(stack, 'user');
        // WHEN
        lib_1.Table.grantListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantTableListStreams" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantTableListStreams(user)).toThrow(/DynamoDB Streams must be enabled on the table Default\/my-table/);
    });
    test('"grantTableListStreams" allows principal to list all streams for this table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantStreamRead" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantStreamRead(user)).toThrow(/DynamoDB Streams must be enabled on the table Default\/my-table/);
    });
    test('"grantStreamRead" allows principal to read and describe the table stream"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantStreamRead(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                    {
                        'Action': [
                            'dynamodb:DescribeStream',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'StreamArn',
                            ],
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('if table has an index grant gives access to the index', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
        table.addGlobalSecondaryIndex({ indexName: 'MyIndex', partitionKey: { name: 'Age', type: lib_1.AttributeType.NUMBER } });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'mytable0324D45C',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'mytable0324D45C',
                                                'Arn',
                                            ],
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'userDefaultPolicy083DF682',
            'Users': [
                {
                    'Ref': 'user2C2B57AE',
                },
            ],
        });
    });
    test('grant for an imported table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = lib_1.Table.fromTableName(stack, 'MyTable', 'my-table');
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grant(user, 'dynamodb:*');
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'dynamodb:*',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                Ref: 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            Users: [
                {
                    Ref: 'user2C2B57AE',
                },
            ],
        });
    });
});
describe('secondary indexes', () => {
    // See https://github.com/aws/aws-cdk/issues/4398
    test('attribute can be used as key attribute in one index, and non-key in another', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'pkey', type: lib_1.AttributeType.NUMBER },
        });
        // WHEN
        table.addGlobalSecondaryIndex({
            indexName: 'IndexA',
            partitionKey: { name: 'foo', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['bar'],
        });
        // THEN
        expect(() => table.addGlobalSecondaryIndex({
            indexName: 'IndexB',
            partitionKey: { name: 'baz', type: lib_1.AttributeType.STRING },
            sortKey: { name: 'bar', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['blah'],
        })).not.toThrow();
    });
});
describe('import', () => {
    test('report error when importing an external/existing table from invalid arn missing resource name', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1::table/';
        // WHEN
        expect(() => lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn)).toThrow(/ARN for DynamoDB table must be in the form: .../);
    });
    test('static fromTableArn(arn) allows importing an external/existing table from arn', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1:11111111:table/MyTable';
        const table = lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            tableArn,
                            { 'Ref': 'AWS::NoValue' },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe(tableArn);
        expect(stack.resolve(table.tableName)).toBe('MyTable');
    });
    test('static fromTableName(name) allows importing an external/existing table from table name', () => {
        const stack = new core_1.Stack();
        const tableName = 'MyTable';
        const table = lib_1.Table.fromTableName(stack, 'ImportedTable', tableName);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadWriteData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            'Ref': 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            'Ref': 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            'Ref': 'AWS::AccountId',
                                        },
                                        ':table/MyTable',
                                    ],
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe('arn:${Token[AWS::Partition.3]}:dynamodb:${Token[AWS::Region.4]}:${Token[AWS::AccountId.0]}:table/MyTable');
        expect(stack.resolve(table.tableName)).toBe(tableName);
    });
    describe('stream permissions on imported tables', () => {
        test('throw if no tableStreamArn is specified', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(() => table.grantTableListStreams(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
            expect(() => table.grantStreamRead(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
        });
        test('creates the correct list streams grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantTableListStreams(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
        test('creates the correct stream read grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantStreamRead(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                        {
                            Action: ['dynamodb:DescribeStream', 'dynamodb:GetRecords', 'dynamodb:GetShardIterator'],
                            Effect: 'Allow',
                            Resource: tableStreamArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
        test('creates the correct index grant if indexes have been provided when importing', () => {
            const stack = new core_1.Stack();
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', {
                tableName: 'MyTableName',
                globalIndexes: ['global'],
                localIndexes: ['local'],
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.AnyPrincipal(),
            });
            table.grantReadData(role);
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan',
                            ],
                            Resource: [
                                {
                                    'Fn::Join': ['', [
                                            'arn:',
                                            { Ref: 'AWS::Partition' },
                                            ':dynamodb:',
                                            { Ref: 'AWS::Region' },
                                            ':',
                                            { Ref: 'AWS::AccountId' },
                                            ':table/MyTableName',
                                        ]],
                                },
                                {
                                    'Fn::Join': ['', [
                                            'arn:',
                                            { Ref: 'AWS::Partition' },
                                            ':dynamodb:',
                                            { Ref: 'AWS::Region' },
                                            ':',
                                            { Ref: 'AWS::AccountId' },
                                            ':table/MyTableName/index/*',
                                        ]],
                                },
                            ],
                        },
                    ],
                },
            });
        });
    });
});
describe('global', () => {
    test('create replicas', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-west-2',
            },
            Condition: 'TableStackRegionNotEqualseuwest2A03859E7',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-central-1',
            },
            Condition: 'TableStackRegionNotEqualseucentral199D46FC0',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toEqual({
            TableStackRegionNotEqualseuwest2A03859E7: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-west-2', { Ref: 'AWS::Region' }] },
                ],
            },
            TableStackRegionNotEqualseucentral199D46FC0: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-central-1', { Ref: 'AWS::Region' }] },
                ],
            },
        });
    });
    test('grantReadData', () => {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        table.addGlobalSecondaryIndex({
            indexName: 'my-index',
            partitionKey: {
                name: 'key',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'User');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'TableCD117FA1',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'TableCD117FA1',
                                                'Arn',
                                            ],
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('grantReadData across regions', () => {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1', {
            env: { region: 'us-east-1' },
        });
        const table = new lib_1.Table(stack1, 'Table', {
            tableName: 'my-table',
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        table.addGlobalSecondaryIndex({
            indexName: 'my-index',
            partitionKey: {
                name: 'key',
                type: lib_1.AttributeType.STRING,
            },
        });
        const stack2 = new core_1.Stack(app, 'Stack2', {
            env: { region: 'eu-west-2' },
        });
        const user = new iam.User(stack2, 'User');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack2).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('grantTableListStreams across regions', () => {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1', {
            env: { region: 'us-east-1' },
        });
        const table = new lib_1.Table(stack1, 'Table', {
            tableName: 'my-table',
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        const stack2 = new core_1.Stack(app, 'Stack2', {
            env: { region: 'eu-west-2' },
        });
        const user = new iam.User(stack2, 'User');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack2).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'dynamodb:ListStreams',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('throws with PROVISIONED billing mode', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            billingMode: lib_1.BillingMode.PROVISIONED,
        })).toThrow(/`PAY_PER_REQUEST`/);
    });
    test('throws when stream is set and not set to NEW_AND_OLD_IMAGES', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            stream: lib_1.StreamViewType.OLD_IMAGE,
        })).toThrow(/`NEW_AND_OLD_IMAGES`/);
    });
    test('throws with replica in same region as stack', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'us-east-1' },
        });
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-1',
                'us-east-1',
                'eu-west-2',
            ],
        })).toThrow(/`replicationRegions` cannot include the region where this stack is deployed/);
    });
    test('no conditions when region is known', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'eu-west-1' },
        });
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toBeUndefined();
    });
});
function testGrant(expectedActions, invocation) {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
    const user = new iam.User(stack, 'user');
    // WHEN
    invocation(user, table);
    // THEN
    const action = expectedActions.length > 1 ? expectedActions.map(a => `dynamodb:${a}`) : `dynamodb:${expectedActions[0]}`;
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': action,
                    'Effect': 'Allow',
                    'Resource': [
                        {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'Arn',
                            ],
                        },
                        {
                            'Ref': 'AWS::NoValue',
                        },
                    ],
                },
            ],
            'Version': '2012-10-17',
        },
        'PolicyName': 'userDefaultPolicy083DF682',
        'Users': [
            {
                'Ref': 'user2C2B57AE',
            },
        ],
    });
}
//# sourceMappingURL=data:application/json;base64,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