# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['perceiver_pytorch']

package_data = \
{'': ['*']}

install_requires = \
['einops>=0.3,<0.4', 'torch>=1.6.0,<2.0.0']

setup_kwargs = {
    'name': 'perceiver-multi-modality-pytorch',
    'version': '1.2.0',
    'description': 'A fork of perceiver-pytorch that supports multiple modalities for the Perceiver architecture.',
    'long_description': '![PyPI](https://img.shields.io/pypi/v/perceiver-multi-modality-pytorch.svg)\n![PyPI](https://img.shields.io/pypi/pyversions/perceiver-multi-modality-pytorch.svg)\n![PyPI](https://img.shields.io/github/license/fac2003/perceiver-mutli-modality-pytorch.svg)\n<img src="./perceiver.png" width="600px"></img>\n\n## Multi Modality Perceiver - Pytorch\n\nImplementation of <a href="https://arxiv.org/abs/2103.03206">Perceiver</a>, with support for multi-modality inputs. Fork\nof (lucidrains repo)[https://github.com/lucidrains/perceiver-pytorch] extended for multi-modality and support for text\nembedding splits chunking across layers. This repo also is closer to the Perceiver preprint because you can use GELU\nactivation in feedforward, while Lucidrains\' repo substitutes GEGLU instead. Set use_gelu to true in the\nMultiModalityPerceiver constructor.\n\nMultiModalityPerceiver also provides means to customize pooling method. You can subclass\nperceiver_pytorch.multi_modality_perceiver.MultiModalityPerceiver and override the pool() method, or use the\nperceiver_pytorch.multi_modality_perceiver.MultiModalityPerceiverNoPooling implementation that returns the hidden\nrepresentation without any pooling. This is useful if you need to train multitask models and want to experiment with,\nsay, using the first 3 latent outputs to predict each a different task.\n\n## Install\n\nTo install the Perceiver implementation with multi-modality (also includes without multi-modality):\n\n```bash\n$ pip install perceiver-multi-modality-pytorch\n```\n\nImport with:\n\n```python\nfrom perceiver_pytorch.modalities import modality_encoding\nfrom perceiver_pytorch.multi_modality_perceiver import MultiModalityPerceiver, InputModality\n```\n\nSee tests/test_multimodality_perceiver.py or\n\n```python\nfrom perceiver_pytorch.modalities import InputModalityWithEmbedding\nfrom perceiver_pytorch.multi_modality_with_text_perceiver import MultiModalityWithTextPerceiver\n```\n\nSee tests/test_multimodality_with_text_perceiver.py\n\nTo install the Perceiver implementation, follow instructions at the\n(lucidrains repo)[https://github.com/lucidrains/perceiver-pytorch]:\n\n## Usage\n\n```python\nimport torch\nfrom perceiver_pytorch import Perceiver\n\nmodel = Perceiver(\n    input_channels=3,  # number of channels for each token of the input\n    input_axis=2,  # number of axis for input data (2 for images, 3 for video)\n    num_freq_bands=6,  # number of freq bands, with original value (2 * K + 1)\n    max_freq=10.,  # maximum frequency, hyperparameter depending on how fine the data is\n    depth=6,  # depth of net\n    num_latents=256,\n    # number of latents, or induced set points, or centroids. different papers giving it different names\n    latent_dim=512,  # latent dimension\n    cross_heads=1,  # number of heads for cross attention. paper said 1\n    latent_heads=8,  # number of heads for latent self attention, 8\n    cross_dim_head=64,\n    latent_dim_head=64,\n    num_classes=1000,  # output number of classes\n    attn_dropout=0.,\n    ff_dropout=0.,\n    weight_tie_layers=False  # whether to weight tie layers (optional, as indicated in the diagram)\n)\n\nimg = torch.randn(1, 224, 224, 3)  # 1 imagenet image, pixelized\n\nmodel(img)  # (1, 1000)\n```\n\n## Multi-modality perceiver\n\nAn attractive feature of the perceiver architecture is that it can process multiple modalities of data in the same\nbatch. This is not obvious from the perceiver forward signature shown above, but a relatively modest change can support\nprocessing video, images and audio with a single model, in one forward. This feature is demonstrated by the\nMultiModalityPerceiver, contributed by Fabien Campagne.\n\n```python\nfrom perceiver_pytorch.multi_modality_perceiver import MultiModalityPerceiver, InputModality\n\nimage_inputs = torch.rand(size=(3, 260, 260, 3), requires_grad=True)\nvideo_inputs = torch.rand(size=(3, 32, 260, 260, 3), requires_grad=True)\naudio_inputs = torch.rand(size=(3, 44100, 1), requires_grad=True)\n\nvideo_modality = InputModality(\n    name=\'video\',\n    input_channels=3,  # number of channels for each token of the input\n    input_axis=3,  # number of axes, 3 for video)\n    num_freq_bands=6,  # number of freq bands, with original value (2 * K + 1)\n    max_freq=4.,  # maximum frequency, hyperparameter depending on how fine the data is\n)\nimage_modality = InputModality(\n    name=\'image\',\n    input_channels=3,  # number of channels for each token of the input\n    input_axis=2,  # number of axes, 2 for images\n    num_freq_bands=6,  # number of freq bands, with original value (2 * K + 1)\n    max_freq=4.,  # maximum frequency, hyperparameter depending on how fine the data is\n)\naudio_modality = InputModality(\n    name=\'audio\',\n    input_channels=1,  # number of channels for mono audio\n    input_axis=1,  # number of axes, 2 for images\n    num_freq_bands=6,  # number of freq bands, with original value (2 * K + 1)\n    max_freq=8.,  # maximum frequency, hyperparameter depending on how fine the data is\n)\nmodel = MultiModalityPerceiver(\n    modalities=(video_modality, image_modality, audio_modality),\n    depth=8,  # depth of net, combined with num_latent_blocks_per_layer to produce full Perceiver\n    num_latents=12,\n    # number of latents, or induced set points, or centroids. different papers giving it different names\n    latent_dim=64,  # latent dimension\n    cross_heads=1,  # number of heads for cross attention. paper said 1\n    latent_heads=8,  # number of heads for latent self attention, 8\n    cross_dim_head=64,\n    latent_dim_head=64,\n    num_classes=1000,  # output number of classes\n    attn_dropout=0.,\n    ff_dropout=0.,\n    weight_tie_layers=True,\n    num_latent_blocks_per_layer=6  # Note that this parameter is 1 in the original Lucidrain implementation\n    # whether to weight tie layers (optional, as indicated in the diagram)\n)\nresult = model({\'image\': image_inputs,\n                \'video\': video_inputs,\n                \'audio\': audio_inputs})\n```\n\n### Text perceiver\n\nWhile the Perceiver architecture described by [jaegle2021perceiver] could support text if text was embedded and each\ndimension of the embedding provided as a channel in the input, this introduces a mismatch between the text embedding\ndimension (typically large, 512/768 or more) and the number of channels used for video and images (typically 3 channels,\none for red, green and blue), or audio\n(1 for mono or 2 for stereo channels). When training text embeddings from scratch, this creates an opportunity, because\nthere should be no need for the perceiver to attend to the entire text embedding in each layer. If we split the text\nembedding into as many chunks as there are layers in a perceiver, we reduce how much we need to pad other modalities,\nand introduce a structure to the learned embeddings, were parts of the text embedding can specialize according to the\nneeds of each layer. The perceiver implementation provided in this repo can be used to explore the question of whether\nsplitting text embeddings across layers is beneficial (you would compare the performance of\nMultiModalityWithTextPerceiver with that of MultiModalityPerceiver).\n\n## Citations\n\n```bibtex\n@misc{jaegle2021perceiver,\n    title   = {Perceiver: General Perception with Iterative Attention},\n    author  = {Andrew Jaegle and Felix Gimeno and Andrew Brock and Andrew Zisserman and Oriol Vinyals and Joao Carreira},\n    year    = {2021},\n    eprint  = {2103.03206},\n    archivePrefix = {arXiv},\n    primaryClass = {cs.CV}\n}\n@misc{campagne2021textperceiver,\n    title   = {Adapting Perceiver for learning with text modalities},\n    author  = {Fabien Campagne},\n    year    = {2021},\n    eprint  = {unpublished results},\n}\n```\n',
    'author': 'Fabien Campagne',
    'author_email': 'fac2003@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/fac2003/perceiver-multi-modality-pytorch',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
