# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sk_transformers', 'sk_transformers.transformer']

package_data = \
{'': ['*']}

install_requires = \
['feature-engine==1.5.2',
 'joblib==1.2.0',
 'numpy==1.23.5',
 'pandas==1.5.2',
 'phonenumbers==8.13.2',
 'scikit-learn==1.2.0']

setup_kwargs = {
    'name': 'sk-transformers',
    'version': '0.5.4',
    'description': 'A collection of various scikit-learn compatible transformers for all kinds of preprocessing and feature engineering',
    'long_description': '![The Transformer](https://raw.githubusercontent.com/chrislemke/sk-transformers/master/docs/assets/images/image.png)\n\n# sk-transformers\n### A collection of various scikit-learn transformers for all kinds of preprocessing and feature engineering steps 🛠\n\n[![testing](https://github.com/chrislemke/sk-transformers/actions/workflows/testing.yml/badge.svg?branch=main)](https://github.com/chrislemke/sk-transformers/actions/workflows/testing.yml)\n[![codecov](https://codecov.io/github/chrislemke/sk-transformers/branch/main/graph/badge.svg?token=LJLXQXX6M8)](https://codecov.io/github/chrislemke/sk-transformers)\n[![deploy package](https://github.com/chrislemke/sk-transformers/actions/workflows/deploy-package.yml/badge.svg)](https://github.com/chrislemke/sk-transformers/actions/workflows/deploy-package.yml)\n[![pypi](https://img.shields.io/pypi/v/sk-transformers)](https://pypi.org/project/sk-transformers/)\n![python version](https://img.shields.io/pypi/pyversions/sk-transformers?logo=python&logoColor=yellow)\n[![downloads](https://img.shields.io/pypi/dm/sk-transformers)](https://pypistats.org/packages/sk-transformers)\n[![docs](https://img.shields.io/badge/docs-mkdoks%20material-blue)](https://chrislemke.github.io/sk-transformers/)\n[![license](https://img.shields.io/github/license/chrislemke/sk-transformers)](https://github.com/chrislemke/sk-transformers/blob/main/LICENSE)\n[![mypy](http://www.mypy-lang.org/static/mypy_badge.svg)](http://mypy-lang.org/)\n[![black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n## Introduction\nEvery tabular data is different. Every column needs to be treated differently. [scikit-learn](https://scikit-learn.org/stable/index.html) has already a nice [collection of dataset transformers](https://scikit-learn.org/stable/data_transforms.html). But the possibilities of data transformation are infinite. This project tries to provide a brought collection of data transformers that can be easily used together with [scikit-learn](https://scikit-learn.org/stable/index.html) - either in a [pipeline](https://scikit-learn.org/stable/modules/generated/sklearn.pipeline.Pipeline.html) or just by its own. See the [usage chapter](#usage) for some examples.\n\nThe idea is simple. It is like a well-equipped toolbox 🧰: You always find the tool you need and sometimes you get inspired by seeing a tool you did not know before. Please feel free to [contribute](https://chrislemke.github.io/sk-transformers/CONTRIBUTING/) your tools and ideas.\n\n## Installation\nIf you are using [pip](https://pip.pypa.io/en/stable/), you can install the package with the following command:\n```bash\npip install sk-transformers\n```\n\nIf you are using [Poetry](https://python-poetry.org/), you can install the package with the following command:\n```bash\npoetry add sk-transformers\n```\n\n## installing dependencies\nWith [pip](https://pip.pypa.io/en/stable/):\n```bash\npip install -r requirements.txt\n```\n\nWith [Poetry](https://python-poetry.org/):\n```bash\npoetry install\n```\n\n## Usage\nLet\'s assume you want to use some method from [NumPy\'s mathematical functions, to sum up the values of column `foo` and column `bar`. You could\nuse the [`MathExpressionTransformer`](https://chrislemke.github.io/sk-transformers/number_transformer-reference/#sk-transformers.transformer.number_transformer.MathExpressionTransformer).\n```python\nimport pandas as pd\nfrom sk_transformers import MathExpressionTransformer\n\nX = pd.DataFrame({"foo": [1, 2, 3], "bar": [4, 5, 6]})\ntransformer = MathExpressionTransformer([("foo", "np.sum", "bar", {"axis": 0})])\ntransformer.fit_transform(X).to_numpy()\n```\n```\narray([[1, 4, 5],\n       [2, 5, 7],\n       [3, 6, 9]])\n```\nEven if we only pass one tuple to the transformer - in this example. Like with most other transformers the idea is to simplify preprocessing by giving the possibility to operate on multiple columns at the same time. In this case, the [`MathExpressionTransformer`](https://chrislemke.github.io/sk-transformers/number_transformer-reference/#sk-transformers.transformer.number_transformer.MathExpressionTransformer) has created an extra column with the name `foo_sum_bar`.\n\nIn the next example, we additionally add the [`MapTransformer`](https://chrislemke.github.io/sk-transformers/generic_transformer-reference/#sk_transformers.transformer.generic_transformer.MapTransformer).\nTogether with [scikit-learn\'s pipelines](https://scikit-learn.org/stable/modules/compose.html#combining-estimators) it would look like this:\n```python\nimport pandas as pd\nfrom sk_transformers import MapTransformer, MathExpressionTransformer\nfrom sklearn.pipeline import Pipeline\n\nX = pd.DataFrame({"foo": [1, 2, 3], "bar": [4, 5, 6]})\nmap_step = MapTransformer([("foo", lambda x: x + 100)])\nsum_step = MathExpressionTransformer([("foo", "np.sum", "bar", {"axis": 0})])\npipeline = Pipeline([("map_step", map_step), ("sum_step", sum_step)])\npipeline.fit_transform(X)\n```\n\n```\n   foo  bar  foo_sum_bar\n0  101    4          105\n1  102    5          107\n2  103    6          109\n```\n\n## Contributing\nWe\'re all kind of in the same boat. Preprocessing/feature engineering in data science is somehow very individual - every feature is different and must be handled and processed differently. But somehow we all have the same problems: sometimes date columns have to be changed. Sometimes strings have to be formatted, sometimes durations have to be calculated, etc. There is a huge number of preprocessing possibilities but we all use the same tools.\n\n[scikit-learns pipelines](https://scikit-learn.org/stable/modules/generated/sklearn.pipeline.Pipeline.html) help to use formalized functions. So why not also share these so-called transformers with others? This open-source project has the goal to collect useful preprocessing pipeline steps. Let us all collect what we used for preprocessing and share it with others. This way we can all benefit from each other\'s work and save a lot of time. So if you have a preprocessing step that you use regularly, please feel free to contribute it to this project. The idea is that this is not only a toolbox but also an inspiration for what is possible. Maybe you have not thought about this preprocessing step before.\n\nPlease check out the [guide](https://chrislemke.github.io/sk-transformers/CONTRIBUTING/) on how to contribute to this project.\n\n## Further information\nFor further information, please refer to the [documentation](https://chrislemke.github.io/sk-transformers/).\n',
    'author': 'Christopher Lemke',
    'author_email': 'chris@syhbl.mozmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://chrislemke.github.io/sk-transformers/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.12',
}


setup(**setup_kwargs)
