"""
Memory Machines API Client
"""
import requests
from typing import Dict, List, Optional, Any, Iterator
from datetime import datetime
from .exceptions import (
    AuthenticationError,
    RateLimitError,
    NotFoundError,
    ValidationError,
    APIError
)
from .auth import get_token


class MemoryClient:
    """
    Official Python client for Memory Machines API.

    Example:
        >>> from memorymachines import MemoryClient

        # Auto-discover API key from config file or environment
        >>> client = MemoryClient()

        # Or specify explicitly
        >>> client = MemoryClient(api_key="mm_sk_...")

        # Or use a specific profile
        >>> client = MemoryClient(profile="work")

        >>> memories = client.search("what did I discuss last week?")
        >>> client.ingest("Meeting notes from today...")
    """

    def __init__(
        self,
        api_key: Optional[str] = None,
        profile: Optional[str] = None,
        base_url: str = "https://memorymachines-core-api-mvp-gateway-6v1lw71z.uc.gateway.dev"
    ):
        """
        Initialize Memory Machines client.

        API key resolution order:
        1. Explicit api_key parameter
        2. MM_API_KEY environment variable
        3. ~/.memorymachines/config.yaml (active profile or specified profile)

        Args:
            api_key: Your Memory Machines API key. If not provided, auto-discovered.
            profile: Config profile to use (default: active profile)
            base_url: API base URL (default: production)

        Raises:
            AuthenticationError: If no API key found
        """
        # Auto-discover API key if not provided
        if api_key is None:
            api_key = get_token(profile=profile)

        if api_key is None:
            raise AuthenticationError(
                "No API key found. Either:\n"
                "  1. Pass api_key='mm_sk_...' to MemoryClient()\n"
                "  2. Set MM_API_KEY environment variable\n"
                "  3. Run `memorymachines login` or `login(token='mm_sk_...')`"
            )

        self.api_key = api_key
        self.base_url = base_url.rstrip('/')
        self.session = requests.Session()
        self.session.headers.update({
            'Authorization': f'Bearer {api_key}',
            'Content-Type': 'application/json'
        })

    def _handle_response(self, response: requests.Response) -> Any:
        """Handle API response and raise appropriate exceptions."""
        try:
            response.raise_for_status()
            return response.json()
        except requests.exceptions.HTTPError as e:
            status_code = e.response.status_code

            if status_code == 401:
                raise AuthenticationError("Invalid or expired API key")
            elif status_code == 404:
                raise NotFoundError("Resource not found")
            elif status_code == 429:
                raise RateLimitError("Rate limit exceeded. Please try again later.")
            elif status_code == 400:
                error_msg = e.response.json().get('error', 'Invalid request')
                raise ValidationError(error_msg)
            else:
                error_msg = e.response.json().get('error', 'API error')
                raise APIError(f"API error ({status_code}): {error_msg}")

    def ingest(self, content: str, source: str = "custom", metadata: Optional[Dict] = None) -> Dict[str, Any]:
        """
        Ingest text content to extract memories.

        Args:
            content: Text content to ingest
            source: Source type (e.g., "email", "notes", "custom")
            metadata: Optional metadata dict

        Returns:
            Response dict with status

        Example:
            >>> client.ingest("Had a great meeting with the team today about Q4 goals")

        Raises:
            AuthenticationError: If API key is invalid
            ValidationError: If content is invalid
            RateLimitError: If rate limit is exceeded
        """
        response = self.session.post(
            f"{self.base_url}/api/ingest",
            json={
                'content': content,
                'source': source,
                'metadata': metadata or {}
            }
        )
        return self._handle_response(response)

    def ingest_batch(self, items: List[Dict[str, Any]]) -> Dict[str, Any]:
        """
        Ingest multiple items at once.

        Args:
            items: List of dicts with 'content', 'source', 'metadata' keys

        Returns:
            Response dict with batch status

        Example:
            >>> items = [
            ...     {'content': 'Meeting notes...', 'source': 'notes'},
            ...     {'content': 'Email from client...', 'source': 'email'}
            ... ]
            >>> client.ingest_batch(items)

        Raises:
            AuthenticationError: If API key is invalid
            ValidationError: If items are invalid
            RateLimitError: If rate limit is exceeded
        """
        response = self.session.post(
            f"{self.base_url}/api/ingest/batch",
            json={'items': items}
        )
        return self._handle_response(response)

    def search(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """
        Search your memories.

        Args:
            query: Natural language search query
            limit: Maximum number of results

        Returns:
            List of memory dicts

        Example:
            >>> memories = client.search("what did I discuss about the product launch?")
            >>> for memory in memories:
            ...     print(memory['content'])

        Raises:
            AuthenticationError: If API key is invalid
        """
        response = self.session.get(
            f"{self.base_url}/api/memories",
            params={'query': query, 'limit': limit}
        )
        result = self._handle_response(response)
        return result.get('memories', [])

    def get_memory(self, memory_id: str) -> Dict[str, Any]:
        """
        Get a specific memory by ID.

        Args:
            memory_id: Memory ID

        Returns:
            Memory dict

        Raises:
            NotFoundError: If memory not found
        """
        response = self.session.get(
            f"{self.base_url}/api/memories/{memory_id}"
        )
        return self._handle_response(response)

    def list_memories(self,
                     limit: int = 50,
                     offset: int = 0,
                     source: Optional[str] = None,
                     since: Optional[datetime] = None) -> Dict[str, Any]:
        """
        List memories with filters.

        Args:
            limit: Number of memories to return
            offset: Pagination offset
            source: Filter by source type
            since: Only return memories after this date

        Returns:
            Dict with 'memories' list and pagination info
        """
        params = {'limit': limit, 'offset': offset}
        if source:
            params['source'] = source
        if since:
            params['since'] = since.isoformat()

        response = self.session.get(
            f"{self.base_url}/api/memories",
            params=params
        )
        return self._handle_response(response)

    def iter_memories(self,
                     batch_size: int = 100,
                     source: Optional[str] = None,
                     since: Optional[datetime] = None) -> Iterator[Dict[str, Any]]:
        """
        Iterate through all memories with automatic pagination.

        Args:
            batch_size: Number of memories to fetch per request
            source: Filter by source type
            since: Only return memories after this date

        Yields:
            Memory dicts one at a time

        Example:
            >>> for memory in client.iter_memories(source="email"):
            ...     print(memory['content'])
        """
        offset = 0
        while True:
            response = self.list_memories(
                limit=batch_size,
                offset=offset,
                source=source,
                since=since
            )
            memories = response.get('memories', [])

            if not memories:
                break

            for memory in memories:
                yield memory

            offset += batch_size

    def delete_memory(self, memory_id: str) -> Dict[str, Any]:
        """
        Delete a specific memory.

        Args:
            memory_id: Memory ID to delete

        Returns:
            Response dict

        Raises:
            NotFoundError: If memory not found
        """
        response = self.session.delete(
            f"{self.base_url}/api/memories/{memory_id}"
        )
        return self._handle_response(response)

    def get_connected_services(self) -> List[str]:
        """
        Get list of connected OAuth services.

        Returns:
            List of service names (e.g., ['gmail', 'slack'])
        """
        response = self.session.get(
            f"{self.base_url}/api/oauth/connected"
        )
        result = self._handle_response(response)
        return result.get('services', [])

    def health_check(self) -> bool:
        """
        Check if API is accessible.

        Returns:
            True if API is healthy
        """
        try:
            response = self.session.get(f"{self.base_url}/health")
            return response.status_code == 200
        except:
            return False
