"""
Memory Machines CLI

Usage:
    memorymachines login [--token TOKEN] [--profile PROFILE]
    memorymachines logout [--profile PROFILE]
    memorymachines whoami
    memorymachines profiles
"""
import argparse
import sys
import getpass
from .auth import login, logout, whoami, list_profiles, set_active_profile


def main():
    parser = argparse.ArgumentParser(
        prog="memorymachines",
        description="Memory Machines CLI"
    )
    subparsers = parser.add_subparsers(dest="command", help="Available commands")

    # login command
    login_parser = subparsers.add_parser("login", help="Save API key to config file")
    login_parser.add_argument(
        "--token", "-t",
        help="API key (will prompt if not provided)"
    )
    login_parser.add_argument(
        "--profile", "-p",
        default="default",
        help="Profile name (default: 'default')"
    )

    # logout command
    logout_parser = subparsers.add_parser("logout", help="Remove API key from config")
    logout_parser.add_argument(
        "--profile", "-p",
        help="Profile to remove (removes all if not specified)"
    )

    # whoami command
    subparsers.add_parser("whoami", help="Show current authentication status")

    # profiles command
    subparsers.add_parser("profiles", help="List all configured profiles")

    # use command (switch profiles)
    use_parser = subparsers.add_parser("use", help="Switch active profile")
    use_parser.add_argument("profile", help="Profile name to activate")

    args = parser.parse_args()

    if args.command == "login":
        token = args.token
        if not token:
            token = getpass.getpass("Enter your Memory Machines API key: ")
        if not token:
            print("Error: No token provided", file=sys.stderr)
            sys.exit(1)
        login(token=token, profile=args.profile)

    elif args.command == "logout":
        logout(profile=args.profile)

    elif args.command == "whoami":
        whoami()

    elif args.command == "profiles":
        profiles = list_profiles()
        if not profiles:
            print("No profiles configured. Run `memorymachines login` first.")
        else:
            for name, info in profiles.items():
                marker = " (active)" if info["active"] else ""
                print(f"  {name}{marker}")

    elif args.command == "use":
        try:
            set_active_profile(args.profile)
        except (FileNotFoundError, ValueError) as e:
            print(f"Error: {e}", file=sys.stderr)
            sys.exit(1)

    else:
        parser.print_help()


if __name__ == "__main__":
    main()
