"""
Authentication utilities for Memory Machines SDK.

Handles API key storage, retrieval, and management.
Supports YAML config file and environment variables.
"""
import os
from pathlib import Path
from typing import Optional

import yaml

# Default config location (like ~/.huggingface/token)
DEFAULT_CONFIG_DIR = Path.home() / ".memorymachines"
DEFAULT_CONFIG_FILE = DEFAULT_CONFIG_DIR / "config.yaml"
ENV_VAR_NAME = "MM_API_KEY"


def get_config_path() -> Path:
    """Get the config file path, respecting MM_CONFIG_PATH env var."""
    custom_path = os.environ.get("MM_CONFIG_PATH")
    if custom_path:
        return Path(custom_path)
    return DEFAULT_CONFIG_FILE


def login(token: str, profile: str = "default") -> None:
    """
    Save API key to config file.

    Args:
        token: Your Memory Machines API key (mm_sk_...)
        profile: Profile name for multiple accounts (default: "default")

    Example:
        >>> from memorymachines import login
        >>> login("mm_sk_abc123")
        Token saved to ~/.memorymachines/config.yaml
    """
    config_path = get_config_path()
    config_path.parent.mkdir(parents=True, exist_ok=True)

    # Load existing config or start fresh
    config = {}
    if config_path.exists():
        with open(config_path, "r") as f:
            config = yaml.safe_load(f) or {}

    # Ensure profiles section exists
    if "profiles" not in config:
        config["profiles"] = {}

    # Save token under profile
    config["profiles"][profile] = {"api_key": token}

    # Set as active profile if it's the first one
    if "active_profile" not in config:
        config["active_profile"] = profile

    # Write config with restricted permissions
    with open(config_path, "w") as f:
        yaml.safe_dump(config, f, default_flow_style=False)

    # Restrict file permissions (owner read/write only)
    os.chmod(config_path, 0o600)

    print(f"Token saved to {config_path}")


def logout(profile: Optional[str] = None) -> None:
    """
    Remove API key from config file.

    Args:
        profile: Profile to remove. If None, removes all profiles.

    Example:
        >>> from memorymachines import logout
        >>> logout()  # Remove all
        >>> logout("work")  # Remove specific profile
    """
    config_path = get_config_path()

    if not config_path.exists():
        print("No config file found. Already logged out.")
        return

    if profile is None:
        # Remove entire config file
        config_path.unlink()
        print(f"Removed {config_path}")
    else:
        # Remove specific profile
        with open(config_path, "r") as f:
            config = yaml.safe_load(f) or {}

        if "profiles" in config and profile in config["profiles"]:
            del config["profiles"][profile]

            # Update active profile if needed
            if config.get("active_profile") == profile:
                remaining = list(config["profiles"].keys())
                config["active_profile"] = remaining[0] if remaining else None

            with open(config_path, "w") as f:
                yaml.safe_dump(config, f, default_flow_style=False)

            print(f"Removed profile '{profile}'")
        else:
            print(f"Profile '{profile}' not found")


def get_token(profile: Optional[str] = None) -> Optional[str]:
    """
    Get API key from config file or environment.

    Resolution order:
    1. MM_API_KEY environment variable
    2. Config file (~/.memorymachines/config.yaml)

    Args:
        profile: Profile name. If None, uses active profile.

    Returns:
        API key string or None if not found.

    Example:
        >>> from memorymachines.auth import get_token
        >>> token = get_token()
        >>> token = get_token(profile="work")
    """
    # 1. Check environment variable first
    env_token = os.environ.get(ENV_VAR_NAME)
    if env_token:
        return env_token

    # 2. Check config file
    config_path = get_config_path()
    if not config_path.exists():
        return None

    with open(config_path, "r") as f:
        config = yaml.safe_load(f) or {}

    profiles = config.get("profiles", {})

    # Use specified profile or active profile
    profile_name = profile or config.get("active_profile", "default")
    profile_data = profiles.get(profile_name, {})

    return profile_data.get("api_key")


def set_active_profile(profile: str) -> None:
    """
    Set the active profile.

    Args:
        profile: Profile name to activate.

    Example:
        >>> from memorymachines import set_active_profile
        >>> set_active_profile("work")
    """
    config_path = get_config_path()

    if not config_path.exists():
        raise FileNotFoundError(f"No config file found at {config_path}. Run login() first.")

    with open(config_path, "r") as f:
        config = yaml.safe_load(f) or {}

    if profile not in config.get("profiles", {}):
        raise ValueError(f"Profile '{profile}' not found. Available: {list(config.get('profiles', {}).keys())}")

    config["active_profile"] = profile

    with open(config_path, "w") as f:
        yaml.safe_dump(config, f, default_flow_style=False)

    print(f"Active profile set to '{profile}'")


def list_profiles() -> dict:
    """
    List all configured profiles.

    Returns:
        Dict with profile names and active status.

    Example:
        >>> from memorymachines import list_profiles
        >>> list_profiles()
        {'default': {'active': True}, 'work': {'active': False}}
    """
    config_path = get_config_path()

    if not config_path.exists():
        return {}

    with open(config_path, "r") as f:
        config = yaml.safe_load(f) or {}

    active = config.get("active_profile")
    profiles = config.get("profiles", {})

    return {
        name: {"active": name == active}
        for name in profiles.keys()
    }


def whoami() -> Optional[str]:
    """
    Check current authentication status.

    Returns:
        Profile name if logged in, None otherwise.

    Example:
        >>> from memorymachines import whoami
        >>> whoami()
        'default'
    """
    token = get_token()
    if not token:
        print("Not logged in. Run `memorymachines login` or `login(token=...)`")
        return None

    config_path = get_config_path()
    if config_path.exists():
        with open(config_path, "r") as f:
            config = yaml.safe_load(f) or {}
        profile = config.get("active_profile", "default")
        print(f"Logged in as profile '{profile}'")
        return profile

    print("Logged in via MM_API_KEY environment variable")
    return "env"
