# Memory Machines Python SDK

Official Python SDK for the Memory Machines API.

## Installation

```bash
pip install memorymachines
```

Or install from source:

```bash
git clone https://github.com/memorymachines/memorymachines-sdk.git
cd memorymachines-sdk
pip install -e .
```

## Quick Start

```bash
# One-time login (saves to ~/.memorymachines/config.yaml)
memorymachines login
```

```python
from memorymachines import MemoryClient

# No API key needed - auto-discovered from config file
client = MemoryClient()

# Ingest content
client.ingest("Had a great meeting about Q4 goals today")

# Search memories
memories = client.search("what did we discuss about Q4?")
for memory in memories:
    print(memory['content'])
```

## Getting an API Key

1. Sign up at [memorymachines.com](https://memorymachines.com)
2. Go to your dashboard
3. Copy your API key (starts with `mm_sk_`)

## Authentication

The SDK auto-discovers your API key in this order:

1. Explicit `api_key` parameter
2. `MM_API_KEY` environment variable
3. `~/.memorymachines/config.yaml` file

### CLI Login (Recommended)

```bash
# Save your API key securely
memorymachines login

# Check who's logged in
memorymachines whoami

# List profiles
memorymachines profiles

# Logout
memorymachines logout
```

### Programmatic Login

```python
from memorymachines import login, logout

# Save token
login("mm_sk_...")

# Remove token
logout()
```

### Multiple Profiles

```bash
# Save multiple accounts
memorymachines login --profile personal
memorymachines login --profile work

# Switch profiles
memorymachines use work
```

```python
# Use specific profile
client = MemoryClient(profile="work")
```

## Usage

### Initialize Client

```python
from memorymachines import MemoryClient

# Auto-discover API key (recommended)
client = MemoryClient()

# Or explicit API key
client = MemoryClient(api_key="mm_sk_...")

# Or specific profile
client = MemoryClient(profile="work")

# Custom endpoint (for development)
client = MemoryClient(base_url="http://localhost:8080")
```

### Ingest Data

```python
# Single item
client.ingest(
    content="Meeting notes from today...",
    source="notes",
    metadata={"date": "2024-10-24"}
)

# Batch ingestion
items = [
    {'content': 'Email from client...', 'source': 'email'},
    {'content': 'Meeting notes...', 'source': 'notes'}
]
client.ingest_batch(items)
```

### Search Memories

```python
# Natural language search
memories = client.search("what did I discuss last week?", limit=10)

# List all memories with filters
response = client.list_memories(
    limit=50,
    source="email",
    since=datetime(2024, 10, 1)
)
```

### Manage Memories

```python
# Get specific memory
memory = client.get_memory(memory_id="mem_123")

# Delete memory
client.delete_memory(memory_id="mem_123")

# Check connected services
services = client.get_connected_services()  # ['gmail', 'slack']
```

## Examples

See the [examples/](examples/) directory for more usage examples:

- [basic_usage.py](examples/basic_usage.py) - Basic SDK operations
- [batch_ingest.py](examples/batch_ingest.py) - Batch data ingestion

## API Reference

### `MemoryClient`

#### `__init__(api_key: str, base_url: str = "https://api.memorymachines.com")`
Initialize the client.

#### `ingest(content: str, source: str = "custom", metadata: dict = None) -> dict`
Ingest text content to extract memories.

#### `ingest_batch(items: List[dict]) -> dict`
Ingest multiple items at once.

#### `search(query: str, limit: int = 10) -> List[dict]`
Search memories with natural language.

#### `list_memories(limit: int = 50, offset: int = 0, source: str = None, since: datetime = None) -> dict`
List memories with filters.

#### `get_memory(memory_id: str) -> dict`
Get a specific memory by ID.

#### `delete_memory(memory_id: str) -> dict`
Delete a memory.

#### `get_connected_services() -> List[str]`
Get list of connected OAuth services.

#### `health_check() -> bool`
Check if API is accessible.

## Development

### Setup

```bash
git clone https://github.com/memorymachines/memorymachines-sdk.git
cd memorymachines-sdk
pip install -e ".[dev]"
```

### Run Examples

```bash
export MEMORY_MACHINES_API_KEY="mm_sk_..."
python examples/basic_usage.py
```

## Support

- Documentation: [docs.memorymachines.com](https://docs.memorymachines.com)
- Issues: [GitHub Issues](https://github.com/memorymachines/memorymachines-sdk/issues)
- Email: support@memorymachines.com

## License

Apache 2.0 - see [LICENSE](LICENSE) file for details.
