# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['batch']
setup_kwargs = {
    'name': 'batchable',
    'version': '0.2.2',
    'description': 'Hide your batch logic away from the actual code.',
    'long_description': '# batchable\n\nAllows hiding the batching logic of requests.\n\n```bash\npip install batchable\n```\n\nThis is the result of a learning day @ solute, together with\n[@digitalarbeiter](https://github.com/digitalarbeiter).\n\n## Idea\n\nWe are often faced with the following situation:\n\n- A stream of objects has to be processed\n- During this process, some kind of lookup has to be made\n\nAs an example, consider this mockup of an e-commerce system processing offers\nfor articles:\n\n```python\ndef transform_offer(offer):\n    return {\n        "id": offer["offer_id"],\n        "shop_id": offer["shop_id"],\n    }\n\nprocessed_offers = [transform_offer(offer) for offer in unprocessed_offers]\n```\n\nSo far, this is straightforward. Now consider the case where you want to add\nthe name of the shop referenced by ID. This name is not stored inside the\nunprocessed offer, but instead has to be retrieved from a (different) database:\n\n```python\ndef transform_offer(offer):\n    return {\n        "id": offer["offer_id"],\n        "shop_name": lookup_shop(offer["shop_id"])["name"],\n    }\n\ndef lookup_shop(shop_id):\n    # returns e.g. {"id": 23, "name": "Fancy shop"}\n    return dict(\n        db.execute(\n            "SELECT id, name FROM shops WHERE id={id}",\n            id=shop_id,\n        ).fetchone(),\n    )\n```\n\nAgain, this works, but it has a major downside: For every offer that is\nprocessed, a new roundtrip is made to the database. We also would do the exact\nsame queries several times, if some offers share the same shop ID (which is\nvery likely). This second problem is solvable by caching the function, e.g. via\n`functools.lru_cache`. But the main problem (one request per offer) remains.\n\nThe solution to this problem is to add batching: You somehow have to collect\nthe shop IDs somewhere, and only make a request once there are _n_ shop IDs\nbeing requested. Doing this is non-trivial, but also not terribly difficult.\nThe problem with this solution is that you now have to restructure your code\nquite a bit. Maybe you have to iterate over the offers twice; once to get _all_\nshop IDs, and then again to do the actual processing. Maybe you\'d do it the\nother way around, where you do several passes (first put only shop IDs in the\noffers while also putting them in some kind of queue, then process the queue,\nand finally enrich the half-processed offers with shop names.\n\n------\n\nThis project aims to solve this issue, by allowing you to write your code just\nlike you normally would, and doing nasty things behind the scenes to enable\nbatching that you don\'t see. First, you import the library:\n\n```python\nimport batch\n```\n\nThen you decorate the function you want to batch with `batch.able`, while\nchanging it to handle _several_ IDs:\n\n```python\n@batch.able(batch_size=10)\ndef lookup_shop(shop_ids):\n    return {\n        row["id"]: dict(row)\n        for row in db.execute(\n            "SELECT id, name FROM shops WHERE id=ANY({ids})",\n            ids=tuple(shop_id),\n        ),\n    }\n```\n\nYou still call this function with a single shop ID, with no functional changes.\nYou can, however, also call it inside a context manager:\n\n```python\nwith batch.ed:\n    processed_offers = [transform_offer(offer) for offer in unprocessed_offers]\n```\n\nThis is again functionally identical, _but_ `lookup_shop` gets called with (up\nto) 10 shop IDs at a time. You can also provide a `default=` argument to the\ndecorator to set a default value for missing rows (otherwise missing rows will\nraise an exception).\n\nIf you want, you can also add a cache to this function — make sure to add it\n_on top_ of the `@batch.able` decorator, so it caches per ID.\n\n\n## Caveats\n\nThe way this works is by having the lookup function return `Proxy` objects that\nare later (either when the batch size is reached, or when leaving the context\nmanager) magically replaced by the actual object. The proxy knows about\nindexing and attribute access, so that will just work as well. The level of\nmagic means however that there are limitations to this technique:\n\n- **CPython only:** proxies are replaced with a devious technique involving the\n  reference-counting garbage collector, meaning this won\'t work on\n  implementations without one (e.g. PyPy).\n- **no thread-safety:** to be honest, it will _probably usually_ just work, but\n  we sure as hell don\'t guarantee it. We do a `gc.collect()` immediately before\n  asking the GC for references to the proxy, but in the meantime a different\n  thread could have decremented the reference count, meaning we could get\n  half-dead objects that haven\'t been reaped yet.\n- **no tuples:** we only replace references in lists and dicts (including\n  instance dictionaries). That means that we are not able to replace references\n  in tuples. It would technically be possible to do this, but the way this\n  library works is surprising enough; we didn\'t want to violate the "immutable\n  objects can\'t be changed" rule.\n- **IDs must be hashable:** probably a no-brainer, but the IDs used as\n  arguments to the lookup functions must be hashable. They almost always are\n  anyways.\n- **no intermediate use:** This is the most dangerous foot-gun. Make sure not\n  to use results of calling `transform_offer` _until you have left the context\n  manager_, because the proxies may not all have been replaced yet.\n\n\n## Complete example\n\nA more complete example can be seen in the file `usage.py`. When executing it,\nobserve where the `Proxy` objects are still shown, and where they have\ndisappeared.\n',
    'author': 'L3viathan',
    'author_email': 'git@l3vi.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/L3viathan/batchable',
    'py_modules': modules,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
