"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const assertions_1 = require("aws-cdk-lib/assertions");
const test_helper_1 = require("./test-helper");
test('New service/new bucket, public API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const familyName = "family-name";
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const construct = new lib_1.FargateToKinesisFirehose(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(construct.vpc !== null);
    expect(construct.service !== null);
    expect(construct.container !== null);
    expect(construct.kinesisFirehose !== null);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: familyName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: clusterName
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    // Confirm we created a Public/Private VPC
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('New service/new bucket, private API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::S3::Bucket', 2);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const construct = new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.kinesisFirehose).toBeDefined();
});
test('Construct deploys Fargate Service in isolated subnets when publicApi is set to false', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    const template = assertions_1.Template.fromStack(stack);
    // The default isolated VPC should have two subnets, 2 route tables, and no nat/internet gateway, or routes
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    expect(stack).toCountResources('AWS::EC2::RouteTable', 2);
    expect(stack).toCountResources('AWS::EC2::Route', 0);
    expect(stack).toCountResources('AWS::EC2::NatGateway', 0);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
    // Get the first isolated subnet
    const subnet1Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/IsolatedSubnet1")
                }
            ]
        }
    });
    const [subnet1] = Object.keys(subnet1Resource);
    // Get the second isolated subnet
    const subnet2Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/IsolatedSubnet2")
                }
            ]
        }
    });
    const [subnet2] = Object.keys(subnet2Resource);
    // Verify the Fargate Service is deployed into the two isolated subnets
    expect(stack).toHaveResourceLike('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: subnet1
                    },
                    {
                        Ref: subnet2
                    }
                ]
            },
        }
    });
});
test('Construct deploys Fargate Service in private subnets when publicApi is set to true', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    // The default public/private VPC should have 4 subnets (two public, two private)
    // 4 route tables, 4 routes, 2 NAT Gateways and 1 Internet Gateway
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::EC2::Subnet', 4);
    expect(stack).toCountResources('AWS::EC2::RouteTable', 4);
    expect(stack).toCountResources('AWS::EC2::Route', 4);
    expect(stack).toCountResources('AWS::EC2::NatGateway', 2);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 1);
    const template = assertions_1.Template.fromStack(stack);
    // Get the first private subnet
    const privateSubnet1Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/PrivateSubnet1")
                }
            ]
        }
    });
    const [privateSubnet1] = Object.keys(privateSubnet1Resource);
    // Get the second private subnet
    const privateSubnet2Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/PrivateSubnet2")
                }
            ]
        }
    });
    const [privateSubnet2] = Object.keys(privateSubnet2Resource);
    // Verify the Fargate Service is deployed into the two private subnets
    expect(stack).toHaveResourceLike('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: privateSubnet1
                    },
                    {
                        Ref: privateSubnet2
                    }
                ]
            },
        }
    });
});
test('Construct uses vpcProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose,
        vpcProps: {
            vpcName: 'my-vpc'
        }
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct uses existingVpc when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const existingVpc = defaults.getTestVpc(stack, false, {
        vpcName: 'my-vpc'
    });
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose,
        existingVpc
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct creates VPC Interface Endpoints for ECR and Kinesis Streams', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-firehose'
                ]
            ]
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.ecr.api'
                ]
            ]
        },
    });
});
test('Container has default stream name environment variable', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'FIREHOSE_DELIVERYSTREAM_NAME',
                        Value: destination.kinesisFirehose.deliveryStreamName
                    }
                ],
            }
        ]
    });
});
test('Container stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        firehoseEnvironmentVariableName: 'my-stream-name'
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'my-stream-name',
                        Value: destination.kinesisFirehose.deliveryStreamName
                    }
                ],
            }
        ]
    });
});
test('Construct grants PutRecord permission for the Fargate Service to write to the Kinesis Stream', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: 'Allow',
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Construct defaults to the latest version of the ECR image', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:latest'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses ecrImageVersion when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        ecrImageVersion: 'my-version'
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:my-version'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses clusterProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: {
            clusterName: 'my-cluster'
        }
    });
    expect(stack).toCountResources('AWS::ECS::Cluster', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
});
test('Construct uses containerDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        containerDefinitionProps: {
            containerName: 'my-container'
        }
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
});
test('Construct uses fargateTaskDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateTaskDefinitionProps: {
            family: 'my-family'
        }
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: 'my-family'
    });
});
test('Construct uses fargateServiceProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateServiceProps: {
            serviceName: 'my-service',
            desiredCount: 7
        }
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: 'my-service',
        DesiredCount: 7
    });
});
test('Construct uses existingFargateServiceObject when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const existingVpc = defaults.getTestVpc(stack);
    const [existingFargateServiceObject, existingContainerDefinitionObject] = defaults.CreateFargateService(stack, 'test-existing-fargate-service', existingVpc, { clusterName: 'my-cluster' }, defaults.fakeEcrRepoArn, undefined, { family: 'my-family' }, { containerName: 'my-container' }, { serviceName: 'my-service' });
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingVpc,
        existingKinesisFirehose: destination.kinesisFirehose,
        existingFargateServiceObject,
        existingContainerDefinitionObject
    });
    expect(stack).toCountResources('AWS::ECS::Cluster', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: 'my-family',
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: 'my-service',
    });
});
test('Test fail if existingFirehose does not have a stream name', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const mutatedFirehose = defaults.overrideProps(destination.kinesisFirehose, {});
    mutatedFirehose.deliveryStreamName = undefined;
    const app = () => {
        new lib_1.FargateToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: mutatedFirehose,
            publicApi: false,
            ecrRepositoryArn: defaults.fakeEcrRepoArn,
        });
    };
    expect(app).toThrowError(/existingKinesisFirehose must have a defined deliveryStreamName/);
});
//# sourceMappingURL=data:application/json;base64,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