"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
describe('profiling group', () => {
    test('attach read permission to Profiling group via fromProfilingGroupArn', () => {
        const stack = new core_1.Stack();
        // dummy role to test out read permissions on ProfilingGroup
        const readAppRole = new aws_iam_1.Role(stack, 'ReadAppRole', {
            assumedBy: new aws_iam_1.AccountRootPrincipal(),
        });
        const profilingGroup = lib_1.ProfilingGroup.fromProfilingGroupArn(stack, 'MyProfilingGroup', 'arn:aws:codeguru-profiler:us-east-1:1234567890:profilingGroup/MyAwesomeProfilingGroup');
        profilingGroup.grantRead(readAppRole);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'ReadAppRole52FE6317': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        'Ref': 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        'Ref': 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                },
                'ReadAppRoleDefaultPolicy4BB8955C': {
                    'Type': 'AWS::IAM::Policy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': [
                                        'codeguru-profiler:GetProfile',
                                        'codeguru-profiler:DescribeProfilingGroup',
                                    ],
                                    'Effect': 'Allow',
                                    'Resource': 'arn:aws:codeguru-profiler:us-east-1:1234567890:profilingGroup/MyAwesomeProfilingGroup',
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'PolicyName': 'ReadAppRoleDefaultPolicy4BB8955C',
                        'Roles': [
                            {
                                'Ref': 'ReadAppRole52FE6317',
                            },
                        ],
                    },
                },
            },
        });
    });
    test('attach publish permission to Profiling group via fromProfilingGroupName', () => {
        const stack = new core_1.Stack();
        // dummy role to test out publish permissions on ProfilingGroup
        const publishAppRole = new aws_iam_1.Role(stack, 'PublishAppRole', {
            assumedBy: new aws_iam_1.AccountRootPrincipal(),
        });
        const profilingGroup = lib_1.ProfilingGroup.fromProfilingGroupName(stack, 'MyProfilingGroup', 'MyAwesomeProfilingGroup');
        profilingGroup.grantPublish(publishAppRole);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'PublishAppRole9FEBD682': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        'Ref': 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        'Ref': 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                },
                'PublishAppRoleDefaultPolicyCA1E15C3': {
                    'Type': 'AWS::IAM::Policy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': [
                                        'codeguru-profiler:ConfigureAgent',
                                        'codeguru-profiler:PostAgentProfile',
                                    ],
                                    'Effect': 'Allow',
                                    'Resource': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':codeguru-profiler:',
                                                {
                                                    'Ref': 'AWS::Region',
                                                },
                                                ':',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':profilingGroup/MyAwesomeProfilingGroup',
                                            ],
                                        ],
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'PolicyName': 'PublishAppRoleDefaultPolicyCA1E15C3',
                        'Roles': [
                            {
                                'Ref': 'PublishAppRole9FEBD682',
                            },
                        ],
                    },
                },
            },
        });
    });
    test('default profiling group', () => {
        const stack = new core_1.Stack();
        new lib_1.ProfilingGroup(stack, 'MyProfilingGroup', {
            profilingGroupName: 'MyAwesomeProfilingGroup',
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyProfilingGroup829F0507': {
                    'Type': 'AWS::CodeGuruProfiler::ProfilingGroup',
                    'Properties': {
                        'ProfilingGroupName': 'MyAwesomeProfilingGroup',
                    },
                },
            },
        });
    });
    test('allows setting its ComputePlatform', () => {
        const stack = new core_1.Stack();
        new lib_1.ProfilingGroup(stack, 'MyProfilingGroup', {
            profilingGroupName: 'MyAwesomeProfilingGroup',
            computePlatform: lib_1.ComputePlatform.AWS_LAMBDA,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeGuruProfiler::ProfilingGroup', {
            'ComputePlatform': 'AWSLambda',
        }));
    });
    test('default profiling group without name', () => {
        const stack = new core_1.Stack();
        new lib_1.ProfilingGroup(stack, 'MyProfilingGroup', {});
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyProfilingGroup829F0507': {
                    'Type': 'AWS::CodeGuruProfiler::ProfilingGroup',
                    'Properties': {
                        'ProfilingGroupName': 'MyProfilingGroup',
                    },
                },
            },
        });
    });
    test('default profiling group without name when name exceeding limit is generated', () => {
        const stack = new core_1.Stack();
        new lib_1.ProfilingGroup(stack, 'MyProfilingGroupWithAReallyLongProfilingGroupNameThatExceedsTheLimitOfProfilingGroupNameSize_InOrderToDoSoTheNameMustBeGreaterThanTwoHundredAndFiftyFiveCharacters_InSuchCasesWePickUpTheFirstOneTwentyCharactersFromTheBeginningAndTheEndAndConcatenateThemToGetTheIdentifier', {});
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyProfilingGroupWithAReallyLongProfilingGroupNameThatExceedsTheLimitOfProfilingGroupNameSizeInOrderToDoSoTheNameMustBeGreaterThanTwoHundredAndFiftyFiveCharactersInSuchCasesWePickUpTheFirstOneTwentyCharactersFromTheBeginningAndTheEndAndConca4B39908C': {
                    'Type': 'AWS::CodeGuruProfiler::ProfilingGroup',
                    'Properties': {
                        'ProfilingGroupName': 'MyProfilingGroupWithAReallyLongProfilingGroupNameThatExceedsTheLimitOfProfilingGroupNameSizeInOrderToDoSoTheNameMustBeGrnTwoHundredAndFiftyFiveCharactersInSuchCasesWePickUpTheFirstOneTwentyCharactersFromTheBeginningAndTheEndAndConca2FE009B0',
                    },
                },
            },
        });
    });
    test('grant publish permissions profiling group', () => {
        const stack = new core_1.Stack();
        const profilingGroup = new lib_1.ProfilingGroup(stack, 'MyProfilingGroup', {
            profilingGroupName: 'MyAwesomeProfilingGroup',
        });
        const publishAppRole = new aws_iam_1.Role(stack, 'PublishAppRole', {
            assumedBy: new aws_iam_1.AccountRootPrincipal(),
        });
        profilingGroup.grantPublish(publishAppRole);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyProfilingGroup829F0507': {
                    'Type': 'AWS::CodeGuruProfiler::ProfilingGroup',
                    'Properties': {
                        'ProfilingGroupName': 'MyAwesomeProfilingGroup',
                    },
                },
                'PublishAppRole9FEBD682': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        'Ref': 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        'Ref': 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                },
                'PublishAppRoleDefaultPolicyCA1E15C3': {
                    'Type': 'AWS::IAM::Policy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': [
                                        'codeguru-profiler:ConfigureAgent',
                                        'codeguru-profiler:PostAgentProfile',
                                    ],
                                    'Effect': 'Allow',
                                    'Resource': {
                                        'Fn::GetAtt': [
                                            'MyProfilingGroup829F0507',
                                            'Arn',
                                        ],
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'PolicyName': 'PublishAppRoleDefaultPolicyCA1E15C3',
                        'Roles': [
                            {
                                'Ref': 'PublishAppRole9FEBD682',
                            },
                        ],
                    },
                },
            },
        });
    });
    test('grant read permissions profiling group', () => {
        const stack = new core_1.Stack();
        const profilingGroup = new lib_1.ProfilingGroup(stack, 'MyProfilingGroup', {
            profilingGroupName: 'MyAwesomeProfilingGroup',
        });
        const readAppRole = new aws_iam_1.Role(stack, 'ReadAppRole', {
            assumedBy: new aws_iam_1.AccountRootPrincipal(),
        });
        profilingGroup.grantRead(readAppRole);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyProfilingGroup829F0507': {
                    'Type': 'AWS::CodeGuruProfiler::ProfilingGroup',
                    'Properties': {
                        'ProfilingGroupName': 'MyAwesomeProfilingGroup',
                    },
                },
                'ReadAppRole52FE6317': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        'Ref': 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        'Ref': 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                },
                'ReadAppRoleDefaultPolicy4BB8955C': {
                    'Type': 'AWS::IAM::Policy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': [
                                        'codeguru-profiler:GetProfile',
                                        'codeguru-profiler:DescribeProfilingGroup',
                                    ],
                                    'Effect': 'Allow',
                                    'Resource': {
                                        'Fn::GetAtt': [
                                            'MyProfilingGroup829F0507',
                                            'Arn',
                                        ],
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'PolicyName': 'ReadAppRoleDefaultPolicy4BB8955C',
                        'Roles': [
                            {
                                'Ref': 'ReadAppRole52FE6317',
                            },
                        ],
                    },
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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