from flask import Blueprint, render_template, request, url_for, redirect, current_app, abort
from flask_babel import lazy_gettext

from uffd.navbar import register_navbar
from uffd.csrf import csrf_protect
from uffd.session import login_required
from uffd.service.models import Service, get_services
from uffd.user.models import Group
from uffd.oauth2.models import OAuth2Client, OAuth2LogoutURI
from uffd.api.models import APIClient
from uffd.database import db

bp = Blueprint('service', __name__, template_folder='templates')

def service_admin_acl():
	return request.user and request.user.is_in_group(current_app.config['ACL_ADMIN_GROUP'])

def service_overview_acl():
	return len(get_services(request.user)) > 0 or service_admin_acl()

@bp.route('/services/')
@register_navbar(lazy_gettext('Services'), icon='sitemap', blueprint=bp, visible=service_overview_acl)
def overview():
	services = get_services(request.user)
	if not service_overview_acl():
		abort(404)
	banner = current_app.config.get('SERVICES_BANNER')
	# Set the banner to None if it is not public and no user is logged in
	if not (current_app.config['SERVICES_BANNER_PUBLIC'] or request.user):
		banner = None
	return render_template('service/overview.html', user=request.user, services=services, banner=banner)

@bp.route('/service/admin')
@login_required(service_admin_acl)
def index():
	return render_template('service/index.html', services=Service.query.all())

@bp.route('/service/new')
@bp.route('/service/<int:id>')
@login_required(service_admin_acl)
def show(id=None):
	service = Service() if id is None else Service.query.get_or_404(id)
	all_groups = Group.query.all()
	return render_template('service/show.html', service=service, all_groups=all_groups)

@bp.route('/service/new', methods=['POST'])
@bp.route('/service/<int:id>', methods=['POST'])
@csrf_protect(blueprint=bp)
@login_required(service_admin_acl)
def edit_submit(id=None):
	if id is None:
		service = Service()
		db.session.add(service)
	else:
		service = Service.query.get_or_404(id)
	service.name = request.form['name']
	if not request.form['access-group']:
		service.limit_access = True
		service.access_group = None
	elif request.form['access-group'] == 'all':
		service.limit_access = False
		service.access_group = None
	else:
		service.limit_access = True
		service.access_group = Group.query.get(request.form['access-group'])
	db.session.commit()
	return redirect(url_for('service.show', id=service.id))

@bp.route('/service/<int:id>/delete')
@csrf_protect(blueprint=bp)
@login_required(service_admin_acl)
def delete(id):
	service = Service.query.get_or_404(id)
	db.session.delete(service)
	db.session.commit()
	return redirect(url_for('service.index'))

@bp.route('/service/<int:service_id>/oauth2/new')
@bp.route('/service/<int:service_id>/oauth2/<int:db_id>')
@login_required(service_admin_acl)
def oauth2_show(service_id, db_id=None):
	service = Service.query.get_or_404(service_id)
	client = OAuth2Client() if db_id is None else OAuth2Client.query.filter_by(service_id=service_id, db_id=db_id).first_or_404()
	return render_template('service/oauth2.html', service=service, client=client)

@bp.route('/service/<int:service_id>/oauth2/new', methods=['POST'])
@bp.route('/service/<int:service_id>/oauth2/<int:db_id>', methods=['POST'])
@csrf_protect(blueprint=bp)
@login_required(service_admin_acl)
def oauth2_submit(service_id, db_id=None):
	service = Service.query.get_or_404(service_id)
	if db_id is None:
		client = OAuth2Client(service=service)
		db.session.add(client)
	else:
		client = OAuth2Client.query.filter_by(service_id=service_id, db_id=db_id).first_or_404()
	client.client_id = request.form['client_id']
	if request.form['client_secret']:
		client.client_secret = request.form['client_secret']
	if not client.client_secret:
		abort(400)
	client.redirect_uris = request.form['redirect_uris'].strip().split('\n')
	client.logout_uris = []
	for line in request.form['logout_uris'].split('\n'):
		line = line.strip()
		if not line:
			continue
		method, uri = line.split(' ', 2)
		client.logout_uris.append(OAuth2LogoutURI(method=method, uri=uri))
	db.session.commit()
	return redirect(url_for('service.show', id=service.id))

@bp.route('/service/<int:service_id>/oauth2/<int:db_id>/delete')
@csrf_protect(blueprint=bp)
@login_required(service_admin_acl)
def oauth2_delete(service_id, db_id=None):
	service = Service.query.get_or_404(service_id)
	client = OAuth2Client.query.filter_by(service_id=service_id, db_id=db_id).first_or_404()
	db.session.delete(client)
	db.session.commit()
	return redirect(url_for('service.show', id=service.id))

@bp.route('/service/<int:service_id>/api/new')
@bp.route('/service/<int:service_id>/api/<int:id>')
@login_required(service_admin_acl)
def api_show(service_id, id=None):
	service = Service.query.get_or_404(service_id)
	client = APIClient() if id is None else APIClient.query.filter_by(service_id=service_id, id=id).first_or_404()
	return render_template('service/api.html', service=service, client=client)

@bp.route('/service/<int:service_id>/api/new', methods=['POST'])
@bp.route('/service/<int:service_id>/api/<int:id>', methods=['POST'])
@csrf_protect(blueprint=bp)
@login_required(service_admin_acl)
def api_submit(service_id, id=None):
	service = Service.query.get_or_404(service_id)
	if id is None:
		client = APIClient(service=service)
		db.session.add(client)
	else:
		client = APIClient.query.filter_by(service_id=service_id, id=id).first_or_404()
	client.auth_username = request.form['auth_username']
	if request.form['auth_password']:
		client.auth_password = request.form['auth_password']
	if not client.auth_password:
		abort(400)
	client.perm_users = request.form.get('perm_users') == '1'
	client.perm_checkpassword = request.form.get('perm_checkpassword') == '1'
	client.perm_mail_aliases = request.form.get('perm_mail_aliases') == '1'
	db.session.commit()
	return redirect(url_for('service.show', id=service.id))

@bp.route('/service/<int:service_id>/api/<int:id>/delete')
@csrf_protect(blueprint=bp)
@login_required(service_admin_acl)
def api_delete(service_id, id=None):
	service = Service.query.get_or_404(service_id)
	client = APIClient.query.filter_by(service_id=service_id, id=id).first_or_404()
	db.session.delete(client)
	db.session.commit()
	return redirect(url_for('service.show', id=service.id))
