"use strict";
const lib_1 = require("../lib");
const encoding_1 = require("../lib/encoding");
const token_map_1 = require("../lib/token-map");
const evaluate_cfn_1 = require("./evaluate-cfn");
class Promise2 extends lib_1.Token {
    resolve() {
        return {
            Data: {
                stringProp: 'hello',
                numberProp: 1234,
            },
            Recurse: new lib_1.Token(() => 42)
        };
    }
}
class Promise1 extends lib_1.Token {
    constructor() {
        super(...arguments);
        this.p2 = [new Promise2(), new Promise2()];
    }
    resolve() {
        return this.p2;
    }
}
class BaseDataType {
    constructor(foo) {
        this.foo = foo;
    }
}
class DataType extends BaseDataType {
    constructor() {
        super(12);
        this.goo = 'hello';
    }
}
/**
 * Return various flavors of Tokens that resolve to the given value
 */
function literalTokensThatResolveTo(value) {
    return [
        new lib_1.Token(value),
        new lib_1.Token(() => value)
    ];
}
/**
 * Return various flavors of Tokens that resolve to the given value
 */
function cloudFormationTokensThatResolveTo(value) {
    return [
        new lib_1.Token(value),
        new lib_1.Token(() => value)
    ];
}
/**
 * Return Tokens in both flavors that resolve to the given string
 */
function tokensThatResolveTo(value) {
    return literalTokensThatResolveTo(value).concat(cloudFormationTokensThatResolveTo(value));
}
/**
 * Wrapper for resolve that creates an throwaway Construct to call it on
 *
 * So I don't have to change all call sites in this file.
 */
function resolve(x) {
    return new lib_1.App().node.resolve(x);
}
module.exports = {
    'resolve a plain old object should just return the object'(test) {
        const obj = { PlainOldObject: 123, Array: [1, 2, 3] };
        test.deepEqual(resolve(obj), obj);
        test.done();
    },
    'if a value is an object with a token value, it will be evaluated'(test) {
        const obj = {
            RegularValue: 'hello',
            LazyValue: new lib_1.Token('World')
        };
        test.deepEqual(resolve(obj), {
            RegularValue: 'hello',
            LazyValue: 'World'
        });
        test.done();
    },
    'tokens are evaluated anywhere in the object tree'(test) {
        const obj = new Promise1();
        const actual = resolve({ Obj: obj });
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: "hello",
                        numberProp: 1234
                    },
                    Recurse: 42
                },
                {
                    Data: {
                        stringProp: "hello",
                        numberProp: 1234
                    },
                    Recurse: 42
                }
            ]
        });
        test.done();
    },
    'tokens are evaluated recursively'(test) {
        const obj = new Promise1();
        const actual = resolve(new lib_1.Token(() => ({ Obj: obj })));
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: "hello",
                        numberProp: 1234
                    },
                    Recurse: 42
                },
                {
                    Data: {
                        stringProp: "hello",
                        numberProp: 1234
                    },
                    Recurse: 42
                }
            ]
        });
        test.done();
    },
    'empty arrays or objects are kept'(test) {
        test.deepEqual(resolve({}), {});
        test.deepEqual(resolve([]), []);
        const obj = {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [undefined, undefined],
                    PropD: 'Yoohoo'
                }
            }
        };
        test.deepEqual(resolve(obj), {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [],
                    PropD: 'Yoohoo'
                }
            }
        });
        test.done();
    },
    'if an object has a "resolve" property that is not a function, it is not considered a token'(test) {
        test.deepEqual(resolve({ a_token: { resolve: () => 78787 } }), { a_token: 78787 });
        test.deepEqual(resolve({ not_a_token: { resolve: 12 } }), { not_a_token: { resolve: 12 } });
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'if a resolvable object inherits from a class that is also resolvable, the "constructor" function will not get in the way (uses Object.keys instead of "for in")'(test) {
        test.deepEqual(resolve({ prop: new DataType() }), { prop: { foo: 12, goo: 'hello' } });
        test.done();
    },
    'isToken(obj) can be used to determine if an object is a token'(test) {
        test.ok(lib_1.Token.isToken({ resolve: () => 123 }));
        test.ok(lib_1.Token.isToken({ a: 1, b: 2, resolve: () => 'hello' }));
        test.ok(!lib_1.Token.isToken({ a: 1, b: 2, resolve: 3 }));
        test.done();
    },
    'Token can be used to create tokens that contain a constant value'(test) {
        test.equal(resolve(new lib_1.Token(12)), 12);
        test.equal(resolve(new lib_1.Token('hello')), 'hello');
        test.deepEqual(resolve(new lib_1.Token(['hi', 'there'])), ['hi', 'there']);
        test.done();
    },
    'resolving leaves a Date object in working order'(test) {
        const date = new Date('2000-01-01');
        const resolved = resolve(date);
        test.equal(date.toString(), resolved.toString());
        test.done();
    },
    'tokens can be stringified and evaluated to conceptual value'(test) {
        // GIVEN
        const token = new lib_1.Token(() => 'woof woof');
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'The dog says: woof woof');
        test.done();
    },
    'tokens stringification can be reversed'(test) {
        // GIVEN
        const token = new lib_1.Token(() => 'woof woof');
        // THEN
        test.equal(token, token_map_1.TokenMap.instance().lookupString(`${token}`));
        test.done();
    },
    'concatenated tokens are undefined'(test) {
        // GIVEN
        const token = new lib_1.Token(() => 'woof woof');
        // WHEN
        test.equal(undefined, token_map_1.TokenMap.instance().lookupString(`${token}bla`));
        test.equal(undefined, token_map_1.TokenMap.instance().lookupString(`bla${token}`));
        test.equal(undefined, token_map_1.TokenMap.instance().lookupString(`bla`));
        test.done();
    },
    'Tokens stringification and reversing of CloudFormation Tokens is implemented using Fn::Join'(test) {
        // GIVEN
        const token = new lib_1.Token(() => ({ woof: 'woof' }));
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(resolved, {
            'Fn::Join': ['', ['The dog says: ', { woof: 'woof' }]]
        });
        test.done();
    },
    'Doubly nested strings evaluate correctly in scalar context'(test) {
        // GIVEN
        const token1 = new lib_1.Token(() => "world");
        const token2 = new lib_1.Token(() => `hello ${token1}`);
        // WHEN
        const resolved1 = resolve(token2.toString());
        const resolved2 = resolve(token2);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved1), "hello world");
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved2), "hello world");
        test.done();
    },
    'integer Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const token of literalTokensThatResolveTo(1)) {
            // WHEN
            const stringified = `the number is ${token}`;
            const resolved = resolve(stringified);
            // THEN
            test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'the number is 1');
        }
        test.done();
    },
    'intrinsic Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const bucketName of cloudFormationTokensThatResolveTo({ Ref: 'MyBucket' })) {
            // WHEN
            const resolved = resolve(`my bucket is named ${bucketName}`);
            // THEN
            const context = { MyBucket: 'TheName' };
            test.equal(evaluate_cfn_1.evaluateCFN(resolved, context), 'my bucket is named TheName');
        }
        test.done();
    },
    'tokens resolve properly in initial position'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo('Hello')) {
            // WHEN
            const resolved = resolve(`${token} world`);
            // THEN
            test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
        }
        test.done();
    },
    'side-by-side Tokens resolve correctly'(test) {
        // GIVEN
        for (const token1 of tokensThatResolveTo('Hello ')) {
            for (const token2 of tokensThatResolveTo('world')) {
                // WHEN
                const resolved = resolve(`${token1}${token2}`);
                // THEN
                test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
            }
        }
        test.done();
    },
    'tokens can be used in hash keys but must resolve to a string'(test) {
        // GIVEN
        const token = new lib_1.Token(() => 'I am a string');
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested in hash keys'(test) {
        // GIVEN
        const token = new lib_1.Token(() => new lib_1.Token(() => new lib_1.Token(() => 'I am a string')));
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested and concatenated in hash keys'(test) {
        // GIVEN
        const innerToken = new lib_1.Token(() => 'toot');
        const token = new lib_1.Token(() => `${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: `boom chicago`
        };
        // THEN
        test.deepEqual(resolve(s), { 'toot the woot': 'boom chicago' });
        test.done();
    },
    'can find nested tokens in hash keys'(test) {
        // GIVEN
        const innerToken = new lib_1.Token(() => 'toot');
        const token = new lib_1.Token(() => `${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: `boom chicago`
        };
        // THEN
        const tokens = lib_1.findTokens(new lib_1.Stack(), () => s);
        test.ok(tokens.some(t => t === innerToken), 'Cannot find innerToken');
        test.ok(tokens.some(t => t === token), 'Cannot find token');
        test.done();
    },
    'fails if token in a hash key resolves to a non-string'(test) {
        // GIVEN
        const token = new lib_1.Token({ Ref: 'Other' });
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`
        };
        // THEN
        test.throws(() => resolve(s), 'The key "${Token[TOKEN.19]}" has been resolved to {"Ref":"Other"} but must be resolvable to a string');
        test.done();
    },
    'list encoding': {
        'can encode Token to string and resolve the encoding'(test) {
            // GIVEN
            const token = new lib_1.Token({ Ref: 'Other' });
            // WHEN
            const struct = {
                XYZ: token.toList()
            };
            // THEN
            test.deepEqual(resolve(struct), {
                XYZ: { Ref: 'Other' }
            });
            test.done();
        },
        'cannot add to encoded list'(test) {
            // GIVEN
            const token = new lib_1.Token({ Ref: 'Other' });
            // WHEN
            const encoded = token.toList();
            encoded.push('hello');
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /Cannot add elements to list token/);
            test.done();
        },
        'cannot add to strings in encoded list'(test) {
            // GIVEN
            const token = new lib_1.Token({ Ref: 'Other' });
            // WHEN
            const encoded = token.toList();
            encoded[0] += 'hello';
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /concatenate strings in/);
            test.done();
        },
        'can pass encoded lists to FnSelect'(test) {
            // GIVEN
            const encoded = new lib_1.Token({ Ref: 'Other' }).toList();
            // WHEN
            const struct = lib_1.Fn.select(1, encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Select': [1, { Ref: 'Other' }]
            });
            test.done();
        },
        'can pass encoded lists to FnJoin'(test) {
            // GIVEN
            const encoded = new lib_1.Token({ Ref: 'Other' }).toList();
            // WHEN
            const struct = lib_1.Fn.join('/', encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }]
            });
            test.done();
        },
        'can pass encoded lists to FnJoin, even if join is stringified'(test) {
            // GIVEN
            const encoded = new lib_1.Token({ Ref: 'Other' }).toList();
            // WHEN
            const struct = lib_1.Fn.join('/', encoded).toString();
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }]
            });
            test.done();
        },
    },
    'number encoding': {
        'basic integer encoding works'(test) {
            test.equal(16, encoding_1.extractTokenDouble(encoding_1.createTokenDouble(16)));
            test.done();
        },
        'arbitrary integers can be encoded, stringified, and recovered'(test) {
            for (let i = 0; i < 100; i++) {
                // We can encode all numbers up to 2^48-1
                const x = Math.floor(Math.random() * (Math.pow(2, 48) - 1));
                const encoded = encoding_1.createTokenDouble(x);
                // Roundtrip through JSONification
                const roundtripped = JSON.parse(JSON.stringify({ theNumber: encoded })).theNumber;
                const decoded = encoding_1.extractTokenDouble(roundtripped);
                test.equal(decoded, x, `Fail roundtrip encoding of ${x}`);
            }
            test.done();
        },
        'arbitrary numbers are correctly detected as not being tokens'(test) {
            test.equal(undefined, encoding_1.extractTokenDouble(0));
            test.equal(undefined, encoding_1.extractTokenDouble(1243));
            test.equal(undefined, encoding_1.extractTokenDouble(4835e+532));
            test.done();
        },
        'can number-encode and resolve Token objects'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const x = new lib_1.Token(() => 123);
            // THEN
            const encoded = x.toNumber();
            test.equal(true, lib_1.Token.isToken(encoded), 'encoded number does not test as token');
            // THEN
            const resolved = stack.node.resolve({ value: encoded });
            test.deepEqual(resolved, { value: 123 });
            test.done();
        },
    },
    'stack trace is captured at token creation'(test) {
        function fn1() {
            function fn2() {
                class ExposeTrace extends lib_1.Token {
                    get creationTrace() {
                        return this.trace;
                    }
                }
                return new ExposeTrace(() => 'hello');
            }
            return fn2();
        }
        const token = fn1();
        test.ok(token.creationTrace.find(x => x.includes('fn1')));
        test.ok(token.creationTrace.find(x => x.includes('fn2')));
        test.done();
    },
    'newError returns an error with the creation stack trace'(test) {
        function fn1() {
            function fn2() {
                function fn3() {
                    class ThrowingToken extends lib_1.Token {
                        throwError(message) {
                            throw this.newError(message);
                        }
                    }
                    return new ThrowingToken('boom');
                }
                return fn3();
            }
            return fn2();
        }
        const token = fn1();
        test.throws(() => token.throwError('message!'), /Token created:/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC50b2tlbnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnRva2Vucy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQ0EsZ0NBQW1FO0FBQ25FLDhDQUF3RTtBQUN4RSxnREFBNEM7QUFDNUMsaURBQTZDO0FBMmdCN0MsTUFBTSxRQUFTLFNBQVEsV0FBSztJQUNuQixPQUFPO1FBQ1osT0FBTztZQUNMLElBQUksRUFBRTtnQkFDSixVQUFVLEVBQUUsT0FBTztnQkFDbkIsVUFBVSxFQUFFLElBQUk7YUFDakI7WUFDRCxPQUFPLEVBQUUsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDO1NBQzdCLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFFRCxNQUFNLFFBQVMsU0FBUSxXQUFLO0lBQTVCOztRQUNTLE9BQUUsR0FBRyxDQUFFLElBQUksUUFBUSxFQUFFLEVBQUUsSUFBSSxRQUFRLEVBQUUsQ0FBRSxDQUFDO0lBS2pELENBQUM7SUFIUSxPQUFPO1FBQ1osT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDO0lBQ2pCLENBQUM7Q0FDRjtBQUVELE1BQU0sWUFBWTtJQUNoQixZQUFxQixHQUFXO1FBQVgsUUFBRyxHQUFILEdBQUcsQ0FBUTtJQUNoQyxDQUFDO0NBQ0Y7QUFFRCxNQUFNLFFBQVMsU0FBUSxZQUFZO0lBR2pDO1FBQ0UsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBSEwsUUFBRyxHQUFHLE9BQU8sQ0FBQztJQUlyQixDQUFDO0NBQ0Y7QUFFRDs7R0FFRztBQUNILFNBQVMsMEJBQTBCLENBQUMsS0FBVTtJQUM1QyxPQUFPO1FBQ0wsSUFBSSxXQUFLLENBQUMsS0FBSyxDQUFDO1FBQ2hCLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQztLQUN2QixDQUFDO0FBQ0osQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxpQ0FBaUMsQ0FBQyxLQUFVO0lBQ25ELE9BQU87UUFDTCxJQUFJLFdBQUssQ0FBQyxLQUFLLENBQUM7UUFDaEIsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDO0tBQ3ZCLENBQUM7QUFDSixDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLG1CQUFtQixDQUFDLEtBQWE7SUFDeEMsT0FBTywwQkFBMEIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsaUNBQWlDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztBQUM1RixDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQVMsT0FBTyxDQUFDLENBQU07SUFDckIsT0FBTyxJQUFJLFNBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDcEMsQ0FBQztBQTVrQkQsaUJBQVM7SUFDUCwwREFBMEQsQ0FBQyxJQUFVO1FBQ25FLE1BQU0sR0FBRyxHQUFHLEVBQUUsY0FBYyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsQ0FBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBRSxFQUFFLENBQUM7UUFDeEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDbEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtFQUFrRSxDQUFDLElBQVU7UUFDM0UsTUFBTSxHQUFHLEdBQUc7WUFDVixZQUFZLEVBQUUsT0FBTztZQUNyQixTQUFTLEVBQUUsSUFBSSxXQUFLLENBQUMsT0FBTyxDQUFDO1NBQzlCLENBQUM7UUFFRixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUMzQixZQUFZLEVBQUUsT0FBTztZQUNyQixTQUFTLEVBQUUsT0FBTztTQUNuQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0RBQWtELENBQUMsSUFBVTtRQUMzRCxNQUFNLEdBQUcsR0FBRyxJQUFJLFFBQVEsRUFBRSxDQUFDO1FBQzNCLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBRXJDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFO1lBQ3JCLEdBQUcsRUFBRTtnQkFDSDtvQkFDQSxJQUFJLEVBQUU7d0JBQ0osVUFBVSxFQUFFLE9BQU87d0JBQ25CLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxPQUFPLEVBQUUsRUFBRTtpQkFDVjtnQkFDRDtvQkFDQSxJQUFJLEVBQUU7d0JBQ0osVUFBVSxFQUFFLE9BQU87d0JBQ25CLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxPQUFPLEVBQUUsRUFBRTtpQkFDVjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtDQUFrQyxDQUFDLElBQVU7UUFDM0MsTUFBTSxHQUFHLEdBQUcsSUFBSSxRQUFRLEVBQUUsQ0FBQztRQUMzQixNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUV4RCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRTtZQUNyQixHQUFHLEVBQUU7Z0JBQ0g7b0JBQ0EsSUFBSSxFQUFFO3dCQUNKLFVBQVUsRUFBRSxPQUFPO3dCQUNuQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFLEVBQUU7aUJBQ1Y7Z0JBQ0Q7b0JBQ0EsSUFBSSxFQUFFO3dCQUNKLFVBQVUsRUFBRSxPQUFPO3dCQUNuQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsT0FBTyxFQUFFLEVBQUU7aUJBQ1Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrQ0FBa0MsQ0FBQyxJQUFVO1FBQzNDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEVBQUcsQ0FBQyxFQUFFLEVBQUcsQ0FBQyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEVBQUcsQ0FBQyxFQUFFLEVBQUcsQ0FBQyxDQUFDO1FBRWxDLE1BQU0sR0FBRyxHQUFHO1lBQ1YsS0FBSyxFQUFFLElBQUk7WUFDWCxLQUFLLEVBQUUsRUFBRztZQUNWLEtBQUssRUFBRSxFQUFHO1lBQ1YsS0FBSyxFQUFFLE9BQU87WUFDZCxLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFLEVBQUc7Z0JBQ1YsS0FBSyxFQUFFO29CQUNMLEtBQUssRUFBRSxDQUFFLFNBQVMsRUFBRSxTQUFTLENBQUU7b0JBQy9CLEtBQUssRUFBRSxRQUFRO2lCQUNoQjthQUNGO1NBQ0YsQ0FBQztRQUVGLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQzNCLEtBQUssRUFBRSxJQUFJO1lBQ1gsS0FBSyxFQUFFLEVBQUc7WUFDVixLQUFLLEVBQUUsRUFBRztZQUNWLEtBQUssRUFBRSxPQUFPO1lBQ2QsS0FBSyxFQUFFO2dCQUNMLEtBQUssRUFBRSxFQUFHO2dCQUNWLEtBQUssRUFBRTtvQkFDTCxLQUFLLEVBQUUsRUFBRztvQkFDVixLQUFLLEVBQUUsUUFBUTtpQkFDaEI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0RkFBNEYsQ0FBQyxJQUFVO1FBQ3JHLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEVBQUUsT0FBTyxFQUFFLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFDLENBQUMsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ2xGLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEVBQUUsV0FBVyxFQUFFLEVBQUUsT0FBTyxFQUFFLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBSSxFQUFFLFdBQVcsRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDOUYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJDQUEyQztJQUMzQyxpS0FBaUssQ0FBQyxJQUFVO1FBQzFLLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksUUFBUSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrREFBK0QsQ0FBQyxJQUFVO1FBQ3hFLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFLLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFdBQUssQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNwRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0VBQWtFLENBQUMsSUFBVTtRQUMzRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLFdBQUssQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksV0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxXQUFLLENBQUMsQ0FBRSxJQUFJLEVBQUUsT0FBTyxDQUFFLENBQUMsQ0FBQyxFQUFFLENBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBRSxDQUFDLENBQUM7UUFDekUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlEQUFpRCxDQUFDLElBQVU7UUFDMUQsTUFBTSxJQUFJLEdBQUcsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDcEMsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRS9CLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUFFLFFBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2REFBNkQsQ0FBQyxJQUFVO1FBQ3RFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUUzQyxPQUFPO1FBQ1AsTUFBTSxXQUFXLEdBQUcsaUJBQWlCLEtBQUssRUFBRSxDQUFDO1FBQzdDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUV0QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQywwQkFBVyxDQUFDLFFBQVEsQ0FBQyxFQUFFLHlCQUF5QixDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHdDQUF3QyxDQUFDLElBQVU7UUFDakQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRTNDLE9BQU87UUFDUCxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUNBQW1DLENBQUMsSUFBVTtRQUM1QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFM0MsT0FBTztRQUNQLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsWUFBWSxDQUFDLEdBQUcsS0FBSyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsWUFBWSxDQUFDLE1BQU0sS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFFL0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZGQUE2RixDQUFDLElBQVU7UUFDdEcsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRWxELE9BQU87UUFDUCxNQUFNLFdBQVcsR0FBRyxpQkFBaUIsS0FBSyxFQUFFLENBQUM7UUFDN0MsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRXRDLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRTtZQUN2QixVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0REFBNEQsQ0FBQyxJQUFVO1FBQ3JFLFFBQVE7UUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN4QyxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxTQUFTLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFFbEQsT0FBTztRQUNQLE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUM3QyxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsMEJBQVcsQ0FBQyxTQUFTLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsU0FBUyxDQUFDLDBCQUFXLENBQUMsU0FBUyxDQUFDLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFdEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG9FQUFvRSxDQUFDLElBQVU7UUFDN0UsUUFBUTtRQUNSLEtBQUssTUFBTSxLQUFLLElBQUksMEJBQTBCLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDakQsT0FBTztZQUNQLE1BQU0sV0FBVyxHQUFHLGlCQUFpQixLQUFLLEVBQUUsQ0FBQztZQUM3QyxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7WUFFdEMsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsMEJBQVcsQ0FBQyxRQUFRLENBQUMsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1NBQzFEO1FBQ0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNFQUFzRSxDQUFDLElBQVU7UUFDL0UsUUFBUTtRQUNSLEtBQUssTUFBTSxVQUFVLElBQUksaUNBQWlDLENBQUMsRUFBRSxHQUFHLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRTtZQUMvRSxPQUFPO1lBQ1AsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLHNCQUFzQixVQUFVLEVBQUUsQ0FBQyxDQUFDO1lBRTdELE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxFQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUMsQ0FBQztZQUN0QyxJQUFJLENBQUMsS0FBSyxDQUFDLDBCQUFXLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxFQUFFLDRCQUE0QixDQUFDLENBQUM7U0FDMUU7UUFFRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkNBQTZDLENBQUMsSUFBVTtRQUN0RCxRQUFRO1FBQ1IsS0FBSyxNQUFNLEtBQUssSUFBSSxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNoRCxPQUFPO1lBQ1AsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLEdBQUcsS0FBSyxRQUFRLENBQUMsQ0FBQztZQUUzQyxPQUFPO1lBQ1AsSUFBSSxDQUFDLEtBQUssQ0FBQywwQkFBVyxDQUFDLFFBQVEsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1NBQ2xEO1FBRUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVDQUF1QyxDQUFDLElBQVU7UUFDaEQsUUFBUTtRQUNSLEtBQUssTUFBTSxNQUFNLElBQUksbUJBQW1CLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDbEQsS0FBSyxNQUFNLE1BQU0sSUFBSSxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsRUFBRTtnQkFDakQsT0FBTztnQkFDUCxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsR0FBRyxNQUFNLEdBQUcsTUFBTSxFQUFFLENBQUMsQ0FBQztnQkFFL0MsT0FBTztnQkFDUCxJQUFJLENBQUMsS0FBSyxDQUFDLDBCQUFXLENBQUMsUUFBUSxDQUFDLEVBQUUsYUFBYSxDQUFDLENBQUM7YUFDbEQ7U0FDRjtRQUVELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4REFBOEQsQ0FBQyxJQUFVO1FBQ3ZFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUc7WUFDUixDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxFQUFFLFFBQVEsS0FBSyxFQUFFO1NBQ3BDLENBQUM7UUFFRixPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxlQUFlLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFakYsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHO1lBQ1IsQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsRUFBRSxRQUFRLEtBQUssRUFBRTtTQUNwQyxDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsZUFBZSxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0RBQW9ELENBQUMsSUFBVTtRQUM3RCxRQUFRO1FBQ1IsTUFBTSxVQUFVLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDM0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsR0FBRyxVQUFVLFdBQVcsQ0FBQyxDQUFDO1FBRXhELE9BQU87UUFDUCxNQUFNLENBQUMsR0FBRztZQUNSLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLEVBQUUsY0FBYztTQUNuQyxDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsZUFBZSxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFDaEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFDQUFxQyxDQUFDLElBQVU7UUFDOUMsUUFBUTtRQUNSLE1BQU0sVUFBVSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzNDLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsVUFBVSxXQUFXLENBQUMsQ0FBQztRQUV4RCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUc7WUFDUixDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxFQUFFLGNBQWM7U0FDbkMsQ0FBQztRQUVGLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxnQkFBVSxDQUFDLElBQUksV0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLFVBQVUsQ0FBQyxFQUFFLHdCQUF3QixDQUFDLENBQUM7UUFDdEUsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLEtBQUssQ0FBQyxFQUFFLG1CQUFtQixDQUFDLENBQUM7UUFDNUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVEQUF1RCxDQUFDLElBQVU7UUFDaEUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFFMUMsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHO1lBQ1IsQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsRUFBRSxRQUFRLEtBQUssRUFBRTtTQUNwQyxDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLHNHQUFzRyxDQUFDLENBQUM7UUFDdEksSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGVBQWUsRUFBRTtRQUNmLHFEQUFxRCxDQUFDLElBQVU7WUFDOUQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFFMUMsT0FBTztZQUNQLE1BQU0sTUFBTSxHQUFHO2dCQUNiLEdBQUcsRUFBRSxLQUFLLENBQUMsTUFBTSxFQUFFO2FBQ3BCLENBQUM7WUFFRixPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQzlCLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUM7YUFDckIsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDRCQUE0QixDQUFDLElBQVU7WUFDckMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFFMUMsT0FBTztZQUNQLE1BQU0sT0FBTyxHQUFhLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUN6QyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBRXRCLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDbkIsQ0FBQyxFQUFFLG1DQUFtQyxDQUFDLENBQUM7WUFFeEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHVDQUF1QyxDQUFDLElBQVU7WUFDaEQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFFMUMsT0FBTztZQUNQLE1BQU0sT0FBTyxHQUFhLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUN6QyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksT0FBTyxDQUFDO1lBRXRCLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDbkIsQ0FBQyxFQUFFLHdCQUF3QixDQUFDLENBQUM7WUFFN0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG9DQUFvQyxDQUFDLElBQVU7WUFDN0MsUUFBUTtZQUNSLE1BQU0sT0FBTyxHQUFhLElBQUksV0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUM7WUFFL0QsT0FBTztZQUNQLE1BQU0sTUFBTSxHQUFHLFFBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBRXJDLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDOUIsWUFBWSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBQyxDQUFDO2FBQ25DLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxrQ0FBa0MsQ0FBQyxJQUFVO1lBQzNDLFFBQVE7WUFDUixNQUFNLE9BQU8sR0FBYSxJQUFJLFdBQUssQ0FBQyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBRS9ELE9BQU87WUFDUCxNQUFNLE1BQU0sR0FBRyxRQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQzlCLFVBQVUsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUMsQ0FBQzthQUNuQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsK0RBQStELENBQUMsSUFBVTtZQUN4RSxRQUFRO1lBQ1IsTUFBTSxPQUFPLEdBQWEsSUFBSSxXQUFLLENBQUMsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUUvRCxPQUFPO1lBQ1AsTUFBTSxNQUFNLEdBQUcsUUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7WUFFaEQsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUM5QixVQUFVLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFDLENBQUM7YUFDbkMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsaUJBQWlCLEVBQUU7UUFDakIsOEJBQThCLENBQUMsSUFBVTtZQUN2QyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSw2QkFBa0IsQ0FBQyw0QkFBaUIsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDMUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELCtEQUErRCxDQUFDLElBQVU7WUFDeEUsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEdBQUcsRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDNUIseUNBQXlDO2dCQUN6QyxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBRTVELE1BQU0sT0FBTyxHQUFHLDRCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNyQyxrQ0FBa0M7Z0JBQ2xDLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNsRixNQUFNLE9BQU8sR0FBRyw2QkFBa0IsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDakQsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLDhCQUE4QixDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQzNEO1lBRUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDhEQUE4RCxDQUFDLElBQVU7WUFDdkUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsNkJBQWtCLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM3QyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSw2QkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLDZCQUFrQixDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFFckQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDZDQUE2QyxDQUFDLElBQVU7WUFDdEQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxDQUFDLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7WUFFL0IsT0FBTztZQUNQLE1BQU0sT0FBTyxHQUFHLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUM3QixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxXQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLHVDQUF1QyxDQUFDLENBQUM7WUFFbEYsT0FBTztZQUNQLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDeEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUUsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUV6QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCwyQ0FBMkMsQ0FBQyxJQUFVO1FBQ3BELFNBQVMsR0FBRztZQUNWLFNBQVMsR0FBRztnQkFDVixNQUFNLFdBQVksU0FBUSxXQUFLO29CQUM3QixJQUFXLGFBQWE7d0JBQ3RCLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQztvQkFDcEIsQ0FBQztpQkFDRjtnQkFFRCxPQUFPLElBQUksV0FBVyxDQUFDLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3hDLENBQUM7WUFFRCxPQUFPLEdBQUcsRUFBRSxDQUFDO1FBQ2YsQ0FBQztRQUVELE1BQU0sS0FBSyxHQUFHLEdBQUcsRUFBRSxDQUFDO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxRCxJQUFJLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlEQUF5RCxDQUFDLElBQVU7UUFDbEUsU0FBUyxHQUFHO1lBQ1YsU0FBUyxHQUFHO2dCQUNWLFNBQVMsR0FBRztvQkFDVixNQUFNLGFBQWMsU0FBUSxXQUFLO3dCQUN4QixVQUFVLENBQUMsT0FBZTs0QkFDL0IsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO3dCQUMvQixDQUFDO3FCQUNGO29CQUNELE9BQU8sSUFBSSxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ25DLENBQUM7Z0JBRUQsT0FBTyxHQUFHLEVBQUUsQ0FBQztZQUNmLENBQUM7WUFDRCxPQUFPLEdBQUcsRUFBRSxDQUFDO1FBQ2YsQ0FBQztRQUNELE1BQU0sS0FBSyxHQUFHLEdBQUcsRUFBRSxDQUFDO1FBQ3BCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCB7IEFwcCBhcyBSb290LCBmaW5kVG9rZW5zLCBGbiwgU3RhY2ssIFRva2VuIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IGNyZWF0ZVRva2VuRG91YmxlLCBleHRyYWN0VG9rZW5Eb3VibGUgfSBmcm9tICcuLi9saWIvZW5jb2RpbmcnO1xuaW1wb3J0IHsgVG9rZW5NYXAgfSBmcm9tICcuLi9saWIvdG9rZW4tbWFwJztcbmltcG9ydCB7IGV2YWx1YXRlQ0ZOIH0gZnJvbSAnLi9ldmFsdWF0ZS1jZm4nO1xuXG5leHBvcnQgPSB7XG4gICdyZXNvbHZlIGEgcGxhaW4gb2xkIG9iamVjdCBzaG91bGQganVzdCByZXR1cm4gdGhlIG9iamVjdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IG9iaiA9IHsgUGxhaW5PbGRPYmplY3Q6IDEyMywgQXJyYXk6IFsgMSwgMiwgMyBdIH07XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZShvYmopLCBvYmopO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpZiBhIHZhbHVlIGlzIGFuIG9iamVjdCB3aXRoIGEgdG9rZW4gdmFsdWUsIGl0IHdpbGwgYmUgZXZhbHVhdGVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgb2JqID0ge1xuICAgICAgUmVndWxhclZhbHVlOiAnaGVsbG8nLFxuICAgICAgTGF6eVZhbHVlOiBuZXcgVG9rZW4oJ1dvcmxkJylcbiAgICB9O1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZShvYmopLCB7XG4gICAgICBSZWd1bGFyVmFsdWU6ICdoZWxsbycsXG4gICAgICBMYXp5VmFsdWU6ICdXb3JsZCdcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0b2tlbnMgYXJlIGV2YWx1YXRlZCBhbnl3aGVyZSBpbiB0aGUgb2JqZWN0IHRyZWUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBvYmogPSBuZXcgUHJvbWlzZTEoKTtcbiAgICBjb25zdCBhY3R1YWwgPSByZXNvbHZlKHsgT2JqOiBvYmogfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChhY3R1YWwsIHtcbiAgICAgIE9iajogW1xuICAgICAgICB7XG4gICAgICAgIERhdGE6IHtcbiAgICAgICAgICBzdHJpbmdQcm9wOiBcImhlbGxvXCIsXG4gICAgICAgICAgbnVtYmVyUHJvcDogMTIzNFxuICAgICAgICB9LFxuICAgICAgICBSZWN1cnNlOiA0MlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgIERhdGE6IHtcbiAgICAgICAgICBzdHJpbmdQcm9wOiBcImhlbGxvXCIsXG4gICAgICAgICAgbnVtYmVyUHJvcDogMTIzNFxuICAgICAgICB9LFxuICAgICAgICBSZWN1cnNlOiA0MlxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndG9rZW5zIGFyZSBldmFsdWF0ZWQgcmVjdXJzaXZlbHknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBvYmogPSBuZXcgUHJvbWlzZTEoKTtcbiAgICBjb25zdCBhY3R1YWwgPSByZXNvbHZlKG5ldyBUb2tlbigoKSA9PiAoeyBPYmo6IG9iaiB9KSkpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoYWN0dWFsLCB7XG4gICAgICBPYmo6IFtcbiAgICAgICAge1xuICAgICAgICBEYXRhOiB7XG4gICAgICAgICAgc3RyaW5nUHJvcDogXCJoZWxsb1wiLFxuICAgICAgICAgIG51bWJlclByb3A6IDEyMzRcbiAgICAgICAgfSxcbiAgICAgICAgUmVjdXJzZTogNDJcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICBEYXRhOiB7XG4gICAgICAgICAgc3RyaW5nUHJvcDogXCJoZWxsb1wiLFxuICAgICAgICAgIG51bWJlclByb3A6IDEyMzRcbiAgICAgICAgfSxcbiAgICAgICAgUmVjdXJzZTogNDJcbiAgICAgICAgfVxuICAgICAgXVxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2VtcHR5IGFycmF5cyBvciBvYmplY3RzIGFyZSBrZXB0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZSh7IH0pLCB7IH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlc29sdmUoWyBdKSwgWyBdKTtcblxuICAgIGNvbnN0IG9iaiA9IHtcbiAgICAgIFByb3AxOiAxMjM0LFxuICAgICAgUHJvcDI6IHsgfSxcbiAgICAgIFByb3AzOiBbIF0sXG4gICAgICBQcm9wNDogJ2hlbGxvJyxcbiAgICAgIFByb3A1OiB7XG4gICAgICAgIFByb3BBOiB7IH0sXG4gICAgICAgIFByb3BCOiB7XG4gICAgICAgICAgUHJvcEM6IFsgdW5kZWZpbmVkLCB1bmRlZmluZWQgXSxcbiAgICAgICAgICBQcm9wRDogJ1lvb2hvbydcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH07XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXNvbHZlKG9iaiksIHtcbiAgICAgIFByb3AxOiAxMjM0LFxuICAgICAgUHJvcDI6IHsgfSxcbiAgICAgIFByb3AzOiBbIF0sXG4gICAgICBQcm9wNDogJ2hlbGxvJyxcbiAgICAgIFByb3A1OiB7XG4gICAgICAgIFByb3BBOiB7IH0sXG4gICAgICAgIFByb3BCOiB7XG4gICAgICAgICAgUHJvcEM6IFsgXSxcbiAgICAgICAgICBQcm9wRDogJ1lvb2hvbydcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2lmIGFuIG9iamVjdCBoYXMgYSBcInJlc29sdmVcIiBwcm9wZXJ0eSB0aGF0IGlzIG5vdCBhIGZ1bmN0aW9uLCBpdCBpcyBub3QgY29uc2lkZXJlZCBhIHRva2VuJyh0ZXN0OiBUZXN0KSB7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZSh7IGFfdG9rZW46IHsgcmVzb2x2ZTogKCkgPT4gNzg3ODcgfX0pLCB7IGFfdG9rZW46IDc4Nzg3IH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlc29sdmUoeyBub3RfYV90b2tlbjogeyByZXNvbHZlOiAxMiB9IH0pLCAgIHsgbm90X2FfdG9rZW46IHsgcmVzb2x2ZTogMTIgfSB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICdpZiBhIHJlc29sdmFibGUgb2JqZWN0IGluaGVyaXRzIGZyb20gYSBjbGFzcyB0aGF0IGlzIGFsc28gcmVzb2x2YWJsZSwgdGhlIFwiY29uc3RydWN0b3JcIiBmdW5jdGlvbiB3aWxsIG5vdCBnZXQgaW4gdGhlIHdheSAodXNlcyBPYmplY3Qua2V5cyBpbnN0ZWFkIG9mIFwiZm9yIGluXCIpJyh0ZXN0OiBUZXN0KSB7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZSh7IHByb3A6IG5ldyBEYXRhVHlwZSgpIH0pLCB7IHByb3A6IHsgZm9vOiAxMiwgZ29vOiAnaGVsbG8nIH0gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2lzVG9rZW4ob2JqKSBjYW4gYmUgdXNlZCB0byBkZXRlcm1pbmUgaWYgYW4gb2JqZWN0IGlzIGEgdG9rZW4nKHRlc3Q6IFRlc3QpIHtcbiAgICB0ZXN0Lm9rKFRva2VuLmlzVG9rZW4oeyByZXNvbHZlOiAoKSA9PiAxMjMgfSkpO1xuICAgIHRlc3Qub2soVG9rZW4uaXNUb2tlbih7IGE6IDEsIGI6IDIsIHJlc29sdmU6ICgpID0+ICdoZWxsbycgfSkpO1xuICAgIHRlc3Qub2soIVRva2VuLmlzVG9rZW4oeyBhOiAxLCBiOiAyLCByZXNvbHZlOiAzIH0pKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnVG9rZW4gY2FuIGJlIHVzZWQgdG8gY3JlYXRlIHRva2VucyB0aGF0IGNvbnRhaW4gYSBjb25zdGFudCB2YWx1ZScodGVzdDogVGVzdCkge1xuICAgIHRlc3QuZXF1YWwocmVzb2x2ZShuZXcgVG9rZW4oMTIpKSwgMTIpO1xuICAgIHRlc3QuZXF1YWwocmVzb2x2ZShuZXcgVG9rZW4oJ2hlbGxvJykpLCAnaGVsbG8nKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXNvbHZlKG5ldyBUb2tlbihbICdoaScsICd0aGVyZScgXSkpLCBbICdoaScsICd0aGVyZScgXSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Jlc29sdmluZyBsZWF2ZXMgYSBEYXRlIG9iamVjdCBpbiB3b3JraW5nIG9yZGVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgZGF0ZSA9IG5ldyBEYXRlKCcyMDAwLTAxLTAxJyk7XG4gICAgY29uc3QgcmVzb2x2ZWQgPSByZXNvbHZlKGRhdGUpO1xuXG4gICAgdGVzdC5lcXVhbChkYXRlLnRvU3RyaW5nKCksIHJlc29sdmVkLnRvU3RyaW5nKCkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0b2tlbnMgY2FuIGJlIHN0cmluZ2lmaWVkIGFuZCBldmFsdWF0ZWQgdG8gY29uY2VwdHVhbCB2YWx1ZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgdG9rZW4gPSBuZXcgVG9rZW4oKCkgPT4gJ3dvb2Ygd29vZicpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0cmluZ2lmaWVkID0gYFRoZSBkb2cgc2F5czogJHt0b2tlbn1gO1xuICAgIGNvbnN0IHJlc29sdmVkID0gcmVzb2x2ZShzdHJpbmdpZmllZCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoZXZhbHVhdGVDRk4ocmVzb2x2ZWQpLCAnVGhlIGRvZyBzYXlzOiB3b29mIHdvb2YnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndG9rZW5zIHN0cmluZ2lmaWNhdGlvbiBjYW4gYmUgcmV2ZXJzZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHRva2VuID0gbmV3IFRva2VuKCgpID0+ICd3b29mIHdvb2YnKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmVxdWFsKHRva2VuLCBUb2tlbk1hcC5pbnN0YW5jZSgpLmxvb2t1cFN0cmluZyhgJHt0b2tlbn1gKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvbmNhdGVuYXRlZCB0b2tlbnMgYXJlIHVuZGVmaW5lZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgdG9rZW4gPSBuZXcgVG9rZW4oKCkgPT4gJ3dvb2Ygd29vZicpO1xuXG4gICAgLy8gV0hFTlxuICAgIHRlc3QuZXF1YWwodW5kZWZpbmVkLCBUb2tlbk1hcC5pbnN0YW5jZSgpLmxvb2t1cFN0cmluZyhgJHt0b2tlbn1ibGFgKSk7XG4gICAgdGVzdC5lcXVhbCh1bmRlZmluZWQsIFRva2VuTWFwLmluc3RhbmNlKCkubG9va3VwU3RyaW5nKGBibGEke3Rva2VufWApKTtcbiAgICB0ZXN0LmVxdWFsKHVuZGVmaW5lZCwgVG9rZW5NYXAuaW5zdGFuY2UoKS5sb29rdXBTdHJpbmcoYGJsYWApKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdUb2tlbnMgc3RyaW5naWZpY2F0aW9uIGFuZCByZXZlcnNpbmcgb2YgQ2xvdWRGb3JtYXRpb24gVG9rZW5zIGlzIGltcGxlbWVudGVkIHVzaW5nIEZuOjpKb2luJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB0b2tlbiA9IG5ldyBUb2tlbigoKSA9PiAoeyB3b29mOiAnd29vZicgfSkpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0cmluZ2lmaWVkID0gYFRoZSBkb2cgc2F5czogJHt0b2tlbn1gO1xuICAgIGNvbnN0IHJlc29sdmVkID0gcmVzb2x2ZShzdHJpbmdpZmllZCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZWQsIHtcbiAgICAgICdGbjo6Sm9pbic6IFsnJywgWydUaGUgZG9nIHNheXM6ICcsIHsgd29vZjogJ3dvb2YnIH1dXVxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdEb3VibHkgbmVzdGVkIHN0cmluZ3MgZXZhbHVhdGUgY29ycmVjdGx5IGluIHNjYWxhciBjb250ZXh0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB0b2tlbjEgPSBuZXcgVG9rZW4oKCkgPT4gXCJ3b3JsZFwiKTtcbiAgICBjb25zdCB0b2tlbjIgPSBuZXcgVG9rZW4oKCkgPT4gYGhlbGxvICR7dG9rZW4xfWApO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHJlc29sdmVkMSA9IHJlc29sdmUodG9rZW4yLnRvU3RyaW5nKCkpO1xuICAgIGNvbnN0IHJlc29sdmVkMiA9IHJlc29sdmUodG9rZW4yKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChldmFsdWF0ZUNGTihyZXNvbHZlZDEpLCBcImhlbGxvIHdvcmxkXCIpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGV2YWx1YXRlQ0ZOKHJlc29sdmVkMiksIFwiaGVsbG8gd29ybGRcIik7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW50ZWdlciBUb2tlbnMgY2FuIGJlIHN0cmluZ2lmaWVkIGFuZCBldmFsdWF0ZSB0byBjb25jZXB0dWFsIHZhbHVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBmb3IgKGNvbnN0IHRva2VuIG9mIGxpdGVyYWxUb2tlbnNUaGF0UmVzb2x2ZVRvKDEpKSB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBzdHJpbmdpZmllZCA9IGB0aGUgbnVtYmVyIGlzICR7dG9rZW59YDtcbiAgICAgIGNvbnN0IHJlc29sdmVkID0gcmVzb2x2ZShzdHJpbmdpZmllZCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGV2YWx1YXRlQ0ZOKHJlc29sdmVkKSwgJ3RoZSBudW1iZXIgaXMgMScpO1xuICAgIH1cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW50cmluc2ljIFRva2VucyBjYW4gYmUgc3RyaW5naWZpZWQgYW5kIGV2YWx1YXRlIHRvIGNvbmNlcHR1YWwgdmFsdWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGZvciAoY29uc3QgYnVja2V0TmFtZSBvZiBjbG91ZEZvcm1hdGlvblRva2Vuc1RoYXRSZXNvbHZlVG8oeyBSZWY6ICdNeUJ1Y2tldCcgfSkpIHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHJlc29sdmVkID0gcmVzb2x2ZShgbXkgYnVja2V0IGlzIG5hbWVkICR7YnVja2V0TmFtZX1gKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgY29uc3QgY29udGV4dCA9IHtNeUJ1Y2tldDogJ1RoZU5hbWUnfTtcbiAgICAgIHRlc3QuZXF1YWwoZXZhbHVhdGVDRk4ocmVzb2x2ZWQsIGNvbnRleHQpLCAnbXkgYnVja2V0IGlzIG5hbWVkIFRoZU5hbWUnKTtcbiAgICB9XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndG9rZW5zIHJlc29sdmUgcHJvcGVybHkgaW4gaW5pdGlhbCBwb3NpdGlvbicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgZm9yIChjb25zdCB0b2tlbiBvZiB0b2tlbnNUaGF0UmVzb2x2ZVRvKCdIZWxsbycpKSB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCByZXNvbHZlZCA9IHJlc29sdmUoYCR7dG9rZW59IHdvcmxkYCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZXF1YWwoZXZhbHVhdGVDRk4ocmVzb2x2ZWQpLCAnSGVsbG8gd29ybGQnKTtcbiAgICB9XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc2lkZS1ieS1zaWRlIFRva2VucyByZXNvbHZlIGNvcnJlY3RseScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgZm9yIChjb25zdCB0b2tlbjEgb2YgdG9rZW5zVGhhdFJlc29sdmVUbygnSGVsbG8gJykpIHtcbiAgICAgIGZvciAoY29uc3QgdG9rZW4yIG9mIHRva2Vuc1RoYXRSZXNvbHZlVG8oJ3dvcmxkJykpIHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCByZXNvbHZlZCA9IHJlc29sdmUoYCR7dG9rZW4xfSR7dG9rZW4yfWApO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgdGVzdC5lcXVhbChldmFsdWF0ZUNGTihyZXNvbHZlZCksICdIZWxsbyB3b3JsZCcpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0b2tlbnMgY2FuIGJlIHVzZWQgaW4gaGFzaCBrZXlzIGJ1dCBtdXN0IHJlc29sdmUgdG8gYSBzdHJpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHRva2VuID0gbmV3IFRva2VuKCgpID0+ICdJIGFtIGEgc3RyaW5nJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcyA9IHtcbiAgICAgIFt0b2tlbi50b1N0cmluZygpXTogYGJvb20gJHt0b2tlbn1gXG4gICAgfTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXNvbHZlKHMpLCB7ICdJIGFtIGEgc3RyaW5nJzogJ2Jvb20gSSBhbSBhIHN0cmluZycgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rva2VucyBjYW4gYmUgbmVzdGVkIGluIGhhc2gga2V5cycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgdG9rZW4gPSBuZXcgVG9rZW4oKCkgPT4gbmV3IFRva2VuKCgpID0+IG5ldyBUb2tlbigoKSA9PiAnSSBhbSBhIHN0cmluZycpKSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcyA9IHtcbiAgICAgIFt0b2tlbi50b1N0cmluZygpXTogYGJvb20gJHt0b2tlbn1gXG4gICAgfTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXNvbHZlKHMpLCB7ICdJIGFtIGEgc3RyaW5nJzogJ2Jvb20gSSBhbSBhIHN0cmluZycgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rva2VucyBjYW4gYmUgbmVzdGVkIGFuZCBjb25jYXRlbmF0ZWQgaW4gaGFzaCBrZXlzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBpbm5lclRva2VuID0gbmV3IFRva2VuKCgpID0+ICd0b290Jyk7XG4gICAgY29uc3QgdG9rZW4gPSBuZXcgVG9rZW4oKCkgPT4gYCR7aW5uZXJUb2tlbn0gdGhlIHdvb3RgKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzID0ge1xuICAgICAgW3Rva2VuLnRvU3RyaW5nKCldOiBgYm9vbSBjaGljYWdvYFxuICAgIH07XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZShzKSwgeyAndG9vdCB0aGUgd29vdCc6ICdib29tIGNoaWNhZ28nIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gZmluZCBuZXN0ZWQgdG9rZW5zIGluIGhhc2gga2V5cycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgaW5uZXJUb2tlbiA9IG5ldyBUb2tlbigoKSA9PiAndG9vdCcpO1xuICAgIGNvbnN0IHRva2VuID0gbmV3IFRva2VuKCgpID0+IGAke2lubmVyVG9rZW59IHRoZSB3b290YCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcyA9IHtcbiAgICAgIFt0b2tlbi50b1N0cmluZygpXTogYGJvb20gY2hpY2Fnb2BcbiAgICB9O1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHRva2VucyA9IGZpbmRUb2tlbnMobmV3IFN0YWNrKCksICgpID0+IHMpO1xuICAgIHRlc3Qub2sodG9rZW5zLnNvbWUodCA9PiB0ID09PSBpbm5lclRva2VuKSwgJ0Nhbm5vdCBmaW5kIGlubmVyVG9rZW4nKTtcbiAgICB0ZXN0Lm9rKHRva2Vucy5zb21lKHQgPT4gdCA9PT0gdG9rZW4pLCAnQ2Fubm90IGZpbmQgdG9rZW4nKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZmFpbHMgaWYgdG9rZW4gaW4gYSBoYXNoIGtleSByZXNvbHZlcyB0byBhIG5vbi1zdHJpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHRva2VuID0gbmV3IFRva2VuKHsgUmVmOiAnT3RoZXInIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHMgPSB7XG4gICAgICBbdG9rZW4udG9TdHJpbmcoKV06IGBib29tICR7dG9rZW59YFxuICAgIH07XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gcmVzb2x2ZShzKSwgJ1RoZSBrZXkgXCIke1Rva2VuW1RPS0VOLjE5XX1cIiBoYXMgYmVlbiByZXNvbHZlZCB0byB7XCJSZWZcIjpcIk90aGVyXCJ9IGJ1dCBtdXN0IGJlIHJlc29sdmFibGUgdG8gYSBzdHJpbmcnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbGlzdCBlbmNvZGluZyc6IHtcbiAgICAnY2FuIGVuY29kZSBUb2tlbiB0byBzdHJpbmcgYW5kIHJlc29sdmUgdGhlIGVuY29kaW5nJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgdG9rZW4gPSBuZXcgVG9rZW4oeyBSZWY6ICdPdGhlcicgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHN0cnVjdCA9IHtcbiAgICAgICAgWFlaOiB0b2tlbi50b0xpc3QoKVxuICAgICAgfTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZShzdHJ1Y3QpLCB7XG4gICAgICAgIFhZWjogeyBSZWY6ICdPdGhlcid9XG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW5ub3QgYWRkIHRvIGVuY29kZWQgbGlzdCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHRva2VuID0gbmV3IFRva2VuKHsgUmVmOiAnT3RoZXInIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBlbmNvZGVkOiBzdHJpbmdbXSA9IHRva2VuLnRvTGlzdCgpO1xuICAgICAgZW5jb2RlZC5wdXNoKCdoZWxsbycpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIHJlc29sdmUoZW5jb2RlZCk7XG4gICAgICB9LCAvQ2Fubm90IGFkZCBlbGVtZW50cyB0byBsaXN0IHRva2VuLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2Fubm90IGFkZCB0byBzdHJpbmdzIGluIGVuY29kZWQgbGlzdCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHRva2VuID0gbmV3IFRva2VuKHsgUmVmOiAnT3RoZXInIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBlbmNvZGVkOiBzdHJpbmdbXSA9IHRva2VuLnRvTGlzdCgpO1xuICAgICAgZW5jb2RlZFswXSArPSAnaGVsbG8nO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIHJlc29sdmUoZW5jb2RlZCk7XG4gICAgICB9LCAvY29uY2F0ZW5hdGUgc3RyaW5ncyBpbi8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBwYXNzIGVuY29kZWQgbGlzdHMgdG8gRm5TZWxlY3QnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBlbmNvZGVkOiBzdHJpbmdbXSA9IG5ldyBUb2tlbih7IFJlZjogJ090aGVyJyB9KS50b0xpc3QoKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3Qgc3RydWN0ID0gRm4uc2VsZWN0KDEsIGVuY29kZWQpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LmRlZXBFcXVhbChyZXNvbHZlKHN0cnVjdCksIHtcbiAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbMSwgeyBSZWY6ICdPdGhlcid9XVxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHBhc3MgZW5jb2RlZCBsaXN0cyB0byBGbkpvaW4nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBlbmNvZGVkOiBzdHJpbmdbXSA9IG5ldyBUb2tlbih7IFJlZjogJ090aGVyJyB9KS50b0xpc3QoKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3Qgc3RydWN0ID0gRm4uam9pbignLycsIGVuY29kZWQpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LmRlZXBFcXVhbChyZXNvbHZlKHN0cnVjdCksIHtcbiAgICAgICAgJ0ZuOjpKb2luJzogWycvJywgeyBSZWY6ICdPdGhlcid9XVxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHBhc3MgZW5jb2RlZCBsaXN0cyB0byBGbkpvaW4sIGV2ZW4gaWYgam9pbiBpcyBzdHJpbmdpZmllZCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGVuY29kZWQ6IHN0cmluZ1tdID0gbmV3IFRva2VuKHsgUmVmOiAnT3RoZXInIH0pLnRvTGlzdCgpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBzdHJ1Y3QgPSBGbi5qb2luKCcvJywgZW5jb2RlZCkudG9TdHJpbmcoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZShzdHJ1Y3QpLCB7XG4gICAgICAgICdGbjo6Sm9pbic6IFsnLycsIHsgUmVmOiAnT3RoZXInfV1cbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdudW1iZXIgZW5jb2RpbmcnOiB7XG4gICAgJ2Jhc2ljIGludGVnZXIgZW5jb2Rpbmcgd29ya3MnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3QuZXF1YWwoMTYsIGV4dHJhY3RUb2tlbkRvdWJsZShjcmVhdGVUb2tlbkRvdWJsZSgxNikpKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYXJiaXRyYXJ5IGludGVnZXJzIGNhbiBiZSBlbmNvZGVkLCBzdHJpbmdpZmllZCwgYW5kIHJlY292ZXJlZCcodGVzdDogVGVzdCkge1xuICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCAxMDA7IGkrKykge1xuICAgICAgICAvLyBXZSBjYW4gZW5jb2RlIGFsbCBudW1iZXJzIHVwIHRvIDJeNDgtMVxuICAgICAgICBjb25zdCB4ID0gTWF0aC5mbG9vcihNYXRoLnJhbmRvbSgpICogKE1hdGgucG93KDIsIDQ4KSAtIDEpKTtcblxuICAgICAgICBjb25zdCBlbmNvZGVkID0gY3JlYXRlVG9rZW5Eb3VibGUoeCk7XG4gICAgICAgIC8vIFJvdW5kdHJpcCB0aHJvdWdoIEpTT05pZmljYXRpb25cbiAgICAgICAgY29uc3Qgcm91bmR0cmlwcGVkID0gSlNPTi5wYXJzZShKU09OLnN0cmluZ2lmeSh7IHRoZU51bWJlcjogZW5jb2RlZCB9KSkudGhlTnVtYmVyO1xuICAgICAgICBjb25zdCBkZWNvZGVkID0gZXh0cmFjdFRva2VuRG91YmxlKHJvdW5kdHJpcHBlZCk7XG4gICAgICAgIHRlc3QuZXF1YWwoZGVjb2RlZCwgeCwgYEZhaWwgcm91bmR0cmlwIGVuY29kaW5nIG9mICR7eH1gKTtcbiAgICAgIH1cblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdhcmJpdHJhcnkgbnVtYmVycyBhcmUgY29ycmVjdGx5IGRldGVjdGVkIGFzIG5vdCBiZWluZyB0b2tlbnMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3QuZXF1YWwodW5kZWZpbmVkLCBleHRyYWN0VG9rZW5Eb3VibGUoMCkpO1xuICAgICAgdGVzdC5lcXVhbCh1bmRlZmluZWQsIGV4dHJhY3RUb2tlbkRvdWJsZSgxMjQzKSk7XG4gICAgICB0ZXN0LmVxdWFsKHVuZGVmaW5lZCwgZXh0cmFjdFRva2VuRG91YmxlKDQ4MzVlKzUzMikpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBudW1iZXItZW5jb2RlIGFuZCByZXNvbHZlIFRva2VuIG9iamVjdHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgeCA9IG5ldyBUb2tlbigoKSA9PiAxMjMpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBjb25zdCBlbmNvZGVkID0geC50b051bWJlcigpO1xuICAgICAgdGVzdC5lcXVhbCh0cnVlLCBUb2tlbi5pc1Rva2VuKGVuY29kZWQpLCAnZW5jb2RlZCBudW1iZXIgZG9lcyBub3QgdGVzdCBhcyB0b2tlbicpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBjb25zdCByZXNvbHZlZCA9IHN0YWNrLm5vZGUucmVzb2x2ZSh7IHZhbHVlOiBlbmNvZGVkIH0pO1xuICAgICAgdGVzdC5kZWVwRXF1YWwocmVzb2x2ZWQsIHsgdmFsdWU6IDEyMyB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICAnc3RhY2sgdHJhY2UgaXMgY2FwdHVyZWQgYXQgdG9rZW4gY3JlYXRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICBmdW5jdGlvbiBmbjEoKSB7XG4gICAgICBmdW5jdGlvbiBmbjIoKSB7XG4gICAgICAgIGNsYXNzIEV4cG9zZVRyYWNlIGV4dGVuZHMgVG9rZW4ge1xuICAgICAgICAgIHB1YmxpYyBnZXQgY3JlYXRpb25UcmFjZSgpIHtcbiAgICAgICAgICAgIHJldHVybiB0aGlzLnRyYWNlO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBuZXcgRXhwb3NlVHJhY2UoKCkgPT4gJ2hlbGxvJyk7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiBmbjIoKTtcbiAgICB9XG5cbiAgICBjb25zdCB0b2tlbiA9IGZuMSgpO1xuICAgIHRlc3Qub2sodG9rZW4uY3JlYXRpb25UcmFjZS5maW5kKHggPT4geC5pbmNsdWRlcygnZm4xJykpKTtcbiAgICB0ZXN0Lm9rKHRva2VuLmNyZWF0aW9uVHJhY2UuZmluZCh4ID0+IHguaW5jbHVkZXMoJ2ZuMicpKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ25ld0Vycm9yIHJldHVybnMgYW4gZXJyb3Igd2l0aCB0aGUgY3JlYXRpb24gc3RhY2sgdHJhY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICBmdW5jdGlvbiBmbjEoKSB7XG4gICAgICBmdW5jdGlvbiBmbjIoKSB7XG4gICAgICAgIGZ1bmN0aW9uIGZuMygpIHtcbiAgICAgICAgICBjbGFzcyBUaHJvd2luZ1Rva2VuIGV4dGVuZHMgVG9rZW4ge1xuICAgICAgICAgICAgcHVibGljIHRocm93RXJyb3IobWVzc2FnZTogc3RyaW5nKSB7XG4gICAgICAgICAgICAgIHRocm93IHRoaXMubmV3RXJyb3IobWVzc2FnZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICAgIHJldHVybiBuZXcgVGhyb3dpbmdUb2tlbignYm9vbScpO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIGZuMygpO1xuICAgICAgfVxuICAgICAgcmV0dXJuIGZuMigpO1xuICAgIH1cbiAgICBjb25zdCB0b2tlbiA9IGZuMSgpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IHRva2VuLnRocm93RXJyb3IoJ21lc3NhZ2UhJyksIC9Ub2tlbiBjcmVhdGVkOi8pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9XG59O1xuXG5jbGFzcyBQcm9taXNlMiBleHRlbmRzIFRva2VuIHtcbiAgcHVibGljIHJlc29sdmUoKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIERhdGE6IHtcbiAgICAgICAgc3RyaW5nUHJvcDogJ2hlbGxvJyxcbiAgICAgICAgbnVtYmVyUHJvcDogMTIzNCxcbiAgICAgIH0sXG4gICAgICBSZWN1cnNlOiBuZXcgVG9rZW4oKCkgPT4gNDIpXG4gICAgfTtcbiAgfVxufVxuXG5jbGFzcyBQcm9taXNlMSBleHRlbmRzIFRva2VuIHtcbiAgcHVibGljIHAyID0gWyBuZXcgUHJvbWlzZTIoKSwgbmV3IFByb21pc2UyKCkgXTtcblxuICBwdWJsaWMgcmVzb2x2ZSgpIHtcbiAgICByZXR1cm4gdGhpcy5wMjtcbiAgfVxufVxuXG5jbGFzcyBCYXNlRGF0YVR5cGUge1xuICBjb25zdHJ1Y3RvcihyZWFkb25seSBmb286IG51bWJlcikge1xuICB9XG59XG5cbmNsYXNzIERhdGFUeXBlIGV4dGVuZHMgQmFzZURhdGFUeXBlIHtcbiAgcHVibGljIGdvbyA9ICdoZWxsbyc7XG5cbiAgY29uc3RydWN0b3IoKSB7XG4gICAgc3VwZXIoMTIpO1xuICB9XG59XG5cbi8qKlxuICogUmV0dXJuIHZhcmlvdXMgZmxhdm9ycyBvZiBUb2tlbnMgdGhhdCByZXNvbHZlIHRvIHRoZSBnaXZlbiB2YWx1ZVxuICovXG5mdW5jdGlvbiBsaXRlcmFsVG9rZW5zVGhhdFJlc29sdmVUbyh2YWx1ZTogYW55KTogVG9rZW5bXSB7XG4gIHJldHVybiBbXG4gICAgbmV3IFRva2VuKHZhbHVlKSxcbiAgICBuZXcgVG9rZW4oKCkgPT4gdmFsdWUpXG4gIF07XG59XG5cbi8qKlxuICogUmV0dXJuIHZhcmlvdXMgZmxhdm9ycyBvZiBUb2tlbnMgdGhhdCByZXNvbHZlIHRvIHRoZSBnaXZlbiB2YWx1ZVxuICovXG5mdW5jdGlvbiBjbG91ZEZvcm1hdGlvblRva2Vuc1RoYXRSZXNvbHZlVG8odmFsdWU6IGFueSk6IFRva2VuW10ge1xuICByZXR1cm4gW1xuICAgIG5ldyBUb2tlbih2YWx1ZSksXG4gICAgbmV3IFRva2VuKCgpID0+IHZhbHVlKVxuICBdO1xufVxuXG4vKipcbiAqIFJldHVybiBUb2tlbnMgaW4gYm90aCBmbGF2b3JzIHRoYXQgcmVzb2x2ZSB0byB0aGUgZ2l2ZW4gc3RyaW5nXG4gKi9cbmZ1bmN0aW9uIHRva2Vuc1RoYXRSZXNvbHZlVG8odmFsdWU6IHN0cmluZyk6IFRva2VuW10ge1xuICByZXR1cm4gbGl0ZXJhbFRva2Vuc1RoYXRSZXNvbHZlVG8odmFsdWUpLmNvbmNhdChjbG91ZEZvcm1hdGlvblRva2Vuc1RoYXRSZXNvbHZlVG8odmFsdWUpKTtcbn1cblxuLyoqXG4gKiBXcmFwcGVyIGZvciByZXNvbHZlIHRoYXQgY3JlYXRlcyBhbiB0aHJvd2F3YXkgQ29uc3RydWN0IHRvIGNhbGwgaXQgb25cbiAqXG4gKiBTbyBJIGRvbid0IGhhdmUgdG8gY2hhbmdlIGFsbCBjYWxsIHNpdGVzIGluIHRoaXMgZmlsZS5cbiAqL1xuZnVuY3Rpb24gcmVzb2x2ZSh4OiBhbnkpIHtcbiAgcmV0dXJuIG5ldyBSb290KCkubm9kZS5yZXNvbHZlKHgpO1xufVxuIl19