"use strict";
const cfn_resource_1 = require("../lib/cfn-resource");
const tag_manager_1 = require("../lib/tag-manager");
module.exports = {
    '#setTag() supports setting a tag regardless of Type'(test) {
        const notTaggable = new tag_manager_1.TagManager(cfn_resource_1.TagType.NotTaggable, 'AWS::Resource::Type');
        notTaggable.setTag('key', 'value');
        test.deepEqual(notTaggable.renderTags(), undefined);
        test.done();
    },
    'when a tag does not exist': {
        '#removeTag() does not throw an error'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
            test.doesNotThrow(() => (mgr.removeTag('dne', 0)));
            test.done();
        },
        '#setTag() creates the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'notanymore' }]);
            test.done();
        }
    },
    'when a tag does exist': {
        '#removeTag() deletes the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore', 0);
            mgr.removeTag('dne', 0);
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#setTag() overwrites the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            mgr.setTag('dne', 'iwin');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'iwin' }]);
            test.done();
        }
    },
    'when there are no tags': {
        '#renderTags() returns undefined'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
    },
    '#renderTags() handles standard, map, and ASG tag formats'(test) {
        const tagged = [];
        const standard = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
        const asg = new tag_manager_1.TagManager(cfn_resource_1.TagType.AutoScalingGroup, 'AWS::Resource::Type');
        const mapper = new tag_manager_1.TagManager(cfn_resource_1.TagType.Map, 'AWS::Resource::Type');
        tagged.push(standard);
        tagged.push(asg);
        tagged.push(mapper);
        for (const res of tagged) {
            res.setTag('foo', 'bar');
            res.setTag('asg', 'only', 0, false);
        }
        test.deepEqual(standard.renderTags(), [
            { key: 'foo', value: 'bar' },
            { key: 'asg', value: 'only' },
        ]);
        test.deepEqual(asg.renderTags(), [
            { key: 'foo', value: 'bar', propagateAtLaunch: true },
            { key: 'asg', value: 'only', propagateAtLaunch: false },
        ]);
        test.deepEqual(mapper.renderTags(), {
            foo: 'bar',
            asg: 'only',
        });
        test.done();
    },
    'tags with higher or equal priority always take precedence'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 2);
        test.deepEqual(mgr.renderTags(), undefined);
        test.done();
    },
    'excludeResourceTypes only tags resources that do not match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Fake::Resource');
        test.equal(false, mgr.applyTagAspectHere([], ['AWS::Fake::Resource']));
        test.equal(true, mgr.applyTagAspectHere([], ['AWS::Wrong::Resource']));
        test.done();
    },
    'includeResourceTypes only tags resources that match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.Standard, 'AWS::Fake::Resource');
        test.equal(true, mgr.applyTagAspectHere(['AWS::Fake::Resource'], []));
        test.equal(false, mgr.applyTagAspectHere(['AWS::Wrong::Resource'], []));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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