"use strict";
const lib_1 = require("../lib");
class TaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.Standard, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.properties;
    }
}
class AsgTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.AutoScalingGroup, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.properties;
    }
}
class MapTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.Map, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.properties;
    }
}
module.exports = {
    'Tag visit all children of the applied node'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.apply(new lib_1.Tag('foo', 'bar'));
        test.deepEqual(res.node.aspects.length, 1);
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(map.tags.renderTags(), { foo: 'bar' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'foo', value: 'bar', propagateAtLaunch: true }]);
        test.done();
    },
    'The last aspect applied takes precedence'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.apply(new lib_1.Tag('foo', 'bar'));
        res.node.apply(new lib_1.Tag('foo', 'foobar'));
        res.node.apply(new lib_1.Tag('foo', 'baz'));
        res2.node.apply(new lib_1.Tag('foo', 'good'));
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'baz' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'good' }]);
        test.done();
    },
    'RemoveTag will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        root.node.apply(new lib_1.Tag('root', 'was here'));
        res.node.apply(new lib_1.Tag('first', 'there is only 1'));
        res.node.apply(new lib_1.RemoveTag('root'));
        res.node.apply(new lib_1.RemoveTag('doesnotexist'));
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'the #visit function is idempotent'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.apply(new lib_1.Tag('foo', 'bar'));
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.done();
    },
    'removeTag Aspects by default will override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.apply(new lib_1.RemoveTag('key'));
        res2.node.apply(new lib_1.Tag('key', 'value'));
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), undefined);
        test.done();
    },
    'removeTag Aspects with priority 0 will not override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.apply(new lib_1.RemoveTag('key', { priority: 0 }));
        res2.node.apply(new lib_1.Tag('key', 'value'));
        root.node.prepareTree();
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'key', value: 'value' }]);
        test.done();
    },
    'Aspects are merged with tags created by L1 Constructor'(test) {
        const root = new lib_1.Stack();
        const aspectBranch = new TaggableResource(root, 'FakeBranchA', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite' },
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        const asgResource = new AsgTaggableResource(aspectBranch, 'FakeAsg', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite', propagateAtLaunch: false },
                    { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                ],
            },
        });
        const mapTaggable = new MapTaggableResource(aspectBranch, 'FakeSam', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: {
                    aspects: 'overwrite',
                    cfn: 'is cool',
                },
            },
        });
        const cfnBranch = new TaggableResource(root, 'FakeBranchB', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        aspectBranch.node.apply(new lib_1.Tag('aspects', 'rule'));
        root.node.prepareTree();
        test.deepEqual(aspectBranch.testProperties().tags, [{ key: 'aspects', value: 'rule' }, { key: 'cfn', value: 'is cool' }]);
        test.deepEqual(asgResource.testProperties().tags, [
            { key: 'aspects', value: 'rule', propagateAtLaunch: true },
            { key: 'cfn', value: 'is cool', propagateAtLaunch: true }
        ]);
        test.deepEqual(mapTaggable.testProperties().tags, {
            aspects: 'rule',
            cfn: 'is cool',
        });
        test.deepEqual(cfnBranch.testProperties().tags, [{ key: 'cfn', value: 'is cool' }]);
        test.done();
    },
    'when invalid tag properties are passed from L1s': {
        'map passed instead of array it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new TaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                        },
                    },
                });
            });
            test.throws(() => {
                new AsgTaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                            propagateAtLaunch: true,
                        },
                    },
                });
            });
            test.done();
        },
        'if array is passed instead of map it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new MapTaggableResource(root, 'FakeSam', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: [
                            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                            { key: 'aspects', value: 'overwrite' },
                        ],
                    },
                });
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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