"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk = require("../lib");
const lib_1 = require("../lib");
function createModernApp() {
    return new cdk.App({
        context: {
            [cxapi.DISABLE_VERSION_REPORTING]: 'true',
        }
    });
}
function list(outdir) {
    return fs.readdirSync(outdir).sort();
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
function writeJson(outdir, file, data) {
    fs.writeFileSync(path.join(outdir, file), JSON.stringify(data, undefined, 2));
}
module.exports = {
    'synthesis with an empty app'(test) {
        // GIVEN
        const app = createModernApp();
        // WHEN
        const session = app.run();
        // THEN
        test.same(app.run(), session); // same session if we run() again
        test.deepEqual(list(session.directory), ['cdk.out', 'manifest.json']);
        test.deepEqual(readJson(session.directory, 'manifest.json').artifacts, {});
        test.done();
    },
    'single empty stack'(test) {
        // GIVEN
        const app = createModernApp();
        new cdk.Stack(app, 'one-stack');
        // WHEN
        const session = app.run();
        // THEN
        test.deepEqual(list(session.directory), [
            'cdk.out',
            'manifest.json',
            'one-stack.template.json'
        ]);
        test.done();
    },
    'some random construct implements "synthesize"'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'one-stack');
        class MyConstruct extends cdk.Construct {
            synthesize(s) {
                writeJson(s.outdir, 'foo.json', { bar: 123 });
                s.addArtifact('my-random-construct', {
                    type: cxapi.ArtifactType.AwsCloudFormationStack,
                    environment: 'aws://12345/bar',
                    properties: {
                        templateFile: 'foo.json'
                    }
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct');
        // WHEN
        const session = app.run();
        // THEN
        test.deepEqual(list(session.directory), [
            'cdk.out',
            'foo.json',
            'manifest.json',
            'one-stack.template.json'
        ]);
        test.deepEqual(readJson(session.directory, 'foo.json'), { bar: 123 });
        test.deepEqual(session.manifest, {
            version: '0.33.0',
            artifacts: {
                'my-random-construct': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://12345/bar',
                    properties: { templateFile: 'foo.json' }
                },
                'one-stack': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://unknown-account/unknown-region',
                    properties: { templateFile: 'one-stack.template.json' },
                }
            },
        });
        test.done();
    },
    'it should be possible to synthesize without an app'(test) {
        const calls = new Array();
        class SynthesizeMe extends lib_1.Construct {
            constructor() {
                super(undefined, 'id');
            }
            validate() {
                calls.push('validate');
                return [];
            }
            prepare() {
                calls.push('prepare');
            }
            synthesize(session) {
                calls.push('synthesize');
                session.addArtifact('art', {
                    type: cxapi.ArtifactType.AwsCloudFormationStack,
                    properties: { templateFile: 'hey.json' },
                    environment: 'aws://unknown-account/us-east-1'
                });
                writeJson(session.outdir, 'hey.json', { hello: 123 });
            }
        }
        const root = new SynthesizeMe();
        const synth = new lib_1.Synthesizer();
        const assembly = synth.synthesize(root, { outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'outdir')) });
        test.deepEqual(calls, ['prepare', 'validate', 'synthesize']);
        const stack = assembly.getStack('art');
        test.deepEqual(stack.template, { hello: 123 });
        test.deepEqual(stack.properties, { templateFile: 'hey.json' });
        test.deepEqual(stack.environment, { region: 'us-east-1', account: 'unknown-account', name: 'aws://unknown-account/us-east-1' });
        test.done();
    },
    'stack.setParameterValue can be used to assign parameters'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'my-stack');
        const param = new cdk.CfnParameter(stack, 'MyParam', { type: 'string' });
        // WHEN
        stack.setParameterValue(param, 'Foo');
        // THEN
        const session = app.run();
        const props = session.getStack('my-stack').properties;
        test.deepEqual(props.parameters, {
            MyParam: 'Foo'
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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