"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
class StackWithPostProcessor extends lib_1.Stack {
    // ...
    _toCloudFormation() {
        const template = super._toCloudFormation();
        // manipulate template (e.g. rename "Key" to "key")
        template.Resources.myResource.Properties.Environment.key =
            template.Resources.myResource.Properties.Environment.Key;
        delete template.Resources.myResource.Properties.Environment.Key;
        return template;
    }
}
module.exports = {
    'a stack can be serialized into a CloudFormation template, initially it\'s empty'(test) {
        const stack = new lib_1.Stack();
        test.deepEqual(stack._toCloudFormation(), {});
        test.done();
    },
    'stack objects have some template-level propeties, such as Description, Version, Transform'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.templateFormatVersion = 'MyTemplateVersion';
        stack.templateOptions.description = 'This is my description';
        stack.templateOptions.transform = 'SAMy';
        test.deepEqual(stack._toCloudFormation(), {
            Description: 'This is my description',
            AWSTemplateFormatVersion: 'MyTemplateVersion',
            Transform: 'SAMy'
        });
        test.done();
    },
    'Stack.isStack indicates that a construct is a stack'(test) {
        const stack = new lib_1.Stack();
        const c = new lib_1.Construct(stack, 'Construct');
        test.ok(lib_1.Stack.isStack(stack));
        test.ok(!lib_1.Stack.isStack(c));
        test.done();
    },
    'stack.id is not included in the logical identities of resources within it'(test) {
        const stack = new lib_1.Stack(undefined, 'MyStack');
        new lib_1.CfnResource(stack, 'MyResource', { type: 'MyResourceType' });
        test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'MyResourceType' } } });
        test.done();
    },
    'stack.templateOptions can be used to set template-level options'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.description = 'StackDescription';
        stack.templateOptions.templateFormatVersion = 'TemplateVersion';
        stack.templateOptions.transform = 'Transform';
        stack.templateOptions.metadata = {
            MetadataKey: 'MetadataValue'
        };
        test.deepEqual(stack._toCloudFormation(), {
            Description: 'StackDescription',
            Transform: 'Transform',
            AWSTemplateFormatVersion: 'TemplateVersion',
            Metadata: { MetadataKey: 'MetadataValue' }
        });
        test.done();
    },
    // This approach will only apply to TypeScript code, but at least it's a temporary
    // workaround for people running into issues caused by SDK-3003.
    // We should come up with a proper solution that involved jsii callbacks (when they exist)
    // so this can be implemented by jsii languages as well.
    'Overriding `Stack._toCloudFormation` allows arbitrary post-processing of the generated template during synthesis'(test) {
        const stack = new StackWithPostProcessor();
        new lib_1.CfnResource(stack, 'myResource', {
            type: 'AWS::MyResource',
            properties: {
                MyProp1: 'hello',
                MyProp2: 'howdy',
                Environment: {
                    Key: 'value'
                }
            }
        });
        test.deepEqual(stack._toCloudFormation(), { Resources: { myResource: { Type: 'AWS::MyResource',
                    Properties: { MyProp1: 'hello',
                        MyProp2: 'howdy',
                        Environment: { key: 'value' } } } } });
        test.done();
    },
    'Construct.findResource(logicalId) can be used to retrieve a resource by its path'(test) {
        const stack = new lib_1.Stack();
        test.ok(!stack.node.tryFindChild('foo'), 'empty stack');
        const r1 = new lib_1.CfnResource(stack, 'Hello', { type: 'MyResource' });
        test.equal(stack.findResource(r1.stackPath), r1, 'look up top-level');
        const child = new lib_1.Construct(stack, 'Child');
        const r2 = new lib_1.CfnResource(child, 'Hello', { type: 'MyResource' });
        test.equal(stack.findResource(r2.stackPath), r2, 'look up child');
        test.done();
    },
    'Stack.findResource will fail if the element is not a resource'(test) {
        const stack = new lib_1.Stack();
        const p = new lib_1.CfnParameter(stack, 'MyParam', { type: 'String' });
        test.throws(() => stack.findResource(p.node.path));
        test.done();
    },
    'Stack.getByPath can be used to find any CloudFormation element (Parameter, Output, etc)'(test) {
        const stack = new lib_1.Stack();
        const p = new lib_1.CfnParameter(stack, 'MyParam', { type: 'String' });
        const o = new lib_1.CfnOutput(stack, 'MyOutput', { value: 'boom' });
        const c = new lib_1.CfnCondition(stack, 'MyCondition');
        test.equal(stack.node.findChild(p.node.path), p);
        test.equal(stack.node.findChild(o.node.path), o);
        test.equal(stack.node.findChild(c.node.path), c);
        test.done();
    },
    'Stack names can have hyphens in them'(test) {
        const root = new lib_1.App();
        new lib_1.Stack(root, 'Hello-World');
        // Did not throw
        test.done();
    },
    'Include should support non-hash top-level template elements like "Description"'(test) {
        const stack = new lib_1.Stack();
        const template = {
            Description: 'hello, world'
        };
        new lib_1.Include(stack, 'Include', { template });
        const output = stack._toCloudFormation();
        test.equal(typeof output.Description, 'string');
        test.done();
    },
    'Pseudo values attached to one stack can be referenced in another stack'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        // THEN
        // Need to do this manually now, since we're in testing mode. In a normal CDK app,
        // this happens as part of app.run().
        app.node.prepareTree();
        test.deepEqual(stack1._toCloudFormation(), {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(stack2._toCloudFormation(), {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack references are detected in resource properties'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', { type: 'BLA' });
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another resource
        new lib_1.CfnResource(stack2, 'SomeResource', { type: 'AWS::Some::Resource', properties: {
                someProperty: new lib_1.Token(() => resource1.ref),
            } });
        // THEN
        // Need to do this manually now, since we're in testing mode. In a normal CDK app,
        // this happens as part of app.run().
        app.node.prepareTree();
        test.deepEqual(stack2._toCloudFormation(), {
            Resources: {
                SomeResource: {
                    Type: 'AWS::Some::Resource',
                    Properties: {
                        someProperty: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefResource1D5D905A' }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack references in lazy tokens work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: new lib_1.Token(() => account1) });
        app.node.prepareTree();
        // THEN
        test.deepEqual(stack1._toCloudFormation(), {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(stack2._toCloudFormation(), {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack use of Region returns nonscoped intrinsic'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnOutput(stack2, 'DemOutput', { value: stack1.region });
        // THEN
        // Need to do this manually now, since we're in testing mode. In a normal CDK app,
        // this happens as part of app.run().
        app.node.prepareTree();
        test.deepEqual(stack2._toCloudFormation(), {
            Outputs: {
                DemOutput: {
                    Value: { Ref: 'AWS::Region' },
                }
            }
        });
        test.done();
    },
    'cross-stack references in strings work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: `TheAccountIs${account1}` });
        app.node.prepareTree();
        // THEN
        test.deepEqual(stack2._toCloudFormation(), {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::Join': ['', ['TheAccountIs', { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }]] }
                }
            }
        });
        test.done();
    },
    'cannot create cyclic reference between stacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        const account2 = new lib_1.ScopedAws(stack2).accountId;
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        new lib_1.CfnParameter(stack1, 'SomeParameter', { type: 'String', default: account2 });
        test.throws(() => {
            app.node.prepareTree();
            // tslint:disable-next-line:max-line-length
        }, "'Stack2' depends on 'Stack1' (Stack2/SomeParameter -> Stack1.AWS::AccountId). Adding this dependency (Stack1/SomeParameter -> Stack2.AWS::AccountId) would create a cyclic reference.");
        test.done();
    },
    'stacks know about their dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        app.node.prepareTree();
        // THEN
        test.deepEqual(stack2.dependencies().map(s => s.node.id), ['Stack1']);
        test.done();
    },
    'cannot create references to stacks in other regions/accounts'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2', { env: { account: '123456789012', region: 'es-norst-2' } });
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        test.throws(() => {
            app.node.prepareTree();
        }, /Can only reference cross stacks in the same region and account/);
        test.done();
    },
    'stack with region supplied via props returns literal value'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        // THEN
        test.equal(stack.node.resolve(stack.region), 'es-norst-1');
        test.done();
    },
    'stack with region supplied via context returns symbolic value'(test) {
        // GIVEN
        const app = new lib_1.App();
        app.node.setContext(cxapi.DEFAULT_REGION_CONTEXT_KEY, 'es-norst-1');
        const stack = new lib_1.Stack(app, 'Stack1');
        // THEN
        test.deepEqual(stack.node.resolve(stack.region), { Ref: 'AWS::Region' });
        test.done();
    },
    'overrideLogicalId(id) can be used to override the logical ID of a resource'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bonjour = new lib_1.CfnResource(stack, 'BonjourResource', { type: 'Resource::Type' });
        // { Ref } and { GetAtt }
        new lib_1.CfnResource(stack, 'RefToBonjour', { type: 'Other::Resource', properties: {
                RefToBonjour: bonjour.ref.toString(),
                GetAttBonjour: bonjour.getAtt('TheAtt').toString()
            } });
        bonjour.overrideLogicalId('BOOM');
        // THEN
        test.deepEqual(stack._toCloudFormation(), { Resources: { BOOM: { Type: 'Resource::Type' },
                RefToBonjour: { Type: 'Other::Resource',
                    Properties: { RefToBonjour: { Ref: 'BOOM' },
                        GetAttBonjour: { 'Fn::GetAtt': ['BOOM', 'TheAtt'] } } } } });
        test.done();
    },
    'Stack name can be overridden via properties'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, 'Stack', { stackName: 'otherName' });
        // THEN
        test.deepEqual(stack.name, 'otherName');
        test.done();
    },
    'Stack name is inherited from App name if available'(test) {
        // WHEN
        const root = new lib_1.App();
        const app = new lib_1.Construct(root, 'Prod');
        const stack = new lib_1.Stack(app, 'Stack');
        // THEN
        test.deepEqual(stack.name, 'ProdStackD5279B22');
        test.done();
    },
    'stack construct id does not go through stack name validation if there is an explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack = new lib_1.Stack(app, 'invalid as : stack name, but thats fine', {
            stackName: 'valid-stack-name'
        });
        // THEN
        const session = app.run();
        test.deepEqual(stack.name, 'valid-stack-name');
        test.ok(session.tryGetArtifact('valid-stack-name'));
        test.done();
    },
    'stack validation is performed on explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stack(app, 'boom', { stackName: 'invalid:stack:name' }), /Stack name must match the regular expression/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zdGFjay5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3Quc3RhY2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLHlDQUEwQztBQUUxQyxnQ0FBOEg7QUF1YzlILE1BQU0sc0JBQXVCLFNBQVEsV0FBSztJQUV4QyxNQUFNO0lBRUMsaUJBQWlCO1FBQ3RCLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1FBRTNDLG1EQUFtRDtRQUNuRCxRQUFRLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLEdBQUc7WUFDdEQsUUFBUSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUM7UUFDM0QsT0FBTyxRQUFRLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQztRQUVoRSxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0NBQ0Y7QUFuZEQsaUJBQVM7SUFDUCxpRkFBaUYsQ0FBQyxJQUFVO1FBQzFGLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxFQUFHLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkZBQTJGLENBQUMsSUFBVTtRQUNwRyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLEtBQUssQ0FBQyxlQUFlLENBQUMscUJBQXFCLEdBQUcsbUJBQW1CLENBQUM7UUFDbEUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxXQUFXLEdBQUcsd0JBQXdCLENBQUM7UUFDN0QsS0FBSyxDQUFDLGVBQWUsQ0FBQyxTQUFTLEdBQUcsTUFBTSxDQUFDO1FBQ3pDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLEVBQUU7WUFDeEMsV0FBVyxFQUFFLHdCQUF3QjtZQUNyQyx3QkFBd0IsRUFBRSxtQkFBbUI7WUFDN0MsU0FBUyxFQUFFLE1BQU07U0FDbEIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFEQUFxRCxDQUFDLElBQVU7UUFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLENBQUMsR0FBRyxJQUFJLGVBQVMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDNUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFdBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMzQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkVBQTJFLENBQUMsSUFBVTtRQUNwRixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDOUMsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBRWpFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNyRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsaUVBQWlFLENBQUMsSUFBVTtRQUMxRSxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBRTFCLEtBQUssQ0FBQyxlQUFlLENBQUMsV0FBVyxHQUFHLGtCQUFrQixDQUFDO1FBQ3ZELEtBQUssQ0FBQyxlQUFlLENBQUMscUJBQXFCLEdBQUcsaUJBQWlCLENBQUM7UUFDaEUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxTQUFTLEdBQUcsV0FBVyxDQUFDO1FBQzlDLEtBQUssQ0FBQyxlQUFlLENBQUMsUUFBUSxHQUFHO1lBQy9CLFdBQVcsRUFBRSxlQUFlO1NBQzdCLENBQUM7UUFFRixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxFQUFFO1lBQ3hDLFdBQVcsRUFBRSxrQkFBa0I7WUFDL0IsU0FBUyxFQUFFLFdBQVc7WUFDdEIsd0JBQXdCLEVBQUUsaUJBQWlCO1lBQzNDLFFBQVEsRUFBRSxFQUFFLFdBQVcsRUFBRSxlQUFlLEVBQUU7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtGQUFrRjtJQUNsRixnRUFBZ0U7SUFDaEUsMEZBQTBGO0lBQzFGLHdEQUF3RDtJQUN4RCxrSEFBa0gsQ0FBQyxJQUFVO1FBRTNILE1BQU0sS0FBSyxHQUFHLElBQUksc0JBQXNCLEVBQUUsQ0FBQztRQUUzQyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtZQUNuQyxJQUFJLEVBQUUsaUJBQWlCO1lBQ3ZCLFVBQVUsRUFBRTtnQkFDVixPQUFPLEVBQUUsT0FBTztnQkFDaEIsT0FBTyxFQUFFLE9BQU87Z0JBQ2hCLFdBQVcsRUFBRTtvQkFDWCxHQUFHLEVBQUUsT0FBTztpQkFDYjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxFQUFFLFNBQVMsRUFDbkQsRUFBRSxVQUFVLEVBQ1QsRUFBRSxJQUFJLEVBQUUsaUJBQWlCO29CQUN6QixVQUFVLEVBQ1QsRUFBRSxPQUFPLEVBQUUsT0FBTzt3QkFDbEIsT0FBTyxFQUFFLE9BQU87d0JBQ2hCLFdBQVcsRUFBRSxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFN0MsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtGQUFrRixDQUFDLElBQVU7UUFDM0YsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFeEQsTUFBTSxFQUFFLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUNuRSxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO1FBRXRFLE1BQU0sS0FBSyxHQUFHLElBQUksZUFBUyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM1QyxNQUFNLEVBQUUsR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBRW5FLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsRUFBRSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRWxFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrREFBK0QsQ0FBQyxJQUFVO1FBQ3hFLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxDQUFDLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUVqRSxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5RkFBeUYsQ0FBQyxJQUFVO1FBRWxHLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxDQUFDLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUNqRSxNQUFNLENBQUMsR0FBRyxJQUFJLGVBQVMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFDOUQsTUFBTSxDQUFDLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxhQUFhLENBQUMsQ0FBQztRQUVqRCxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUVqRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0NBQXNDLENBQUMsSUFBVTtRQUMvQyxNQUFNLElBQUksR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBRXZCLElBQUksV0FBSyxDQUFDLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQztRQUMvQixnQkFBZ0I7UUFFaEIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdGQUFnRixDQUFDLElBQVU7UUFDekYsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLFFBQVEsR0FBRztZQUNmLFdBQVcsRUFBRSxjQUFjO1NBQzVCLENBQUM7UUFFRixJQUFJLGFBQU8sQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUU1QyxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztRQUV6QyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sTUFBTSxDQUFDLFdBQVcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsd0VBQXdFLENBQUMsSUFBVTtRQUNqRixRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDeEMsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFTLENBQUMsTUFBTSxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2pELE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUV4QywrQkFBK0I7UUFDL0IsSUFBSSxrQkFBWSxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBRWpGLE9BQU87UUFDUCxrRkFBa0Y7UUFDbEYscUNBQXFDO1FBQ3JDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsaUJBQWlCLEVBQUUsRUFBRTtZQUN6QyxPQUFPLEVBQUU7Z0JBQ1Asb0NBQW9DLEVBQUU7b0JBQ3BDLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQkFDaEMsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLDZDQUE2QyxFQUFFO2lCQUNoRTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsaUJBQWlCLEVBQUUsRUFBRTtZQUN6QyxVQUFVLEVBQUU7Z0JBQ1YsYUFBYSxFQUFFO29CQUNiLElBQUksRUFBRSxRQUFRO29CQUNkLE9BQU8sRUFBRSxFQUFFLGlCQUFpQixFQUFFLDZDQUE2QyxFQUFFO2lCQUM5RTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDREQUE0RCxDQUFDLElBQVU7UUFDckUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sU0FBUyxHQUFHLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDdkUsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLGtDQUFrQztRQUNsQyxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLGNBQWMsRUFBRSxFQUFFLElBQUksRUFBRSxxQkFBcUIsRUFBRSxVQUFVLEVBQUU7Z0JBQ2pGLFlBQVksRUFBRSxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDO2FBQzdDLEVBQUMsQ0FBQyxDQUFDO1FBRUosT0FBTztRQUNQLGtGQUFrRjtRQUNsRixxQ0FBcUM7UUFDckMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUV2QixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRSxFQUFFO1lBQ3pDLFNBQVMsRUFBRTtnQkFDVCxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLHFCQUFxQjtvQkFDM0IsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRSxFQUFFLGlCQUFpQixFQUFFLHlDQUF5QyxFQUFFO3FCQUMvRTtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDRDQUE0QyxDQUFDLElBQVU7UUFDckQsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sUUFBUSxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUNqRCxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFeEMsK0JBQStCO1FBQy9CLElBQUksa0JBQVksQ0FBQyxNQUFNLEVBQUUsZUFBZSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQUUsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRWxHLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7UUFFdkIsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLGlCQUFpQixFQUFFLEVBQUU7WUFDekMsT0FBTyxFQUFFO2dCQUNQLG9DQUFvQyxFQUFFO29CQUNwQyxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ2hDLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSw2Q0FBNkMsRUFBRTtpQkFDaEU7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLGlCQUFpQixFQUFFLEVBQUU7WUFDekMsVUFBVSxFQUFFO2dCQUNWLGFBQWEsRUFBRTtvQkFDYixJQUFJLEVBQUUsUUFBUTtvQkFDZCxPQUFPLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSw2Q0FBNkMsRUFBRTtpQkFDOUU7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1REFBdUQsQ0FBQyxJQUFVO1FBQ2hFLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFeEMsK0JBQStCO1FBQy9CLElBQUksZUFBUyxDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFFN0QsT0FBTztRQUNQLGtGQUFrRjtRQUNsRixxQ0FBcUM7UUFDckMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUV2QixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRSxFQUFFO1lBQ3pDLE9BQU8sRUFBRTtnQkFDUCxTQUFTLEVBQUU7b0JBQ1QsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTtpQkFDOUI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3Q0FBd0MsQ0FBQyxJQUFVO1FBQ2pELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLCtCQUErQjtRQUMvQixJQUFJLGtCQUFZLENBQUMsTUFBTSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLGVBQWUsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWxHLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7UUFFdkIsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLGlCQUFpQixFQUFFLEVBQUU7WUFDekMsVUFBVSxFQUFFO2dCQUNWLGFBQWEsRUFBRTtvQkFDYixJQUFJLEVBQUUsUUFBUTtvQkFDZCxPQUFPLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBRSxFQUFFLEVBQUUsQ0FBRSxjQUFjLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSw2Q0FBNkMsRUFBRSxDQUFFLENBQUMsRUFBRTtpQkFDekg7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQ0FBK0MsQ0FBQyxJQUFVO1FBQ3hELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sUUFBUSxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVqRCxPQUFPO1FBQ1AsSUFBSSxrQkFBWSxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ2pGLElBQUksa0JBQVksQ0FBQyxNQUFNLEVBQUUsZUFBZSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUVqRixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7WUFDdkIsMkNBQTJDO1FBQzdDLENBQUMsRUFBRSx1TEFBdUwsQ0FBQyxDQUFDO1FBRTVMLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzQ0FBc0MsQ0FBQyxJQUFVO1FBQy9DLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLE9BQU87UUFDUCxJQUFJLGtCQUFZLENBQUMsTUFBTSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFFakYsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUV2QixPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFdEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhEQUE4RCxDQUFDLElBQVU7UUFDdkUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFDLENBQUMsQ0FBQztRQUNuRyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFDLENBQUMsQ0FBQztRQUVuRyxPQUFPO1FBQ1AsSUFBSSxrQkFBWSxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBRWpGLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUN6QixDQUFDLEVBQUUsZ0VBQWdFLENBQUMsQ0FBQztRQUVyRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNERBQTRELENBQUMsSUFBVTtRQUNyRSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFLEVBQUMsQ0FBQyxDQUFDO1FBRWxHLE9BQU87UUFDUCxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUUzRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0RBQStELENBQUMsSUFBVTtRQUN4RSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUV0QixHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsMEJBQTBCLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXZDLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBRXpFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0RUFBNEUsQ0FBQyxJQUFVO1FBQ3JGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sT0FBTyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBRXRGLHlCQUF5QjtRQUN6QixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLElBQUksRUFBRSxpQkFBaUIsRUFBRSxVQUFVLEVBQUU7Z0JBQzVFLFlBQVksRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRTtnQkFDcEMsYUFBYSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsUUFBUSxFQUFFO2FBQ25ELEVBQUMsQ0FBQyxDQUFDO1FBRUosT0FBTyxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRWxDLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUNuRCxFQUFFLElBQUksRUFBRSxFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDaEMsWUFBWSxFQUNYLEVBQUUsSUFBSSxFQUFFLGlCQUFpQjtvQkFDdkIsVUFBVSxFQUNULEVBQUUsWUFBWSxFQUFFLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRTt3QkFDN0IsYUFBYSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsTUFBTSxFQUFFLFFBQVEsQ0FBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3pFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2Q0FBNkMsQ0FBQyxJQUFVO1FBQ3RELE9BQU87UUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxTQUFTLEVBQUUsT0FBTyxFQUFFLEVBQUUsU0FBUyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFFeEUsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztRQUV4QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0RBQW9ELENBQUMsSUFBVTtRQUM3RCxPQUFPO1FBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN2QixNQUFNLEdBQUcsR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXRDLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUVoRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsaUdBQWlHLENBQUMsSUFBVTtRQUMxRyxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUV0QixPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLHlDQUF5QyxFQUFFO1lBQ3RFLFNBQVMsRUFBRSxrQkFBa0I7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sT0FBTyxHQUFHLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMxQixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzREFBc0QsQ0FBQyxJQUFVO1FBQy9ELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBRXRCLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxNQUFNLEVBQUUsRUFBRSxTQUFTLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQyxFQUMzRSw4Q0FBOEMsQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGN4YXBpID0gcmVxdWlyZSgnQGF3cy1jZGsvY3gtYXBpJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQXBwLCBDZm5Db25kaXRpb24sIENmbk91dHB1dCwgQ2ZuUGFyYW1ldGVyLCBDZm5SZXNvdXJjZSwgQ29uc3RydWN0LCBJbmNsdWRlLCBTY29wZWRBd3MsIFN0YWNrLCBUb2tlbiB9IGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ2Egc3RhY2sgY2FuIGJlIHNlcmlhbGl6ZWQgaW50byBhIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLCBpbml0aWFsbHkgaXRcXCdzIGVtcHR5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5fdG9DbG91ZEZvcm1hdGlvbigpLCB7IH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjayBvYmplY3RzIGhhdmUgc29tZSB0ZW1wbGF0ZS1sZXZlbCBwcm9wZXRpZXMsIHN1Y2ggYXMgRGVzY3JpcHRpb24sIFZlcnNpb24sIFRyYW5zZm9ybScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgc3RhY2sudGVtcGxhdGVPcHRpb25zLnRlbXBsYXRlRm9ybWF0VmVyc2lvbiA9ICdNeVRlbXBsYXRlVmVyc2lvbic7XG4gICAgc3RhY2sudGVtcGxhdGVPcHRpb25zLmRlc2NyaXB0aW9uID0gJ1RoaXMgaXMgbXkgZGVzY3JpcHRpb24nO1xuICAgIHN0YWNrLnRlbXBsYXRlT3B0aW9ucy50cmFuc2Zvcm0gPSAnU0FNeSc7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2suX3RvQ2xvdWRGb3JtYXRpb24oKSwge1xuICAgICAgRGVzY3JpcHRpb246ICdUaGlzIGlzIG15IGRlc2NyaXB0aW9uJyxcbiAgICAgIEFXU1RlbXBsYXRlRm9ybWF0VmVyc2lvbjogJ015VGVtcGxhdGVWZXJzaW9uJyxcbiAgICAgIFRyYW5zZm9ybTogJ1NBTXknXG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrLmlzU3RhY2sgaW5kaWNhdGVzIHRoYXQgYSBjb25zdHJ1Y3QgaXMgYSBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgYyA9IG5ldyBDb25zdHJ1Y3Qoc3RhY2ssICdDb25zdHJ1Y3QnKTtcbiAgICB0ZXN0Lm9rKFN0YWNrLmlzU3RhY2soc3RhY2spKTtcbiAgICB0ZXN0Lm9rKCFTdGFjay5pc1N0YWNrKGMpKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2suaWQgaXMgbm90IGluY2x1ZGVkIGluIHRoZSBsb2dpY2FsIGlkZW50aXRpZXMgb2YgcmVzb3VyY2VzIHdpdGhpbiBpdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKHVuZGVmaW5lZCwgJ015U3RhY2snKTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdNeVJlc291cmNlJywgeyB0eXBlOiAnTXlSZXNvdXJjZVR5cGUnIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2suX3RvQ2xvdWRGb3JtYXRpb24oKSwgeyBSZXNvdXJjZXM6IHsgTXlSZXNvdXJjZTogeyBUeXBlOiAnTXlSZXNvdXJjZVR5cGUnIH0gfSB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2sudGVtcGxhdGVPcHRpb25zIGNhbiBiZSB1c2VkIHRvIHNldCB0ZW1wbGF0ZS1sZXZlbCBvcHRpb25zJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIHN0YWNrLnRlbXBsYXRlT3B0aW9ucy5kZXNjcmlwdGlvbiA9ICdTdGFja0Rlc2NyaXB0aW9uJztcbiAgICBzdGFjay50ZW1wbGF0ZU9wdGlvbnMudGVtcGxhdGVGb3JtYXRWZXJzaW9uID0gJ1RlbXBsYXRlVmVyc2lvbic7XG4gICAgc3RhY2sudGVtcGxhdGVPcHRpb25zLnRyYW5zZm9ybSA9ICdUcmFuc2Zvcm0nO1xuICAgIHN0YWNrLnRlbXBsYXRlT3B0aW9ucy5tZXRhZGF0YSA9IHtcbiAgICAgIE1ldGFkYXRhS2V5OiAnTWV0YWRhdGFWYWx1ZSdcbiAgICB9O1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2suX3RvQ2xvdWRGb3JtYXRpb24oKSwge1xuICAgICAgRGVzY3JpcHRpb246ICdTdGFja0Rlc2NyaXB0aW9uJyxcbiAgICAgIFRyYW5zZm9ybTogJ1RyYW5zZm9ybScsXG4gICAgICBBV1NUZW1wbGF0ZUZvcm1hdFZlcnNpb246ICdUZW1wbGF0ZVZlcnNpb24nLFxuICAgICAgTWV0YWRhdGE6IHsgTWV0YWRhdGFLZXk6ICdNZXRhZGF0YVZhbHVlJyB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAvLyBUaGlzIGFwcHJvYWNoIHdpbGwgb25seSBhcHBseSB0byBUeXBlU2NyaXB0IGNvZGUsIGJ1dCBhdCBsZWFzdCBpdCdzIGEgdGVtcG9yYXJ5XG4gIC8vIHdvcmthcm91bmQgZm9yIHBlb3BsZSBydW5uaW5nIGludG8gaXNzdWVzIGNhdXNlZCBieSBTREstMzAwMy5cbiAgLy8gV2Ugc2hvdWxkIGNvbWUgdXAgd2l0aCBhIHByb3BlciBzb2x1dGlvbiB0aGF0IGludm9sdmVkIGpzaWkgY2FsbGJhY2tzICh3aGVuIHRoZXkgZXhpc3QpXG4gIC8vIHNvIHRoaXMgY2FuIGJlIGltcGxlbWVudGVkIGJ5IGpzaWkgbGFuZ3VhZ2VzIGFzIHdlbGwuXG4gICdPdmVycmlkaW5nIGBTdGFjay5fdG9DbG91ZEZvcm1hdGlvbmAgYWxsb3dzIGFyYml0cmFyeSBwb3N0LXByb2Nlc3Npbmcgb2YgdGhlIGdlbmVyYXRlZCB0ZW1wbGF0ZSBkdXJpbmcgc3ludGhlc2lzJyh0ZXN0OiBUZXN0KSB7XG5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFja1dpdGhQb3N0UHJvY2Vzc29yKCk7XG5cbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdteVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6TXlSZXNvdXJjZScsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIE15UHJvcDE6ICdoZWxsbycsXG4gICAgICAgIE15UHJvcDI6ICdob3dkeScsXG4gICAgICAgIEVudmlyb25tZW50OiB7XG4gICAgICAgICAgS2V5OiAndmFsdWUnXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLl90b0Nsb3VkRm9ybWF0aW9uKCksIHsgUmVzb3VyY2VzOlxuICAgICAgeyBteVJlc291cmNlOlxuICAgICAgICAgeyBUeXBlOiAnQVdTOjpNeVJlc291cmNlJyxcbiAgICAgICAgIFByb3BlcnRpZXM6XG4gICAgICAgICAgeyBNeVByb3AxOiAnaGVsbG8nLFxuICAgICAgICAgIE15UHJvcDI6ICdob3dkeScsXG4gICAgICAgICAgRW52aXJvbm1lbnQ6IHsga2V5OiAndmFsdWUnIH0gfSB9IH0gfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQ29uc3RydWN0LmZpbmRSZXNvdXJjZShsb2dpY2FsSWQpIGNhbiBiZSB1c2VkIHRvIHJldHJpZXZlIGEgcmVzb3VyY2UgYnkgaXRzIHBhdGgnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgdGVzdC5vayghc3RhY2subm9kZS50cnlGaW5kQ2hpbGQoJ2ZvbycpLCAnZW1wdHkgc3RhY2snKTtcblxuICAgIGNvbnN0IHIxID0gbmV3IENmblJlc291cmNlKHN0YWNrLCAnSGVsbG8nLCB7IHR5cGU6ICdNeVJlc291cmNlJyB9KTtcbiAgICB0ZXN0LmVxdWFsKHN0YWNrLmZpbmRSZXNvdXJjZShyMS5zdGFja1BhdGgpLCByMSwgJ2xvb2sgdXAgdG9wLWxldmVsJyk7XG5cbiAgICBjb25zdCBjaGlsZCA9IG5ldyBDb25zdHJ1Y3Qoc3RhY2ssICdDaGlsZCcpO1xuICAgIGNvbnN0IHIyID0gbmV3IENmblJlc291cmNlKGNoaWxkLCAnSGVsbG8nLCB7IHR5cGU6ICdNeVJlc291cmNlJyB9KTtcblxuICAgIHRlc3QuZXF1YWwoc3RhY2suZmluZFJlc291cmNlKHIyLnN0YWNrUGF0aCksIHIyLCAnbG9vayB1cCBjaGlsZCcpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrLmZpbmRSZXNvdXJjZSB3aWxsIGZhaWwgaWYgdGhlIGVsZW1lbnQgaXMgbm90IGEgcmVzb3VyY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgcCA9IG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2ssICdNeVBhcmFtJywgeyB0eXBlOiAnU3RyaW5nJyB9KTtcblxuICAgIHRlc3QudGhyb3dzKCgpID0+IHN0YWNrLmZpbmRSZXNvdXJjZShwLm5vZGUucGF0aCkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFjay5nZXRCeVBhdGggY2FuIGJlIHVzZWQgdG8gZmluZCBhbnkgQ2xvdWRGb3JtYXRpb24gZWxlbWVudCAoUGFyYW1ldGVyLCBPdXRwdXQsIGV0YyknKHRlc3Q6IFRlc3QpIHtcblxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBjb25zdCBwID0gbmV3IENmblBhcmFtZXRlcihzdGFjaywgJ015UGFyYW0nLCB7IHR5cGU6ICdTdHJpbmcnIH0pO1xuICAgIGNvbnN0IG8gPSBuZXcgQ2ZuT3V0cHV0KHN0YWNrLCAnTXlPdXRwdXQnLCB7IHZhbHVlOiAnYm9vbScgfSk7XG4gICAgY29uc3QgYyA9IG5ldyBDZm5Db25kaXRpb24oc3RhY2ssICdNeUNvbmRpdGlvbicpO1xuXG4gICAgdGVzdC5lcXVhbChzdGFjay5ub2RlLmZpbmRDaGlsZChwLm5vZGUucGF0aCksIHApO1xuICAgIHRlc3QuZXF1YWwoc3RhY2subm9kZS5maW5kQ2hpbGQoby5ub2RlLnBhdGgpLCBvKTtcbiAgICB0ZXN0LmVxdWFsKHN0YWNrLm5vZGUuZmluZENoaWxkKGMubm9kZS5wYXRoKSwgYyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnU3RhY2sgbmFtZXMgY2FuIGhhdmUgaHlwaGVucyBpbiB0aGVtJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBBcHAoKTtcblxuICAgIG5ldyBTdGFjayhyb290LCAnSGVsbG8tV29ybGQnKTtcbiAgICAvLyBEaWQgbm90IHRocm93XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnSW5jbHVkZSBzaG91bGQgc3VwcG9ydCBub24taGFzaCB0b3AtbGV2ZWwgdGVtcGxhdGUgZWxlbWVudHMgbGlrZSBcIkRlc2NyaXB0aW9uXCInKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgdGVtcGxhdGUgPSB7XG4gICAgICBEZXNjcmlwdGlvbjogJ2hlbGxvLCB3b3JsZCdcbiAgICB9O1xuXG4gICAgbmV3IEluY2x1ZGUoc3RhY2ssICdJbmNsdWRlJywgeyB0ZW1wbGF0ZSB9KTtcblxuICAgIGNvbnN0IG91dHB1dCA9IHN0YWNrLl90b0Nsb3VkRm9ybWF0aW9uKCk7XG5cbiAgICB0ZXN0LmVxdWFsKHR5cGVvZiBvdXRwdXQuRGVzY3JpcHRpb24sICdzdHJpbmcnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnUHNldWRvIHZhbHVlcyBhdHRhY2hlZCB0byBvbmUgc3RhY2sgY2FuIGJlIHJlZmVyZW5jZWQgaW4gYW5vdGhlciBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTiAtIHVzZWQgaW4gYW5vdGhlciBzdGFja1xuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IGFjY291bnQxIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIC8vIE5lZWQgdG8gZG8gdGhpcyBtYW51YWxseSBub3csIHNpbmNlIHdlJ3JlIGluIHRlc3RpbmcgbW9kZS4gSW4gYSBub3JtYWwgQ0RLIGFwcCxcbiAgICAvLyB0aGlzIGhhcHBlbnMgYXMgcGFydCBvZiBhcHAucnVuKCkuXG4gICAgYXBwLm5vZGUucHJlcGFyZVRyZWUoKTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS5fdG9DbG91ZEZvcm1hdGlvbigpLCB7XG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIEV4cG9ydHNPdXRwdXRSZWZBV1NBY2NvdW50SWRBRDU2ODA1Nzoge1xuICAgICAgICAgIFZhbHVlOiB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LFxuICAgICAgICAgIEV4cG9ydDogeyBOYW1lOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRSZWZBV1NBY2NvdW50SWRBRDU2ODA1NycgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazIuX3RvQ2xvdWRGb3JtYXRpb24oKSwge1xuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICBTb21lUGFyYW1ldGVyOiB7XG4gICAgICAgICAgVHlwZTogJ1N0cmluZycsXG4gICAgICAgICAgRGVmYXVsdDogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrMTpFeHBvcnRzT3V0cHV0UmVmQVdTQWNjb3VudElkQUQ1NjgwNTcnIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0Nyb3NzLXN0YWNrIHJlZmVyZW5jZXMgYXJlIGRldGVjdGVkIGluIHJlc291cmNlIHByb3BlcnRpZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3QgcmVzb3VyY2UxID0gbmV3IENmblJlc291cmNlKHN0YWNrMSwgJ1Jlc291cmNlJywgeyB0eXBlOiAnQkxBJyB9KTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2syJyk7XG5cbiAgICAvLyBXSEVOIC0gdXNlZCBpbiBhbm90aGVyIHJlc291cmNlXG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrMiwgJ1NvbWVSZXNvdXJjZScsIHsgdHlwZTogJ0FXUzo6U29tZTo6UmVzb3VyY2UnLCBwcm9wZXJ0aWVzOiB7XG4gICAgICBzb21lUHJvcGVydHk6IG5ldyBUb2tlbigoKSA9PiByZXNvdXJjZTEucmVmKSxcbiAgICB9fSk7XG5cbiAgICAvLyBUSEVOXG4gICAgLy8gTmVlZCB0byBkbyB0aGlzIG1hbnVhbGx5IG5vdywgc2luY2Ugd2UncmUgaW4gdGVzdGluZyBtb2RlLiBJbiBhIG5vcm1hbCBDREsgYXBwLFxuICAgIC8vIHRoaXMgaGFwcGVucyBhcyBwYXJ0IG9mIGFwcC5ydW4oKS5cbiAgICBhcHAubm9kZS5wcmVwYXJlVHJlZSgpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLl90b0Nsb3VkRm9ybWF0aW9uKCksIHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBTb21lUmVzb3VyY2U6IHtcbiAgICAgICAgICBUeXBlOiAnQVdTOjpTb21lOjpSZXNvdXJjZScsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgc29tZVByb3BlcnR5OiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRSZWZSZXNvdXJjZTFENUQ5MDVBJyB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Nyb3NzLXN0YWNrIHJlZmVyZW5jZXMgaW4gbGF6eSB0b2tlbnMgd29yaycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTiAtIHVzZWQgaW4gYW5vdGhlciBzdGFja1xuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IG5ldyBUb2tlbigoKSA9PiBhY2NvdW50MSkgfSk7XG5cbiAgICBhcHAubm9kZS5wcmVwYXJlVHJlZSgpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS5fdG9DbG91ZEZvcm1hdGlvbigpLCB7XG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIEV4cG9ydHNPdXRwdXRSZWZBV1NBY2NvdW50SWRBRDU2ODA1Nzoge1xuICAgICAgICAgIFZhbHVlOiB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LFxuICAgICAgICAgIEV4cG9ydDogeyBOYW1lOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRSZWZBV1NBY2NvdW50SWRBRDU2ODA1NycgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazIuX3RvQ2xvdWRGb3JtYXRpb24oKSwge1xuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICBTb21lUGFyYW1ldGVyOiB7XG4gICAgICAgICAgVHlwZTogJ1N0cmluZycsXG4gICAgICAgICAgRGVmYXVsdDogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrMTpFeHBvcnRzT3V0cHV0UmVmQVdTQWNjb3VudElkQUQ1NjgwNTcnIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0Nyb3NzLXN0YWNrIHVzZSBvZiBSZWdpb24gcmV0dXJucyBub25zY29wZWQgaW50cmluc2ljJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMScpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdTdGFjazInKTtcblxuICAgIC8vIFdIRU4gLSB1c2VkIGluIGFub3RoZXIgc3RhY2tcbiAgICBuZXcgQ2ZuT3V0cHV0KHN0YWNrMiwgJ0RlbU91dHB1dCcsIHsgdmFsdWU6IHN0YWNrMS5yZWdpb24gfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgLy8gTmVlZCB0byBkbyB0aGlzIG1hbnVhbGx5IG5vdywgc2luY2Ugd2UncmUgaW4gdGVzdGluZyBtb2RlLiBJbiBhIG5vcm1hbCBDREsgYXBwLFxuICAgIC8vIHRoaXMgaGFwcGVucyBhcyBwYXJ0IG9mIGFwcC5ydW4oKS5cbiAgICBhcHAubm9kZS5wcmVwYXJlVHJlZSgpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLl90b0Nsb3VkRm9ybWF0aW9uKCksIHtcbiAgICAgIE91dHB1dHM6IHtcbiAgICAgICAgRGVtT3V0cHV0OiB7XG4gICAgICAgICAgVmFsdWU6IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjcm9zcy1zdGFjayByZWZlcmVuY2VzIGluIHN0cmluZ3Mgd29yaycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTiAtIHVzZWQgaW4gYW5vdGhlciBzdGFja1xuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IGBUaGVBY2NvdW50SXMke2FjY291bnQxfWAgfSk7XG5cbiAgICBhcHAubm9kZS5wcmVwYXJlVHJlZSgpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMi5fdG9DbG91ZEZvcm1hdGlvbigpLCB7XG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgIFNvbWVQYXJhbWV0ZXI6IHtcbiAgICAgICAgICBUeXBlOiAnU3RyaW5nJyxcbiAgICAgICAgICBEZWZhdWx0OiB7ICdGbjo6Sm9pbic6IFsgJycsIFsgJ1RoZUFjY291bnRJcycsIHsgJ0ZuOjpJbXBvcnRWYWx1ZSc6ICdTdGFjazE6RXhwb3J0c091dHB1dFJlZkFXU0FjY291bnRJZEFENTY4MDU3JyB9IF1dIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Nhbm5vdCBjcmVhdGUgY3ljbGljIHJlZmVyZW5jZSBiZXR3ZWVuIHN0YWNrcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuICAgIGNvbnN0IGFjY291bnQyID0gbmV3IFNjb3BlZEF3cyhzdGFjazIpLmFjY291bnRJZDtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuUGFyYW1ldGVyKHN0YWNrMiwgJ1NvbWVQYXJhbWV0ZXInLCB7IHR5cGU6ICdTdHJpbmcnLCBkZWZhdWx0OiBhY2NvdW50MSB9KTtcbiAgICBuZXcgQ2ZuUGFyYW1ldGVyKHN0YWNrMSwgJ1NvbWVQYXJhbWV0ZXInLCB7IHR5cGU6ICdTdHJpbmcnLCBkZWZhdWx0OiBhY2NvdW50MiB9KTtcblxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIGFwcC5ub2RlLnByZXBhcmVUcmVlKCk7XG4gICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgfSwgXCInU3RhY2syJyBkZXBlbmRzIG9uICdTdGFjazEnIChTdGFjazIvU29tZVBhcmFtZXRlciAtPiBTdGFjazEuQVdTOjpBY2NvdW50SWQpLiBBZGRpbmcgdGhpcyBkZXBlbmRlbmN5IChTdGFjazEvU29tZVBhcmFtZXRlciAtPiBTdGFjazIuQVdTOjpBY2NvdW50SWQpIHdvdWxkIGNyZWF0ZSBhIGN5Y2xpYyByZWZlcmVuY2UuXCIpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrcyBrbm93IGFib3V0IHRoZWlyIGRlcGVuZGVuY2llcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IGFjY291bnQxIH0pO1xuXG4gICAgYXBwLm5vZGUucHJlcGFyZVRyZWUoKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazIuZGVwZW5kZW5jaWVzKCkubWFwKHMgPT4gcy5ub2RlLmlkKSwgWydTdGFjazEnXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY2Fubm90IGNyZWF0ZSByZWZlcmVuY2VzIHRvIHN0YWNrcyBpbiBvdGhlciByZWdpb25zL2FjY291bnRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMScsIHsgZW52OiB7IGFjY291bnQ6ICcxMjM0NTY3ODkwMTInLCByZWdpb246ICdlcy1ub3JzdC0xJyB9fSk7XG4gICAgY29uc3QgYWNjb3VudDEgPSBuZXcgU2NvcGVkQXdzKHN0YWNrMSkuYWNjb3VudElkO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdTdGFjazInLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDU2Nzg5MDEyJywgcmVnaW9uOiAnZXMtbm9yc3QtMicgfX0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IGFjY291bnQxIH0pO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgYXBwLm5vZGUucHJlcGFyZVRyZWUoKTtcbiAgICB9LCAvQ2FuIG9ubHkgcmVmZXJlbmNlIGNyb3NzIHN0YWNrcyBpbiB0aGUgc2FtZSByZWdpb24gYW5kIGFjY291bnQvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjayB3aXRoIHJlZ2lvbiBzdXBwbGllZCB2aWEgcHJvcHMgcmV0dXJucyBsaXRlcmFsIHZhbHVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicsIHJlZ2lvbjogJ2VzLW5vcnN0LTEnIH19KTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmVxdWFsKHN0YWNrLm5vZGUucmVzb2x2ZShzdGFjay5yZWdpb24pLCAnZXMtbm9yc3QtMScpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrIHdpdGggcmVnaW9uIHN1cHBsaWVkIHZpYSBjb250ZXh0IHJldHVybnMgc3ltYm9saWMgdmFsdWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcblxuICAgIGFwcC5ub2RlLnNldENvbnRleHQoY3hhcGkuREVGQVVMVF9SRUdJT05fQ09OVEVYVF9LRVksICdlcy1ub3JzdC0xJyk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2subm9kZS5yZXNvbHZlKHN0YWNrLnJlZ2lvbiksIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ292ZXJyaWRlTG9naWNhbElkKGlkKSBjYW4gYmUgdXNlZCB0byBvdmVycmlkZSB0aGUgbG9naWNhbCBJRCBvZiBhIHJlc291cmNlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IGJvbmpvdXIgPSBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdCb25qb3VyUmVzb3VyY2UnLCB7IHR5cGU6ICdSZXNvdXJjZTo6VHlwZScgfSk7XG5cbiAgICAvLyB7IFJlZiB9IGFuZCB7IEdldEF0dCB9XG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrLCAnUmVmVG9Cb25qb3VyJywgeyB0eXBlOiAnT3RoZXI6OlJlc291cmNlJywgcHJvcGVydGllczoge1xuICAgICAgUmVmVG9Cb25qb3VyOiBib25qb3VyLnJlZi50b1N0cmluZygpLFxuICAgICAgR2V0QXR0Qm9uam91cjogYm9uam91ci5nZXRBdHQoJ1RoZUF0dCcpLnRvU3RyaW5nKClcbiAgICB9fSk7XG5cbiAgICBib25qb3VyLm92ZXJyaWRlTG9naWNhbElkKCdCT09NJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2suX3RvQ2xvdWRGb3JtYXRpb24oKSwgeyBSZXNvdXJjZXM6XG4gICAgICB7IEJPT006IHsgVHlwZTogJ1Jlc291cmNlOjpUeXBlJyB9LFxuICAgICAgICBSZWZUb0JvbmpvdXI6XG4gICAgICAgICB7IFR5cGU6ICdPdGhlcjo6UmVzb3VyY2UnLFxuICAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgICAgeyBSZWZUb0JvbmpvdXI6IHsgUmVmOiAnQk9PTScgfSxcbiAgICAgICAgICAgICAgR2V0QXR0Qm9uam91cjogeyAnRm46OkdldEF0dCc6IFsgJ0JPT00nLCAnVGhlQXR0JyBdIH0gfSB9IH0gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrIG5hbWUgY2FuIGJlIG92ZXJyaWRkZW4gdmlhIHByb3BlcnRpZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCAnU3RhY2snLCB7IHN0YWNrTmFtZTogJ290aGVyTmFtZScgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2submFtZSwgJ290aGVyTmFtZScpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrIG5hbWUgaXMgaW5oZXJpdGVkIGZyb20gQXBwIG5hbWUgaWYgYXZhaWxhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHJvb3QgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgYXBwID0gbmV3IENvbnN0cnVjdChyb290LCAnUHJvZCcpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2submFtZSwgJ1Byb2RTdGFja0Q1Mjc5QjIyJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2sgY29uc3RydWN0IGlkIGRvZXMgbm90IGdvIHRocm91Z2ggc3RhY2sgbmFtZSB2YWxpZGF0aW9uIGlmIHRoZXJlIGlzIGFuIGV4cGxpY2l0IHN0YWNrIG5hbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdpbnZhbGlkIGFzIDogc3RhY2sgbmFtZSwgYnV0IHRoYXRzIGZpbmUnLCB7XG4gICAgICBzdGFja05hbWU6ICd2YWxpZC1zdGFjay1uYW1lJ1xuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHNlc3Npb24gPSBhcHAucnVuKCk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2submFtZSwgJ3ZhbGlkLXN0YWNrLW5hbWUnKTtcbiAgICB0ZXN0Lm9rKHNlc3Npb24udHJ5R2V0QXJ0aWZhY3QoJ3ZhbGlkLXN0YWNrLW5hbWUnKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrIHZhbGlkYXRpb24gaXMgcGVyZm9ybWVkIG9uIGV4cGxpY2l0IHN0YWNrIG5hbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgU3RhY2soYXBwLCAnYm9vbScsIHsgc3RhY2tOYW1lOiAnaW52YWxpZDpzdGFjazpuYW1lJyB9KSxcbiAgICAgIC9TdGFjayBuYW1lIG11c3QgbWF0Y2ggdGhlIHJlZ3VsYXIgZXhwcmVzc2lvbi8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH1cbn07XG5cbmNsYXNzIFN0YWNrV2l0aFBvc3RQcm9jZXNzb3IgZXh0ZW5kcyBTdGFjayB7XG5cbiAgLy8gLi4uXG5cbiAgcHVibGljIF90b0Nsb3VkRm9ybWF0aW9uKCkge1xuICAgIGNvbnN0IHRlbXBsYXRlID0gc3VwZXIuX3RvQ2xvdWRGb3JtYXRpb24oKTtcblxuICAgIC8vIG1hbmlwdWxhdGUgdGVtcGxhdGUgKGUuZy4gcmVuYW1lIFwiS2V5XCIgdG8gXCJrZXlcIilcbiAgICB0ZW1wbGF0ZS5SZXNvdXJjZXMubXlSZXNvdXJjZS5Qcm9wZXJ0aWVzLkVudmlyb25tZW50LmtleSA9XG4gICAgICB0ZW1wbGF0ZS5SZXNvdXJjZXMubXlSZXNvdXJjZS5Qcm9wZXJ0aWVzLkVudmlyb25tZW50LktleTtcbiAgICBkZWxldGUgdGVtcGxhdGUuUmVzb3VyY2VzLm15UmVzb3VyY2UuUHJvcGVydGllcy5FbnZpcm9ubWVudC5LZXk7XG5cbiAgICByZXR1cm4gdGVtcGxhdGU7XG4gIH1cbn1cbiJdfQ==