"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
class Counter extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, { type: 'My::Counter', properties: { Count: props.Count } });
        this.arn = this.getAtt('Arn').toString();
        this.url = this.getAtt('URL').toString();
    }
    increment(by = 1) {
        this.properties.Count += by;
    }
}
function withoutHash(logId) {
    return logId.substr(0, logId.length - 8);
}
class CustomizableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, { type: 'MyResourceType', properties: props });
    }
    setProperty(key, value) {
        this.untypedPropertyOverrides[key] = value;
    }
    renderProperties(properties) {
        return {
            PROP1: properties.prop1,
            PROP2: properties.prop2,
            PROP3: properties.prop3
        };
    }
}
module.exports = {
    'all resources derive from Resource, which derives from Entity'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnResource(stack, 'MyResource', {
            type: 'MyResourceType',
            properties: {
                Prop1: 'p1', Prop2: 123
            }
        });
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                MyResource: {
                    Type: "MyResourceType",
                    Properties: {
                        Prop1: "p1",
                        Prop2: 123
                    }
                }
            }
        });
        test.done();
    },
    'resources must reside within a Stack and fail upon creation if not'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.CfnResource(root, 'R1', { type: 'ResourceType' }));
        test.done();
    },
    'all entities have a logical ID calculated based on their full path in the tree'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { namingScheme: new lib_1.HashedAddressingScheme() });
        const level1 = new lib_1.Construct(stack, 'level1');
        const level2 = new lib_1.Construct(level1, 'level2');
        const level3 = new lib_1.Construct(level2, 'level3');
        const res1 = new lib_1.CfnResource(level1, 'childoflevel1', { type: 'MyResourceType1' });
        const res2 = new lib_1.CfnResource(level3, 'childoflevel3', { type: 'MyResourceType2' });
        test.equal(withoutHash(stack.node.resolve(res1.logicalId)), 'level1childoflevel1');
        test.equal(withoutHash(stack.node.resolve(res2.logicalId)), 'level1level2level3childoflevel3');
        test.done();
    },
    'resource.props can only be accessed by derived classes'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 10 });
        res.increment();
        res.increment(2);
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 13 } }
            }
        });
        test.done();
    },
    'resource attributes can be retrieved using getAtt(s) or attribute properties'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 10 });
        new lib_1.CfnResource(stack, 'YourResource', {
            type: 'Type',
            properties: {
                CounterName: res.getAtt('Name'),
                CounterArn: res.arn,
                CounterURL: res.url,
            }
        });
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 10 } },
                YourResource: {
                    Type: 'Type',
                    Properties: {
                        CounterName: { 'Fn::GetAtt': ['MyResource', 'Name'] },
                        CounterArn: { 'Fn::GetAtt': ['MyResource', 'Arn'] },
                        CounterURL: { 'Fn::GetAtt': ['MyResource', 'URL'] }
                    }
                }
            }
        });
        test.done();
    },
    'ARN-type resource attributes have some common functionality'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 1 });
        new lib_1.CfnResource(stack, 'MyResource2', {
            type: 'Type',
            properties: {
                Perm: res.arn
            }
        });
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                MyResource: { Type: "My::Counter", Properties: { Count: 1 } },
                MyResource2: {
                    Type: "Type",
                    Properties: {
                        Perm: {
                            "Fn::GetAtt": ["MyResource", "Arn"]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'resource.addDependency(e) can be used to add a DependsOn on another resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new Counter(stack, 'Counter1', { Count: 1 });
        const r2 = new Counter(stack, 'Counter2', { Count: 1 });
        const r3 = new lib_1.CfnResource(stack, 'Resource3', { type: 'MyResourceType' });
        r2.node.addDependency(r1);
        r2.node.addDependency(r3);
        stack.node.prepareTree();
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                Counter1: {
                    Type: "My::Counter",
                    Properties: { Count: 1 }
                },
                Counter2: {
                    Type: "My::Counter",
                    Properties: { Count: 1 },
                    DependsOn: [
                        "Counter1",
                        "Resource3"
                    ]
                },
                Resource3: { Type: "MyResourceType" }
            }
        });
        test.done();
    },
    'if addDependency is called multiple times with the same resource, it will only appear once'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const r1 = new Counter(stack, 'Counter1', { Count: 1 });
        const dependent = new lib_1.CfnResource(stack, 'Dependent', { type: 'R' });
        // WHEN
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        // THEN
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                Counter1: {
                    Type: "My::Counter",
                    Properties: {
                        Count: 1
                    }
                },
                Dependent: {
                    Type: "R",
                    DependsOn: [
                        "Counter1"
                    ]
                }
            }
        });
        test.done();
    },
    'conditions can be attached to a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        const cond = new lib_1.CfnCondition(stack, 'MyCondition', { expression: lib_1.Fn.conditionNot(lib_1.Fn.conditionEquals('a', 'b')) });
        r1.options.condition = cond;
        test.deepEqual(stack._toCloudFormation(), {
            Resources: { Resource: { Type: 'Type', Condition: 'MyCondition' } },
            Conditions: { MyCondition: { 'Fn::Not': [{ 'Fn::Equals': ['a', 'b'] }] } }
        });
        test.done();
    },
    'creation/update/updateReplace/deletion policies can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.options.creationPolicy = { autoScalingCreationPolicy: { minSuccessfulInstancesPercent: 10 } };
        // tslint:disable-next-line:max-line-length
        r1.options.updatePolicy = {
            autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: false },
            autoScalingReplacingUpdate: { willReplace: true },
            codeDeployLambdaAliasUpdate: {
                applicationName: 'CodeDeployApplication',
                deploymentGroupName: 'CodeDeployDeploymentGroup',
                beforeAllowTrafficHook: 'lambda1',
            },
        };
        r1.options.deletionPolicy = lib_1.DeletionPolicy.Retain;
        r1.options.updateReplacePolicy = lib_1.DeletionPolicy.Snapshot;
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    CreationPolicy: { AutoScalingCreationPolicy: { MinSuccessfulInstancesPercent: 10 } },
                    UpdatePolicy: {
                        AutoScalingScheduledAction: { IgnoreUnmodifiedGroupSizeProperties: false },
                        AutoScalingReplacingUpdate: { WillReplace: true },
                        CodeDeployLambdaAliasUpdate: {
                            ApplicationName: 'CodeDeployApplication',
                            DeploymentGroupName: 'CodeDeployDeploymentGroup',
                            BeforeAllowTrafficHook: 'lambda1',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Snapshot'
                }
            }
        });
        test.done();
    },
    'update policies UseOnlineResharding flag'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.options.updatePolicy = { useOnlineResharding: true };
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    UpdatePolicy: {
                        UseOnlineResharding: true,
                    },
                }
            }
        });
        test.done();
    },
    'metadata can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.options.metadata = {
            MyKey: 10,
            MyValue: 99
        };
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                Resource: {
                    Type: "Type",
                    Metadata: {
                        MyKey: 10,
                        MyValue: 99
                    }
                }
            }
        });
        test.done();
    },
    'the "type" property is required when creating a resource'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => new lib_1.CfnResource(stack, 'Resource', { notypehere: true }));
        test.done();
    },
    'removal policy is a high level abstraction of deletion policy used by l2'(test) {
        const stack = new lib_1.Stack();
        const orphan = new lib_1.CfnResource(stack, 'Orphan', { type: 'T1' });
        const forbid = new lib_1.CfnResource(stack, 'Forbid', { type: 'T2' });
        const destroy = new lib_1.CfnResource(stack, 'Destroy', { type: 'T3' });
        lib_1.applyRemovalPolicy(orphan, lib_1.RemovalPolicy.Orphan);
        lib_1.applyRemovalPolicy(forbid, lib_1.RemovalPolicy.Forbid);
        lib_1.applyRemovalPolicy(destroy, lib_1.RemovalPolicy.Destroy);
        test.deepEqual(stack._toCloudFormation(), { Resources: { Orphan: { Type: 'T1', DeletionPolicy: 'Retain' },
                Forbid: { Type: 'T2', DeletionPolicy: 'Retain' },
                Destroy: { Type: 'T3' } } });
        test.done();
    },
    'addDependency adds all dependencyElements of dependent constructs'(test) {
        class C1 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.r1 = new lib_1.CfnResource(this, 'R1', { type: 'T1' });
                this.r2 = new lib_1.CfnResource(this, 'R2', { type: 'T2' });
            }
        }
        class C2 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.r3 = new lib_1.CfnResource(this, 'R3', { type: 'T3' });
            }
        }
        // C3 returns [ c2 ] for it's dependency elements
        // this should result in 'flattening' the list of elements.
        class C3 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new C2(this, 'C2');
            }
        }
        const stack = new lib_1.Stack();
        const c1 = new C1(stack, 'MyC1');
        const c2 = new C2(stack, 'MyC2');
        const c3 = new C3(stack, 'MyC3');
        const dependingResource = new lib_1.CfnResource(stack, 'MyResource', { type: 'R' });
        dependingResource.node.addDependency(c1, c2);
        dependingResource.node.addDependency(c3);
        stack.node.prepareTree();
        test.deepEqual(stack._toCloudFormation(), { Resources: { MyC1R1FB2A562F: { Type: 'T1' },
                MyC1R2AE2B5066: { Type: 'T2' },
                MyC2R3809EEAD6: { Type: 'T3' },
                MyC3C2R38CE6F9F7: { Type: 'T3' },
                MyResource: { Type: 'R',
                    DependsOn: ['MyC1R1FB2A562F',
                        'MyC1R2AE2B5066',
                        'MyC2R3809EEAD6',
                        'MyC3C2R38CE6F9F7'] } } });
        test.done();
    },
    'resource.ref returns the {Ref} token'(test) {
        const stack = new lib_1.Stack();
        const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'R' });
        test.deepEqual(stack.node.resolve(r.ref), { Ref: 'MyResource' });
        test.done();
    },
    'overrides': {
        'addOverride(p, v) allows assigning arbitrary values to synthesized resource definitions'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'AWS::Resource::Type' });
            // WHEN
            r.addOverride('Type', 'YouCanEvenOverrideTheType');
            r.addOverride('Metadata', { Key: 12 });
            r.addOverride('Use.Dot.Notation', 'To create subtrees');
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'YouCanEvenOverrideTheType',
                        Use: { Dot: { Notation: 'To create subtrees' } },
                        Metadata: { Key: 12 } } } });
            test.done();
        },
        'addOverride(p, null) will assign an "null" value'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                        }
                    }
                }
            });
            // WHEN
            r.addOverride('Properties.Hello.World.Value2', null);
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello', Value2: null } } } } } });
            test.done();
        },
        'addOverride(p, undefined) can be used to delete a value'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                        }
                    }
                }
            });
            // WHEN
            r.addOverride('Properties.Hello.World.Value2', undefined);
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello' } } } } } });
            test.done();
        },
        'addOverride(p, undefined) will not create empty trees'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'AWS::Resource::Type' });
            // WHEN
            r.addPropertyOverride('Tree.Exists', 42);
            r.addPropertyOverride('Tree.Does.Not.Exist', undefined);
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Tree: { Exists: 42 } } } } });
            test.done();
        },
        'addDeletionOverride(p) and addPropertyDeletionOverride(pp) are sugar `undefined`'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                            Value3: ['foo', 'bar']
                        }
                    }
                }
            });
            // WHEN
            r.addDeletionOverride('Properties.Hello.World.Value2');
            r.addPropertyDeletionOverride('Hello.World.Value3');
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello' } } } } } });
            test.done();
        },
        'addOverride(p, v) will overwrite any non-objects along the path'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: 42
                    }
                }
            });
            // WHEN
            r.addOverride('Properties.Override1', ['Hello', 123]);
            r.addOverride('Properties.Override1.Override2', { Heyy: [1] });
            r.addOverride('Properties.Hello.World.Foo.Bar', 42);
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Foo: { Bar: 42 } } },
                            Override1: {
                                Override2: { Heyy: [1] }
                            } } } } });
            test.done();
        },
        'addPropertyOverride(pp, v) is a sugar for overriding properties'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: { Hello: { World: 42 } }
            });
            // WHEN
            r.addPropertyOverride('Hello.World', { Hey: 'Jude' });
            // THEN
            test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Hey: 'Jude' } } } } } });
            test.done();
        },
        'untypedPropertyOverrides': {
            'can be used by derived classes to specify overrides before render()'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource', {
                    prop1: 'foo'
                });
                r.setProperty('prop2', 'bar');
                test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'MyResourceType',
                            Properties: { PROP1: 'foo', PROP2: 'bar' } } } });
                test.done();
            },
            '"properties" is undefined'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource');
                r.setProperty('prop3', 'zoo');
                test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'MyResourceType',
                            Properties: { PROP3: 'zoo' } } } });
                test.done();
            },
            '"properties" is empty'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource', {});
                r.setProperty('prop3', 'zoo');
                r.setProperty('prop2', 'hey');
                test.deepEqual(stack._toCloudFormation(), { Resources: { MyResource: { Type: 'MyResourceType',
                            Properties: { PROP2: 'hey', PROP3: 'zoo' } } } });
                test.done();
            },
        }
    },
    '"aws:cdk:path" metadata is added if "aws:cdk:path-metadata" context is set to true'(test) {
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.PATH_METADATA_ENABLE_CONTEXT, true);
        const parent = new lib_1.Construct(stack, 'Parent');
        new lib_1.CfnResource(parent, 'MyResource', {
            type: 'MyResourceType',
        });
        test.deepEqual(stack._toCloudFormation(), { Resources: { ParentMyResource4B1FDBCC: { Type: 'MyResourceType',
                    Metadata: { [cxapi.PATH_METADATA_KEY]: 'Parent/MyResource' } } } });
        test.done();
    },
    'cross-stack construct dependencies are not rendered but turned into stack dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stackA = new lib_1.Stack(app, 'StackA');
        const resA = new lib_1.CfnResource(stackA, 'Resource', { type: 'R' });
        const stackB = new lib_1.Stack(app, 'StackB');
        const resB = new lib_1.CfnResource(stackB, 'Resource', { type: 'R' });
        // WHEN
        resB.node.addDependency(resA);
        // THEN
        app.node.prepareTree();
        test.deepEqual(stackB._toCloudFormation(), {
            Resources: {
                Resource: {
                    Type: 'R'
                    // Notice absence of 'DependsOn'
                }
            }
        });
        test.deepEqual(stackB.dependencies().map(s => s.node.id), ['StackA']);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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