"use strict";
const lib_1 = require("../lib");
/**
 * These tests are executed once (for specific ID schemes)
 */
const uniqueTests = {
    'if the naming scheme uniquifies with a hash we can have the same concatenated identifier'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { namingScheme: new lib_1.HashedAddressingScheme() });
        const A = new lib_1.Construct(stack, 'A');
        new lib_1.CfnResource(A, 'BC', { type: 'Resource' });
        // WHEN
        const AB = new lib_1.Construct(stack, 'AB');
        new lib_1.CfnResource(AB, 'C', { type: 'Resource' });
        // THEN: no exception
        test.done();
    },
    'special case: if the resource is top-level, a hash is not added'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { namingScheme: new lib_1.HashedAddressingScheme() });
        // WHEN
        const r = new lib_1.CfnResource(stack, 'MyAwesomeness', { type: 'Resource' });
        // THEN
        test.equal(stack.node.resolve(r.logicalId), 'MyAwesomeness');
        test.done();
    },
    'Logical IDs can be renamed at the stack level'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.renameLogical('ParentThingResource75D1D9CB', 'Renamed');
        // WHEN
        const parent = new lib_1.Construct(stack, 'Parent');
        new lib_1.CfnResource(parent, 'ThingResource', { type: 'AWS::TAAS::Thing' });
        // THEN
        const template = stack._toCloudFormation();
        test.ok('Renamed' in template.Resources);
        test.done();
    },
    'Renames for objects that don\'t exist fail'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.renameLogical('DOESNOTEXIST', 'Renamed');
        // WHEN
        new lib_1.Construct(stack, 'Parent');
        // THEN
        test.throws(() => {
            stack._toCloudFormation();
        });
        test.done();
    },
    'ID Renames that collide with existing IDs should fail'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.renameLogical('ParentThingResource1916E7808', 'ParentThingResource2F19948CB');
        // WHEN
        const parent = new lib_1.Construct(stack, 'Parent');
        new lib_1.CfnResource(parent, 'ThingResource1', { type: 'AWS::TAAS::Thing' });
        // THEN
        test.throws(() => {
            new lib_1.CfnResource(parent, 'ThingResource2', { type: 'AWS::TAAS::Thing' });
        });
        test.done();
    },
    'hashed naming scheme filters constructs named "Resource" from the human portion'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        const parent = new lib_1.Construct(stack, 'Parent');
        const child1 = new lib_1.Construct(parent, 'Child');
        const child2 = new lib_1.Construct(child1, 'Resource');
        new lib_1.CfnResource(child2, 'HeyThere', { type: 'AWS::TAAS::Thing' });
        // THEN
        const template = stack._toCloudFormation();
        test.deepEqual(template, {
            Resources: {
                ParentChildHeyThere35220347: {
                    Type: 'AWS::TAAS::Thing'
                }
            }
        });
        test.done();
    },
    'can transparently wrap constructs using "Default" id'(test) {
        // GIVEN
        const stack1 = new lib_1.Stack();
        const parent1 = new lib_1.Construct(stack1, 'Parent');
        new lib_1.CfnResource(parent1, 'HeyThere', { type: 'AWS::TAAS::Thing' });
        const template1 = stack1._toCloudFormation();
        // AND
        const theId1 = Object.keys(template1.Resources)[0];
        test.equal('AWS::TAAS::Thing', template1.Resources[theId1].Type);
        // WHEN
        const stack2 = new lib_1.Stack();
        const parent2 = new lib_1.Construct(stack2, 'Parent');
        const invisibleWrapper = new lib_1.Construct(parent2, 'Default');
        new lib_1.CfnResource(invisibleWrapper, 'HeyThere', { type: 'AWS::TAAS::Thing' });
        const template2 = stack1._toCloudFormation();
        const theId2 = Object.keys(template2.Resources)[0];
        test.equal('AWS::TAAS::Thing', template2.Resources[theId2].Type);
        // THEN: same ID, same object
        test.equal(theId1, theId2);
        test.done();
    },
    'non-alphanumeric characters are removed from the human part of the logical ID'(test) {
        const scheme = new lib_1.HashedAddressingScheme();
        const val1 = scheme.allocateAddress(['Foo-bar', 'B00m', 'Hello_World', '&&Horray Horray.']);
        const val2 = scheme.allocateAddress(['Foobar', 'B00m', 'HelloWorld', 'HorrayHorray']);
        // same human part, different hash
        test.deepEqual(val1, 'FoobarB00mHelloWorldHorrayHorray640E99FB');
        test.deepEqual(val2, 'FoobarB00mHelloWorldHorrayHorray744334FD');
        test.done();
    },
    'non-alphanumeric characters are removed even if the ID has only one component'(test) {
        const scheme = new lib_1.HashedAddressingScheme();
        const val1 = scheme.allocateAddress(['Foo-bar']);
        // same human part, different hash
        test.deepEqual(val1, 'Foobar');
        test.done();
    }
};
const schemes = {
    "hashing scheme": new lib_1.HashedAddressingScheme(),
};
/**
 * These tests are executed for all generators
 */
const allSchemesTests = {
    'empty identifiers are not allowed'(scheme, test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { namingScheme: scheme });
        // WHEN
        test.throws(() => {
            new lib_1.CfnResource(stack, '.', { type: 'R' });
        });
        test.done();
    },
    'too large identifiers are truncated yet still remain unique'(scheme, test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { namingScheme: scheme });
        const A = new lib_1.Construct(stack, generateString(100));
        const B = new lib_1.Construct(A, generateString(100));
        // WHEN
        const firstPart = generateString(60);
        // The shared part has now exceeded the maximum length of CloudFormation identifiers
        // so the identity generator will have to something smart
        const C1 = new lib_1.CfnResource(B, firstPart + generateString(40), { type: 'Resource' });
        const C2 = new lib_1.CfnResource(B, firstPart + generateString(40), { type: 'Resource' });
        // THEN
        test.ok(C1.logicalId.length <= 255);
        test.ok(C2.logicalId.length <= 255);
        test.notEqual(C1, C2);
        test.done();
    },
    'Refs and dependencies will correctly reflect renames done at the stack level'(scheme, test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { namingScheme: scheme });
        stack.renameLogical('OriginalName', 'NewName');
        // WHEN
        const c1 = new lib_1.CfnResource(stack, 'OriginalName', { type: 'R1' });
        const ref = c1.ref;
        const c2 = new lib_1.CfnResource(stack, 'Construct2', { type: 'R2', properties: { ReferenceToR1: ref } });
        c2.node.addDependency(c1);
        // THEN
        stack.node.prepareTree();
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                NewName: {
                    Type: 'R1'
                },
                Construct2: {
                    Type: 'R2',
                    Properties: {
                        ReferenceToR1: { Ref: 'NewName' }
                    },
                    DependsOn: ['NewName']
                }
            }
        });
        test.done();
    },
};
// Combine the one-off tests and generate tests for each scheme
const exp = uniqueTests;
Object.keys(schemes).forEach(schemeName => {
    const scheme = schemes[schemeName];
    Object.keys(allSchemesTests).forEach(testName => {
        const testFunction = allSchemesTests[testName];
        exp[`${schemeName}: ${testName}`] = (test) => {
            testFunction(scheme, test);
        };
    });
});
function generateString(chars) {
    let s = '';
    for (let i = 0; i < chars; ++i) {
        s += randomAlpha();
    }
    return s;
    function randomAlpha() {
        return String.fromCharCode('a'.charCodeAt(0) + Math.floor(Math.random() * 26));
    }
}
module.exports = exp;
//# sourceMappingURL=data:application/json;base64,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