"use strict";
const fc = require("fast-check");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
function asyncTest(cb) {
    return async (test) => {
        let error;
        try {
            await cb(test);
        }
        catch (e) {
            error = e;
        }
        finally {
            test.doesNotThrow(() => {
                if (error) {
                    throw error;
                }
            });
            test.done();
        }
    };
}
const nonEmptyString = fc.string(1, 16);
const tokenish = fc.array(nonEmptyString, 2, 2).map(arr => ({ [arr[0]]: arr[1] }));
const anyValue = fc.oneof(nonEmptyString, tokenish);
function stringListToken(o) {
    return new lib_1.Token(o).toList();
}
function stringToken(o) {
    return new lib_1.Token(o).toString();
}
module.exports = nodeunit.testCase({
    'eager resolution for non-tokens': {
        'Fn.select'(test) {
            test.deepEqual(lib_1.Fn.select(2, ['hello', 'you', 'dude']), 'dude');
            test.done();
        },
        'Fn.split'(test) {
            test.deepEqual(lib_1.Fn.split(':', 'hello:world:yeah'), ['hello', 'world', 'yeah']);
            test.done();
        }
    },
    'FnJoin': {
        'rejects empty list of arguments to join'(test) {
            test.throws(() => lib_1.Fn.join('.', []));
            test.done();
        },
        'collapse nested FnJoins even if they contain tokens'(test) {
            const stack = new lib_1.Stack();
            const obj = lib_1.Fn.join('', [
                'a',
                lib_1.Fn.join('', [lib_1.Fn.getAtt('a', 'bc').toString(), 'c']),
                'd'
            ]);
            test.deepEqual(stack.node.resolve(obj), { 'Fn::Join': ["",
                    [
                        "a",
                        { 'Fn::GetAtt': ['a', 'bc'] },
                        'cd',
                    ]
                ] });
            test.done();
        },
        'resolves to the value if only one value is joined': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), anyValue, (delimiter, value) => _.isEqual(stack.node.resolve(lib_1.Fn.join(delimiter, [value])), value)), { verbose: true });
        }),
        'pre-concatenates string literals': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 15), (delimiter, values) => stack.node.resolve(lib_1.Fn.join(delimiter, values)) === values.join(delimiter)), { verbose: true });
        }),
        'pre-concatenates around tokens': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 3), tokenish, fc.array(nonEmptyString, 1, 3), (delimiter, prefix, obj, suffix) => _.isEqual(stack.node.resolve(lib_1.Fn.join(delimiter, [...prefix, stringToken(obj), ...suffix])), { 'Fn::Join': [delimiter, [prefix.join(delimiter), obj, suffix.join(delimiter)]] })), { verbose: true, seed: 1539874645005, path: "0:0:0:0:0:0:0:0:0" });
        }),
        'flattens joins nested under joins with same delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(anyValue), fc.array(anyValue, 1, 3), fc.array(anyValue), (delimiter, prefix, nested, suffix) => 
            // Gonna test
            _.isEqual(stack.node.resolve(lib_1.Fn.join(delimiter, [...prefix, lib_1.Fn.join(delimiter, nested), ...suffix])), stack.node.resolve(lib_1.Fn.join(delimiter, [...prefix, ...nested, ...suffix])))), { verbose: true });
        }),
        'does not flatten joins nested under joins with different delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.string(), fc.array(anyValue, 1, 3), fc.array(tokenish, 2, 3), fc.array(anyValue, 3), (delimiter1, delimiter2, prefix, nested, suffix) => {
                fc.pre(delimiter1 !== delimiter2);
                const join = lib_1.Fn.join(delimiter1, [...prefix, lib_1.Fn.join(delimiter2, stringListToken(nested)), ...suffix]);
                const resolved = stack.node.resolve(join);
                return resolved['Fn::Join'][1].find((e) => typeof e === 'object'
                    && ('Fn::Join' in e)
                    && e['Fn::Join'][0] === delimiter2) != null;
            }), { verbose: true });
        }),
    },
});
//# sourceMappingURL=data:application/json;base64,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