"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function firstKey(obj) {
    return Object.keys(obj)[0];
}
/**
 * Get the expected context key from a stack with missing parameters
 */
function expectedContextKey(stack) {
    return firstKey(stack.missingContext);
}
module.exports = {
    'AvailabilityZoneProvider returns a list with dummy values if the context is not available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const azs = new lib_1.AvailabilityZoneProvider(stack).availabilityZones;
        test.deepEqual(azs, ['dummy1a', 'dummy1b', 'dummy1c']);
        test.done();
    },
    'AvailabilityZoneProvider will return context list if available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = new lib_1.AvailabilityZoneProvider(stack).availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, ['us-east-1a', 'us-east-1b']);
        const azs = new lib_1.AvailabilityZoneProvider(stack).availabilityZones;
        test.deepEqual(azs, ['us-east-1a', 'us-east-1b']);
        test.done();
    },
    'AvailabilityZoneProvider will complain if not given a list'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = new lib_1.AvailabilityZoneProvider(stack).availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, 'not-a-list');
        test.throws(() => new lib_1.AvailabilityZoneProvider(stack).availabilityZones);
        test.done();
    },
    'ContextProvider consistently generates a key'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const provider = new lib_1.ContextProvider(stack, 'ssm', {
            parameterName: 'foo',
            anyStringParam: 'bar',
        });
        const key = provider.key;
        test.deepEqual(key, 'ssm:account=12345:anyStringParam=bar:parameterName=foo:region=us-east-1');
        const complex = new lib_1.ContextProvider(stack, 'vpc', {
            cidrBlock: '192.168.0.16',
            tags: { Name: 'MyVPC', Env: 'Preprod' },
            igw: false,
        });
        const complexKey = complex.key;
        test.deepEqual(complexKey, 'vpc:account=12345:cidrBlock=192.168.0.16:igw=false:region=us-east-1:tags.Env=Preprod:tags.Name=MyVPC');
        test.done();
    },
    'Key generation can contain arbitrarily deep structures'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const provider = new lib_1.ContextProvider(stack, 'provider', {
            list: [
                { key: 'key1', value: 'value1' },
                { key: 'key2', value: 'value2' },
            ],
        });
        // THEN
        test.equals(provider.key, 'provider:account=12345:list.0.key=key1:list.0.value=value1:list.1.key=key2:list.1.value=value2:region=us-east-1');
        test.done();
    },
    'SSM parameter provider will return context values if available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.SSMParameterProvider(stack, { parameterName: 'test' }).parameterValue();
        const key = expectedContextKey(stack);
        stack.node.setContext(key, 'abc');
        const ssmp = new lib_1.SSMParameterProvider(stack, { parameterName: 'test' });
        const azs = stack.node.resolve(ssmp.parameterValue());
        test.deepEqual(azs, 'abc');
        test.done();
    },
    'Return default values if "env" is undefined to facilitate unit tests, but also expect metadata to include "error" messages'(test) {
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'test-stack');
        const child = new lib_1.Construct(stack, 'ChildConstruct');
        test.deepEqual(new lib_1.AvailabilityZoneProvider(stack).availabilityZones, ['dummy1a', 'dummy1b', 'dummy1c']);
        test.deepEqual(new lib_1.SSMParameterProvider(child, { parameterName: 'foo' }).parameterValue(), 'dummy');
        const assembly = app.run();
        const output = assembly.getStack('test-stack');
        const metadata = output.metadata;
        const azError = metadata['/test-stack'].find(x => x.type === cxapi.ERROR_METADATA_KEY);
        const ssmError = metadata['/test-stack/ChildConstruct'].find(x => x.type === cxapi.ERROR_METADATA_KEY);
        test.ok(azError && azError.data.includes('Cannot determine scope for context provider availability-zones'));
        test.ok(ssmError && ssmError.data.includes('Cannot determine scope for context provider ssm'));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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