"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function createTree(context) {
    const root = new lib_1.App();
    if (context) {
        Object.keys(context).forEach(key => root.node.setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(root, 'Child1');
    const child2 = new lib_1.Construct(root, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1
    };
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
class ConstructWithRequired extends lib_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.requiredProp = this.node.required(props, 'requiredProp');
        this.anotherRequiredProp = this.node.required(props, 'anotherRequiredProp');
    }
}
module.exports = {
    'the "Root" construct is a special construct which can be used as the root of the tree'(test) {
        const root = new lib_1.App();
        test.equal(root.node.id, '', 'if not specified, name of a root construct is an empty string');
        test.ok(!root.node.scope, 'no parent');
        test.equal(root.node.children.length, 0, 'a construct is created without children'); // no children
        test.done();
    },
    'constructs cannot be created with an empty name unless they are root'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.Construct(root, ''));
        test.done();
    },
    'construct.name returns the name of the construct'(test) {
        const t = createTree();
        test.equal(t.child1.node.id, 'Child1');
        test.equal(t.child2.node.id, 'Child2');
        test.equal(t.child1_1.node.id, 'Child11');
        test.equal(t.child1_2.node.id, 'Child12');
        test.equal(t.child1_1_1.node.id, 'Child111');
        test.equal(t.child2_1.node.id, 'Child21');
        test.done();
    },
    'construct id can use any character except the path separator'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'valid');
        new lib_1.Construct(root, 'ValiD');
        new lib_1.Construct(root, 'Va123lid');
        new lib_1.Construct(root, 'v');
        new lib_1.Construct(root, '  invalid');
        new lib_1.Construct(root, 'invalid   ');
        new lib_1.Construct(root, '123invalid');
        new lib_1.Construct(root, 'in valid');
        new lib_1.Construct(root, 'in_Valid');
        new lib_1.Construct(root, 'in-Valid');
        new lib_1.Construct(root, 'in\\Valid');
        new lib_1.Construct(root, 'in.Valid');
        test.done();
    },
    'if construct id contains path seperators, they will be replaced by double-dash'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
        test.deepEqual(c.node.id, 'Boom--Boom--Bam');
        test.done();
    },
    'if "undefined" is forcefully used as an "id", it will be treated as an empty string'(test) {
        const c = new lib_1.Construct(undefined, undefined);
        test.deepEqual(c.node.id, '');
        test.done();
    },
    "dont allow unresolved tokens to be used in construct IDs"(test) {
        // GIVEN
        const root = new lib_1.App();
        const token = new lib_1.Token(() => 'lazy');
        // WHEN + THEN
        test.throws(() => new lib_1.Construct(root, `MyID: ${token}`), /Cannot use tokens in construct ID: MyID: \${Token/);
        test.done();
    },
    'construct.uniqueId returns a tree-unique alphanumeric id of this construct'(test) {
        const root = new lib_1.App();
        const child1 = new lib_1.Construct(root, 'This is the first child');
        const child2 = new lib_1.Construct(child1, 'Second level');
        const c1 = new lib_1.Construct(child2, 'My construct');
        const c2 = new lib_1.Construct(child1, 'My construct');
        test.deepEqual(c1.node.path, 'This is the first child/Second level/My construct');
        test.deepEqual(c2.node.path, 'This is the first child/My construct');
        test.deepEqual(c1.node.uniqueId, 'ThisisthefirstchildSecondlevelMyconstruct202131E0');
        test.deepEqual(c2.node.uniqueId, 'ThisisthefirstchildMyconstruct8C288DF9');
        test.done();
    },
    'cannot calculate uniqueId if the construct path is ["Default"]'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Default');
        test.throws(() => c.node.uniqueId, /Unable to calculate a unique id for an empty set of components/);
        test.done();
    },
    'construct.node.stack returns the correct stack'(test) {
        const stack = new lib_1.Stack();
        test.same(stack.node.stack, stack);
        const parent = new lib_1.Construct(stack, 'Parent');
        const construct = new lib_1.Construct(parent, 'Construct');
        test.same(construct.node.stack, stack);
        test.done();
    },
    'construct.node.stack throws when there is no parent Stack'(test) {
        const root = new lib_1.App();
        const construct = new lib_1.Construct(root, 'Construct');
        test.throws(() => construct.node.stack, /No stack could be identified for the construct at path/);
        test.done();
    },
    'construct.node.stack.formatArn forwards to the Stack'(test) {
        const stack = new lib_1.Stack();
        const components = { service: 'test', resource: 'test' };
        const dummyArn = 'arn:::dummy';
        stack.formatArn = (args) => {
            test.same(args, components);
            return dummyArn;
        };
        const construct = new lib_1.Construct(stack, 'Construct');
        test.same(construct.node.stack.formatArn(components), dummyArn);
        test.done();
    },
    'construct.node.stack.parseArn forwards to the Stack'(test) {
        const stack = new lib_1.Stack();
        const components = { service: 'test', resource: 'test' };
        const dummyArn = 'arn:::dummy';
        stack.parseArn = (arn) => {
            test.same(arn, dummyArn);
            return components;
        };
        const construct = new lib_1.Construct(stack, 'Construct');
        test.same(construct.node.stack.parseArn(dummyArn), components);
        test.done();
    },
    'construct.getChildren() returns an array of all children'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Child1');
        new lib_1.Construct(root, 'Child2');
        test.equal(child.node.children.length, 0, 'no children');
        test.equal(root.node.children.length, 2, 'two children are expected');
        test.done();
    },
    'construct.findChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.tryFindChild(child.node.id), child, 'findChild(name) can be used to retrieve the child from a parent');
        test.ok(!root.node.tryFindChild('NotFound'), 'findChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.getChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.findChild(child.node.id), child, 'getChild(name) can be used to retrieve the child from a parent');
        test.throws(() => {
            root.node.findChild('NotFound');
        }, '', 'getChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.toString() and construct.toTreeString() can be used for diagnostics'(test) {
        const t = createTree();
        test.equal(t.root.toString(), 'App');
        test.equal(t.child1_1_1.toString(), 'Construct [Child1/Child11/Child111]');
        test.equal(t.child2.toString(), 'Construct [Child2]');
        test.equal(t.root.node.toTreeString(), 'App\n  Construct [Child1]\n    Construct [Child11]\n      Construct [Child111]\n    Construct [Child12]\n  Construct [Child2]\n    Construct [Child21]\n');
        test.done();
    },
    'construct.getContext(key) can be used to read a value from context defined at the root level'(test) {
        const context = {
            ctx1: 12,
            ctx2: 'hello'
        };
        const t = createTree(context);
        test.equal(t.root.node.getContext('ctx1'), 12);
        test.equal(t.child1_1_1.node.getContext('ctx2'), 'hello');
        test.done();
    },
    'construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack'(test) {
        const root = new lib_1.App();
        root.node.setContext('c1', 'root');
        root.node.setContext('c2', 'root');
        const child1 = new lib_1.Construct(root, 'child1');
        child1.node.setContext('c2', 'child1');
        child1.node.setContext('c3', 'child1');
        const child2 = new lib_1.Construct(root, 'child2');
        const child3 = new lib_1.Construct(child1, 'child1child1');
        child3.node.setContext('c1', 'child3');
        child3.node.setContext('c4', 'child3');
        test.equal(root.node.getContext('c1'), 'root');
        test.equal(root.node.getContext('c2'), 'root');
        test.equal(root.node.getContext('c3'), undefined);
        test.equal(child1.node.getContext('c1'), 'root');
        test.equal(child1.node.getContext('c2'), 'child1');
        test.equal(child1.node.getContext('c3'), 'child1');
        test.equal(child2.node.getContext('c1'), 'root');
        test.equal(child2.node.getContext('c2'), 'root');
        test.equal(child2.node.getContext('c3'), undefined);
        test.equal(child3.node.getContext('c1'), 'child3');
        test.equal(child3.node.getContext('c2'), 'child1');
        test.equal(child3.node.getContext('c3'), 'child1');
        test.equal(child3.node.getContext('c4'), 'child3');
        test.done();
    },
    'construct.setContext(key, value) can only be called before adding any children'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'child1');
        test.throws(() => root.node.setContext('k', 'v'));
        test.done();
    },
    'construct.pathParts returns an array of strings of all names from root to node'(test) {
        const tree = createTree();
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'Child1/Child11/Child111');
        test.deepEqual(tree.child2.node.path, 'Child2');
        test.done();
    },
    'if a root construct has a name, it should be included in the path'(test) {
        const tree = createTree({});
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'Child1/Child11/Child111');
        test.done();
    },
    'construct can not be created with the name of a sibling'(test) {
        const root = new lib_1.App();
        // WHEN
        new lib_1.Construct(root, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(root, 'SameName');
        }, /There is already a Construct with name 'SameName' in App/);
        // WHEN
        const c0 = new lib_1.Construct(root, 'c0');
        new lib_1.Construct(c0, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(c0, 'SameName');
        }, /There is already a Construct with name 'SameName' in Construct \[c0\]/);
        test.done();
    },
    'addMetadata(type, data) can be used to attach metadata to constructs FIND_ME'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        test.deepEqual(con.node.metadata, [], 'starts empty');
        con.node.addMetadata('key', 'value');
        con.node.addMetadata('number', 103);
        con.node.addMetadata('array', [123, 456]);
        test.deepEqual(con.node.metadata[0].type, 'key');
        test.deepEqual(con.node.metadata[0].data, 'value');
        test.deepEqual(con.node.metadata[1].data, 103);
        test.deepEqual(con.node.metadata[2].data, [123, 456]);
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace[0].indexOf('FIND_ME') !== -1, 'First stack line should include this function\s name');
        test.done();
    },
    'addMetadata(type, undefined/null) is ignored'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'Foo');
        con.node.addMetadata('Null', null);
        con.node.addMetadata('Undefined', undefined);
        con.node.addMetadata('True', true);
        con.node.addMetadata('False', false);
        con.node.addMetadata('Empty', '');
        const exists = (key) => con.node.metadata.find(x => x.type === key);
        test.ok(!exists('Null'));
        test.ok(!exists('Undefined'));
        test.ok(exists('True'));
        test.ok(exists('False'));
        test.ok(exists('Empty'));
        test.done();
    },
    'addWarning(message) can be used to add a "WARNING" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addWarning('This construct is deprecated, use the other one instead');
        test.deepEqual(con.node.metadata[0].type, cxapi.WARNING_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'This construct is deprecated, use the other one instead');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addError(message) can be used to add a "ERROR" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addError('Stop!');
        test.deepEqual(con.node.metadata[0].type, cxapi.ERROR_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Stop!');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addInfo(message) can be used to add an "INFO" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addInfo('Hey there, how do you do?');
        test.deepEqual(con.node.metadata[0].type, cxapi.INFO_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Hey there, how do you do?');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'multiple children of the same type, with explicit names are welcome'(test) {
        const root = new lib_1.App();
        new MyBeautifulConstruct(root, 'mbc1');
        new MyBeautifulConstruct(root, 'mbc2');
        new MyBeautifulConstruct(root, 'mbc3');
        new MyBeautifulConstruct(root, 'mbc4');
        test.equal(root.node.children.length, 4);
        test.done();
    },
    'construct.required(props, name) can be used to validate that required properties are defined'(test) {
        const root = new lib_1.App();
        // should be ok
        const c = new ConstructWithRequired(root, 'Construct', { requiredProp: 123, anotherRequiredProp: true });
        test.equal(c.requiredProp, 123);
        test.equal(c.anotherRequiredProp, true);
        // should throw
        test.throws(() => new ConstructWithRequired(root, 'C', { optionalProp: 'hello' }));
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.validate() can be implemented to perform validation, construct.validateTree() will return all errors from the subtree (DFS)'(test) {
        class MyConstruct extends lib_1.Construct {
            validate() {
                return ['my-error1', 'my-error2'];
            }
        }
        class YourConstruct extends lib_1.Construct {
            validate() {
                return ['your-error1'];
            }
        }
        class TheirConstruct extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new YourConstruct(this, 'YourConstruct');
            }
            validate() {
                return ['their-error'];
            }
        }
        class TestStack extends lib_1.App {
            constructor() {
                super();
                new MyConstruct(this, 'MyConstruct');
                new TheirConstruct(this, 'TheirConstruct');
            }
            validate() {
                return ['stack-error'];
            }
        }
        const stack = new TestStack();
        const errors = (stack.node.validateTree()).map(v => ({ path: v.source.node.path, message: v.message }));
        // validate DFS
        test.deepEqual(errors, [
            { path: 'MyConstruct', message: 'my-error1' },
            { path: 'MyConstruct', message: 'my-error2' },
            { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
            { path: 'TheirConstruct', message: 'their-error' },
            { path: '', message: 'stack-error' }
        ]);
        test.done();
    },
    'construct.lock() protects against adding children anywhere under this construct (direct or indirect)'(test) {
        class LockableConstruct extends lib_1.Construct {
            lockMe() {
                this.node.lock();
            }
            unlockMe() {
                this.node.unlock();
            }
        }
        const stack = new lib_1.App();
        const c0a = new LockableConstruct(stack, 'c0a');
        const c0b = new lib_1.Construct(stack, 'c0b');
        const c1a = new lib_1.Construct(c0a, 'c1a');
        const c1b = new lib_1.Construct(c0a, 'c1b');
        c0a.lockMe();
        // now we should still be able to add children to c0b, but not to c0a or any its children
        new lib_1.Construct(c0b, 'c1a');
        test.throws(() => new lib_1.Construct(c0a, 'fail1'), /Cannot add children to "c0a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1a, 'fail2'), /Cannot add children to "c0a\/c1a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1b, 'fail3'), /Cannot add children to "c0a\/c1b" during synthesis/);
        c0a.unlockMe();
        new lib_1.Construct(c0a, 'c0aZ');
        new lib_1.Construct(c1a, 'c1aZ');
        new lib_1.Construct(c1b, 'c1bZ');
        test.done();
    },
    'findAll returns a list of all children in either DFS or BFS'(test) {
        // GIVEN
        const c1 = new lib_1.Construct(undefined, '1');
        const c2 = new lib_1.Construct(c1, '2');
        new lib_1.Construct(c1, '3');
        new lib_1.Construct(c2, '4');
        new lib_1.Construct(c2, '5');
        // THEN
        test.deepEqual(c1.node.findAll().map(x => x.node.id), c1.node.findAll(lib_1.ConstructOrder.PreOrder).map(x => x.node.id)); // default is PreOrder
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PreOrder).map(x => x.node.id), ['1', '2', '4', '5', '3']);
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PostOrder).map(x => x.node.id), ['4', '5', '2', '3', '1']);
        test.done();
    },
    'ancestors returns a list of parents up to root'(test) {
        const { child1, child1_1_1 } = createTree();
        test.deepEqual(child1_1_1.node.ancestors().map(x => x.node.id), ['', 'Child1', 'Child11', 'Child111']);
        test.deepEqual(child1_1_1.node.ancestors(child1).map(x => x.node.id), ['Child11', 'Child111']);
        test.deepEqual(child1_1_1.node.ancestors(child1_1_1), []);
        test.done();
    },
    '"root" returns the root construct'(test) {
        const { child1, child2, child1_1_1, root } = createTree();
        test.ok(child1.node.root === root);
        test.ok(child2.node.root === root);
        test.ok(child1_1_1.node.root === root);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jb25zdHJ1Y3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmNvbnN0cnVjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEseUNBQTBDO0FBRTFDLGdDQUE2RjtBQWtlN0YsU0FBUyxVQUFVLENBQUMsT0FBYTtJQUMvQixNQUFNLElBQUksR0FBRyxJQUFJLFNBQUksRUFBRSxDQUFDO0lBQ3hCLElBQUksT0FBTyxFQUFFO1FBQ1gsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztLQUM5RTtJQUVELE1BQU0sTUFBTSxHQUFHLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztJQUM3QyxNQUFNLE1BQU0sR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDN0MsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFTLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ2xELE1BQU0sUUFBUSxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sRUFBRSxTQUFTLENBQUMsQ0FBQztJQUNsRCxNQUFNLFVBQVUsR0FBRyxJQUFJLGVBQVMsQ0FBQyxRQUFRLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDdkQsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFTLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBRWxELE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBRSxRQUFRO0tBQy9ELENBQUM7QUFDSixDQUFDO0FBRUQsTUFBTSxvQkFBcUIsU0FBUSxlQUFTO0lBQzFDLFlBQVksS0FBZ0IsRUFBRSxFQUFVO1FBQ3RDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDbkIsQ0FBQztDQUNGO0FBUUQsTUFBTSxxQkFBc0IsU0FBUSxlQUFTO0lBSTNDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7UUFDekUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQztRQUM5RCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLHFCQUFxQixDQUFDLENBQUM7SUFDOUUsQ0FBQztDQUNGO0FBcmdCRCxpQkFBUztJQUNQLHVGQUF1RixDQUFDLElBQVU7UUFDaEcsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLEVBQUUsRUFBRSwrREFBK0QsQ0FBQyxDQUFDO1FBQzlGLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUseUNBQXlDLENBQUMsQ0FBQyxDQUFDLGNBQWM7UUFDbkcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNFQUFzRSxDQUFDLElBQVU7UUFDL0UsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrREFBa0QsQ0FBQyxJQUFVO1FBQzNELE1BQU0sQ0FBQyxHQUFHLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRTFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4REFBOEQsQ0FBQyxJQUFVO1FBQ3ZFLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzdCLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM3QixJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDaEMsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3pCLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxXQUFXLENBQUUsQ0FBQztRQUNsQyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxDQUFFLENBQUM7UUFDbkMsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFlBQVksQ0FBRSxDQUFDO1FBQ25DLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUUsQ0FBQztRQUNqQyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFFLENBQUM7UUFDakMsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBRSxDQUFDO1FBQ2pDLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxXQUFXLENBQUUsQ0FBQztRQUNsQyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFFLENBQUM7UUFDakMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdGQUFnRixDQUFDLElBQVU7UUFDekYsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixNQUFNLENBQUMsR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxRkFBcUYsQ0FBQyxJQUFVO1FBQzlGLE1BQU0sQ0FBQyxHQUFHLElBQUksZUFBUyxDQUFDLFNBQWdCLEVBQUUsU0FBZ0IsQ0FBQyxDQUFDO1FBQzVELElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDOUIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBEQUEwRCxDQUFDLElBQVU7UUFDbkUsUUFBUTtRQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFdEMsY0FBYztRQUNkLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFNBQVMsS0FBSyxFQUFFLENBQUMsRUFBRSxtREFBbUQsQ0FBQyxDQUFDO1FBQzlHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0RUFBNEUsQ0FBQyxJQUFVO1FBQ3JGLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFFeEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLENBQUM7UUFDOUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFTLENBQUMsTUFBTSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3JELE1BQU0sRUFBRSxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sRUFBRSxjQUFjLENBQUMsQ0FBQztRQUNqRCxNQUFNLEVBQUUsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFFakQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxtREFBbUQsQ0FBQyxDQUFDO1FBQ2xGLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsc0NBQXNDLENBQUMsQ0FBQztRQUNyRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLG1EQUFtRCxDQUFDLENBQUM7UUFDdEYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSx3Q0FBd0MsQ0FBQyxDQUFDO1FBQzNFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnRUFBZ0UsQ0FBQyxJQUFVO1FBQ3pFLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsTUFBTSxDQUFDLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3pDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsZ0VBQWdFLENBQUMsQ0FBQztRQUNyRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZ0RBQWdELENBQUMsSUFBVTtRQUN6RCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDbkMsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFTLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzlDLE1BQU0sU0FBUyxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNyRCxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyREFBMkQsQ0FBQyxJQUFVO1FBQ3BFLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsd0RBQXdELENBQUMsQ0FBQztRQUNsRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0RBQXNELENBQUMsSUFBVTtRQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sVUFBVSxHQUFrQixFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxDQUFDO1FBQ3hFLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQztRQUMvQixLQUFLLENBQUMsU0FBUyxHQUFHLENBQUMsSUFBSSxFQUFFLEVBQUU7WUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDNUIsT0FBTyxRQUFRLENBQUM7UUFDbEIsQ0FBQyxDQUFDO1FBRUYsTUFBTSxTQUFTLEdBQUcsSUFBSSxlQUFTLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ2hFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxREFBcUQsQ0FBQyxJQUFVO1FBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxVQUFVLEdBQWtCLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsTUFBTSxFQUFFLENBQUM7UUFDeEUsTUFBTSxRQUFRLEdBQUcsYUFBYSxDQUFDO1FBQy9CLEtBQUssQ0FBQyxRQUFRLEdBQUcsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUN2QixJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUN6QixPQUFPLFVBQVUsQ0FBQztRQUNwQixDQUFDLENBQUM7UUFFRixNQUFNLFNBQVMsR0FBRyxJQUFJLGVBQVMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBEQUEwRCxDQUFDLElBQVU7UUFDbkUsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixNQUFNLEtBQUssR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDNUMsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzlCLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsMkJBQTJCLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUVBQXlFLENBQUMsSUFBVTtRQUNsRixNQUFNLElBQUksR0FBRyxJQUFJLFNBQUksRUFBRSxDQUFDO1FBQ3hCLE1BQU0sS0FBSyxHQUFHLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLGlFQUFpRSxDQUFDLENBQUM7UUFDbEksSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxFQUFFLDZEQUE2RCxDQUFDLENBQUM7UUFDNUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHdFQUF3RSxDQUFDLElBQVU7UUFDakYsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixNQUFNLEtBQUssR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLEtBQUssRUFBRSxnRUFBZ0UsQ0FBQyxDQUFDO1FBQzlILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDbEMsQ0FBQyxFQUFFLEVBQUUsRUFBRSw0REFBNEQsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrRUFBK0UsQ0FBQyxJQUFVO1FBQ3hGLE1BQU0sQ0FBQyxHQUFHLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFLEVBQUUscUNBQXFDLENBQUMsQ0FBQztRQUMzRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxFQUFFLDBKQUEwSixDQUFDLENBQUM7UUFDbk0sSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhGQUE4RixDQUFDLElBQVU7UUFDdkcsTUFBTSxPQUFPLEdBQUc7WUFDZCxJQUFJLEVBQUUsRUFBRTtZQUNSLElBQUksRUFBRSxPQUFPO1NBQ2QsQ0FBQztRQUVGLE1BQU0sQ0FBQyxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUM5QixJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUMxRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUlBQXlJLENBQUMsSUFBVTtRQUNsSixNQUFNLElBQUksR0FBRyxJQUFJLFNBQUksRUFBRSxDQUFDO1FBQ3hCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFbkMsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN2QyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFdkMsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sTUFBTSxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sRUFBRSxjQUFjLENBQUMsQ0FBQztRQUNyRCxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdkMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXZDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRW5ELElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRXBELElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFbkQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdGQUFnRixDQUFDLElBQVU7UUFDekYsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZ0ZBQWdGLENBQUMsSUFBVTtRQUN6RixNQUFNLElBQUksR0FBRyxVQUFVLEVBQUUsQ0FBQztRQUMxQixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1FBQzVFLE1BQU0sSUFBSSxHQUFHLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM1QixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5REFBeUQsQ0FBQyxJQUFVO1FBQ2xFLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFFeEIsT0FBTztRQUNQLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUVoQyxrQ0FBa0M7UUFDbEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDbEMsQ0FBQyxFQUFFLDBEQUEwRCxDQUFDLENBQUM7UUFFL0QsT0FBTztRQUNQLE1BQU0sRUFBRSxHQUFHLElBQUksZUFBUyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNyQyxJQUFJLGVBQVMsQ0FBQyxFQUFFLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFOUIsa0NBQWtDO1FBQ2xDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsSUFBSSxlQUFTLENBQUMsRUFBRSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ2hDLENBQUMsRUFBRSx1RUFBdUUsQ0FBQyxDQUFDO1FBRTVFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4RUFBOEUsQ0FBQyxJQUFVO1FBQ3ZGLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsTUFBTSxHQUFHLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFBRSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRXRELEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNyQyxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDcEMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLENBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxDQUFDLENBQUM7UUFFNUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLHNEQUFzRCxDQUFDLENBQUM7UUFDdkosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhDQUE4QyxDQUFDLElBQVU7UUFDdkQsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixNQUFNLEdBQUcsR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDdkMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ25DLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM3QyxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDbkMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3JDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVsQyxNQUFNLE1BQU0sR0FBRyxDQUFDLEdBQVcsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxHQUFHLENBQUMsQ0FBQztRQUU1RSxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRkFBbUYsQ0FBQyxJQUFVO1FBQzVGLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsTUFBTSxHQUFHLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHlEQUF5RCxDQUFDLENBQUM7UUFDL0UsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDdEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUseURBQXlELENBQUMsQ0FBQztRQUNyRyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQzdFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrRUFBK0UsQ0FBQyxJQUFVO1FBQ3hGLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFDeEIsTUFBTSxHQUFHLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzNCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDN0UsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhFQUE4RSxDQUFDLElBQVU7UUFDdkYsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixNQUFNLEdBQUcsR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDL0MsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUNuRSxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDN0UsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFFQUFxRSxDQUFDLElBQVU7UUFDOUUsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFJLEVBQUUsQ0FBQztRQUN4QixJQUFJLG9CQUFvQixDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2QyxJQUFJLG9CQUFvQixDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2QyxJQUFJLG9CQUFvQixDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2QyxJQUFJLG9CQUFvQixDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOEZBQThGLENBQUMsSUFBVTtRQUN2RyxNQUFNLElBQUksR0FBRyxJQUFJLFNBQUksRUFBRSxDQUFDO1FBRXhCLGVBQWU7UUFDZixNQUFNLENBQUMsR0FBRyxJQUFJLHFCQUFxQixDQUFDLElBQUksRUFBRSxXQUFXLEVBQUUsRUFBRSxZQUFZLEVBQUUsR0FBRyxFQUFFLG1CQUFtQixFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDekcsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsWUFBWSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLG1CQUFtQixFQUFFLElBQUksQ0FBQyxDQUFDO1FBRXhDLGVBQWU7UUFDZixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUkscUJBQXFCLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQVMsQ0FBQyxDQUFDLENBQUM7UUFDMUYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJDQUEyQztJQUMzQyx1SUFBdUksQ0FBQyxJQUFVO1FBRWhKLE1BQU0sV0FBWSxTQUFRLGVBQVM7WUFDdkIsUUFBUTtnQkFDaEIsT0FBTyxDQUFFLFdBQVcsRUFBRSxXQUFXLENBQUUsQ0FBQztZQUN0QyxDQUFDO1NBQ0Y7UUFFRCxNQUFNLGFBQWMsU0FBUSxlQUFTO1lBQ3pCLFFBQVE7Z0JBQ2hCLE9BQU8sQ0FBRSxhQUFhLENBQUUsQ0FBQztZQUMzQixDQUFDO1NBQ0Y7UUFFRCxNQUFNLGNBQWUsU0FBUSxlQUFTO1lBQ3BDLFlBQVksS0FBZ0IsRUFBRSxFQUFVO2dCQUN0QyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUVqQixJQUFJLGFBQWEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDM0MsQ0FBQztZQUVTLFFBQVE7Z0JBQ2hCLE9BQU8sQ0FBRSxhQUFhLENBQUUsQ0FBQztZQUMzQixDQUFDO1NBQ0Y7UUFFRCxNQUFNLFNBQVUsU0FBUSxTQUFJO1lBQzFCO2dCQUNFLEtBQUssRUFBRSxDQUFDO2dCQUVSLElBQUksV0FBVyxDQUFDLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQztnQkFDckMsSUFBSSxjQUFjLENBQUMsSUFBSSxFQUFFLGdCQUFnQixDQUFDLENBQUM7WUFDN0MsQ0FBQztZQUVTLFFBQVE7Z0JBQ2hCLE9BQVEsQ0FBRSxhQUFhLENBQUUsQ0FBQztZQUM1QixDQUFDO1NBQ0Y7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLFNBQVMsRUFBRSxDQUFDO1FBRTlCLE1BQU0sTUFBTSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXhHLGVBQWU7UUFDZixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRTtZQUNyQixFQUFFLElBQUksRUFBRSxhQUFhLEVBQUUsT0FBTyxFQUFFLFdBQVcsRUFBRTtZQUM3QyxFQUFFLElBQUksRUFBRSxhQUFhLEVBQUUsT0FBTyxFQUFFLFdBQVcsRUFBRTtZQUM3QyxFQUFFLElBQUksRUFBRSw4QkFBOEIsRUFBRSxPQUFPLEVBQUUsYUFBYSxFQUFFO1lBQ2hFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLE9BQU8sRUFBRSxhQUFhLEVBQUU7WUFDbEQsRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxhQUFhLEVBQUU7U0FDckMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNHQUFzRyxDQUFDLElBQVU7UUFFL0csTUFBTSxpQkFBa0IsU0FBUSxlQUFTO1lBQ2hDLE1BQU07Z0JBQ1gsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNuQixDQUFDO1lBRU0sUUFBUTtnQkFDYixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ3JCLENBQUM7U0FDRjtRQUVELE1BQU0sS0FBSyxHQUFHLElBQUksU0FBSSxFQUFFLENBQUM7UUFFekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDaEQsTUFBTSxHQUFHLEdBQUcsSUFBSSxlQUFTLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXhDLE1BQU0sR0FBRyxHQUFHLElBQUksZUFBUyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN0QyxNQUFNLEdBQUcsR0FBRyxJQUFJLGVBQVMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFdEMsR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBRWIseUZBQXlGO1FBQ3pGLElBQUksZUFBUyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUMxQixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksZUFBUyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsRUFBRSwrQ0FBK0MsQ0FBQyxDQUFDO1FBQ2hHLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxlQUFTLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxFQUFFLG9EQUFvRCxDQUFDLENBQUM7UUFDckcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLGVBQVMsQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLEVBQUUsb0RBQW9ELENBQUMsQ0FBQztRQUVyRyxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7UUFFZixJQUFJLGVBQVMsQ0FBQyxHQUFHLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDM0IsSUFBSSxlQUFTLENBQUMsR0FBRyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQzNCLElBQUksZUFBUyxDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUzQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkRBQTZELENBQUMsSUFBVTtRQUN0RSxRQUFRO1FBQ1IsTUFBTSxFQUFFLEdBQUcsSUFBSSxlQUFTLENBQUMsU0FBZ0IsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUNoRCxNQUFNLEVBQUUsR0FBRyxJQUFJLGVBQVMsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDbEMsSUFBSSxlQUFTLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3ZCLElBQUksZUFBUyxDQUFDLEVBQUUsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN2QixJQUFJLGVBQVMsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFFdkIsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLG9CQUFjLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsc0JBQXNCO1FBQzNJLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsb0JBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxDQUFDLENBQUM7UUFDMUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxvQkFBYyxDQUFDLFNBQVMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLENBQUMsQ0FBQztRQUMzRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZ0RBQWdELENBQUMsSUFBVTtRQUN6RCxNQUFNLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxHQUFHLFVBQVUsRUFBRSxDQUFDO1FBRTVDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsVUFBVSxDQUFFLENBQUMsQ0FBQztRQUN6RyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBRSxTQUFTLEVBQUUsVUFBVSxDQUFFLENBQUMsQ0FBQztRQUNqRyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxFQUFFLEVBQUcsQ0FBQyxDQUFDO1FBQzNELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLE1BQU0sRUFBRSxNQUFNLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsR0FBRyxVQUFVLEVBQUUsQ0FBQztRQUMxRCxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQyxDQUFDO1FBQ25DLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxJQUFJLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBjeGFwaSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2N4LWFwaScpO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCB7IEFwcCBhcyBSb290LCBBcm5Db21wb25lbnRzLCBDb25zdHJ1Y3QsIENvbnN0cnVjdE9yZGVyLCBTdGFjaywgVG9rZW4gfSBmcm9tICcuLi9saWInO1xuXG4vLyB0c2xpbnQ6ZGlzYWJsZTp2YXJpYWJsZS1uYW1lXG4vLyB0c2xpbnQ6ZGlzYWJsZTptYXgtbGluZS1sZW5ndGhcblxuZXhwb3J0ID0ge1xuICAndGhlIFwiUm9vdFwiIGNvbnN0cnVjdCBpcyBhIHNwZWNpYWwgY29uc3RydWN0IHdoaWNoIGNhbiBiZSB1c2VkIGFzIHRoZSByb290IG9mIHRoZSB0cmVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG4gICAgdGVzdC5lcXVhbChyb290Lm5vZGUuaWQsICcnLCAnaWYgbm90IHNwZWNpZmllZCwgbmFtZSBvZiBhIHJvb3QgY29uc3RydWN0IGlzIGFuIGVtcHR5IHN0cmluZycpO1xuICAgIHRlc3Qub2soIXJvb3Qubm9kZS5zY29wZSwgJ25vIHBhcmVudCcpO1xuICAgIHRlc3QuZXF1YWwocm9vdC5ub2RlLmNoaWxkcmVuLmxlbmd0aCwgMCwgJ2EgY29uc3RydWN0IGlzIGNyZWF0ZWQgd2l0aG91dCBjaGlsZHJlbicpOyAvLyBubyBjaGlsZHJlblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3RzIGNhbm5vdCBiZSBjcmVhdGVkIHdpdGggYW4gZW1wdHkgbmFtZSB1bmxlc3MgdGhleSBhcmUgcm9vdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJycpKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0Lm5hbWUgcmV0dXJucyB0aGUgbmFtZSBvZiB0aGUgY29uc3RydWN0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgdCA9IGNyZWF0ZVRyZWUoKTtcblxuICAgIHRlc3QuZXF1YWwodC5jaGlsZDEubm9kZS5pZCwgJ0NoaWxkMScpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDIubm9kZS5pZCwgJ0NoaWxkMicpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDFfMS5ub2RlLmlkLCAnQ2hpbGQxMScpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDFfMi5ub2RlLmlkLCAnQ2hpbGQxMicpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDFfMV8xLm5vZGUuaWQsICdDaGlsZDExMScpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDJfMS5ub2RlLmlkLCAnQ2hpbGQyMScpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvbnN0cnVjdCBpZCBjYW4gdXNlIGFueSBjaGFyYWN0ZXIgZXhjZXB0IHRoZSBwYXRoIHNlcGFyYXRvcicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ3ZhbGlkJyk7XG4gICAgbmV3IENvbnN0cnVjdChyb290LCAnVmFsaUQnKTtcbiAgICBuZXcgQ29uc3RydWN0KHJvb3QsICdWYTEyM2xpZCcpO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ3YnKTtcbiAgICBuZXcgQ29uc3RydWN0KHJvb3QsICcgIGludmFsaWQnICk7XG4gICAgbmV3IENvbnN0cnVjdChyb290LCAnaW52YWxpZCAgICcgKTtcbiAgICBuZXcgQ29uc3RydWN0KHJvb3QsICcxMjNpbnZhbGlkJyApO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2luIHZhbGlkJyApO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2luX1ZhbGlkJyApO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2luLVZhbGlkJyApO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2luXFxcXFZhbGlkJyApO1xuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2luLlZhbGlkJyApO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpZiBjb25zdHJ1Y3QgaWQgY29udGFpbnMgcGF0aCBzZXBlcmF0b3JzLCB0aGV5IHdpbGwgYmUgcmVwbGFjZWQgYnkgZG91YmxlLWRhc2gnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFJvb3QoKTtcbiAgICBjb25zdCBjID0gbmV3IENvbnN0cnVjdChyb290LCAnQm9vbS9Cb29tL0JhbScpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGMubm9kZS5pZCwgJ0Jvb20tLUJvb20tLUJhbScpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpZiBcInVuZGVmaW5lZFwiIGlzIGZvcmNlZnVsbHkgdXNlZCBhcyBhbiBcImlkXCIsIGl0IHdpbGwgYmUgdHJlYXRlZCBhcyBhbiBlbXB0eSBzdHJpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBjID0gbmV3IENvbnN0cnVjdCh1bmRlZmluZWQgYXMgYW55LCB1bmRlZmluZWQgYXMgYW55KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjLm5vZGUuaWQsICcnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICBcImRvbnQgYWxsb3cgdW5yZXNvbHZlZCB0b2tlbnMgdG8gYmUgdXNlZCBpbiBjb25zdHJ1Y3QgSURzXCIodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG4gICAgY29uc3QgdG9rZW4gPSBuZXcgVG9rZW4oKCkgPT4gJ2xhenknKTtcblxuICAgIC8vIFdIRU4gKyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IENvbnN0cnVjdChyb290LCBgTXlJRDogJHt0b2tlbn1gKSwgL0Nhbm5vdCB1c2UgdG9rZW5zIGluIGNvbnN0cnVjdCBJRDogTXlJRDogXFwke1Rva2VuLyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvbnN0cnVjdC51bmlxdWVJZCByZXR1cm5zIGEgdHJlZS11bmlxdWUgYWxwaGFudW1lcmljIGlkIG9mIHRoaXMgY29uc3RydWN0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG5cbiAgICBjb25zdCBjaGlsZDEgPSBuZXcgQ29uc3RydWN0KHJvb3QsICdUaGlzIGlzIHRoZSBmaXJzdCBjaGlsZCcpO1xuICAgIGNvbnN0IGNoaWxkMiA9IG5ldyBDb25zdHJ1Y3QoY2hpbGQxLCAnU2Vjb25kIGxldmVsJyk7XG4gICAgY29uc3QgYzEgPSBuZXcgQ29uc3RydWN0KGNoaWxkMiwgJ015IGNvbnN0cnVjdCcpO1xuICAgIGNvbnN0IGMyID0gbmV3IENvbnN0cnVjdChjaGlsZDEsICdNeSBjb25zdHJ1Y3QnKTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKGMxLm5vZGUucGF0aCwgJ1RoaXMgaXMgdGhlIGZpcnN0IGNoaWxkL1NlY29uZCBsZXZlbC9NeSBjb25zdHJ1Y3QnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjMi5ub2RlLnBhdGgsICdUaGlzIGlzIHRoZSBmaXJzdCBjaGlsZC9NeSBjb25zdHJ1Y3QnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjMS5ub2RlLnVuaXF1ZUlkLCAnVGhpc2lzdGhlZmlyc3RjaGlsZFNlY29uZGxldmVsTXljb25zdHJ1Y3QyMDIxMzFFMCcpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGMyLm5vZGUudW5pcXVlSWQsICdUaGlzaXN0aGVmaXJzdGNoaWxkTXljb25zdHJ1Y3Q4QzI4OERGOScpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW5ub3QgY2FsY3VsYXRlIHVuaXF1ZUlkIGlmIHRoZSBjb25zdHJ1Y3QgcGF0aCBpcyBbXCJEZWZhdWx0XCJdJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG4gICAgY29uc3QgYyA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ0RlZmF1bHQnKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBjLm5vZGUudW5pcXVlSWQsIC9VbmFibGUgdG8gY2FsY3VsYXRlIGEgdW5pcXVlIGlkIGZvciBhbiBlbXB0eSBzZXQgb2YgY29tcG9uZW50cy8pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3Qubm9kZS5zdGFjayByZXR1cm5zIHRoZSBjb3JyZWN0IHN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICB0ZXN0LnNhbWUoc3RhY2subm9kZS5zdGFjaywgc3RhY2spO1xuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBDb25zdHJ1Y3Qoc3RhY2ssICdQYXJlbnQnKTtcbiAgICBjb25zdCBjb25zdHJ1Y3QgPSBuZXcgQ29uc3RydWN0KHBhcmVudCwgJ0NvbnN0cnVjdCcpO1xuICAgIHRlc3Quc2FtZShjb25zdHJ1Y3Qubm9kZS5zdGFjaywgc3RhY2spO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3Qubm9kZS5zdGFjayB0aHJvd3Mgd2hlbiB0aGVyZSBpcyBubyBwYXJlbnQgU3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFJvb3QoKTtcbiAgICBjb25zdCBjb25zdHJ1Y3QgPSBuZXcgQ29uc3RydWN0KHJvb3QsICdDb25zdHJ1Y3QnKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBjb25zdHJ1Y3Qubm9kZS5zdGFjaywgL05vIHN0YWNrIGNvdWxkIGJlIGlkZW50aWZpZWQgZm9yIHRoZSBjb25zdHJ1Y3QgYXQgcGF0aC8pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3Qubm9kZS5zdGFjay5mb3JtYXRBcm4gZm9yd2FyZHMgdG8gdGhlIFN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBjb21wb25lbnRzOiBBcm5Db21wb25lbnRzID0geyBzZXJ2aWNlOiAndGVzdCcsIHJlc291cmNlOiAndGVzdCcgfTtcbiAgICBjb25zdCBkdW1teUFybiA9ICdhcm46OjpkdW1teSc7XG4gICAgc3RhY2suZm9ybWF0QXJuID0gKGFyZ3MpID0+IHtcbiAgICAgIHRlc3Quc2FtZShhcmdzLCBjb21wb25lbnRzKTtcbiAgICAgIHJldHVybiBkdW1teUFybjtcbiAgICB9O1xuXG4gICAgY29uc3QgY29uc3RydWN0ID0gbmV3IENvbnN0cnVjdChzdGFjaywgJ0NvbnN0cnVjdCcpO1xuICAgIHRlc3Quc2FtZShjb25zdHJ1Y3Qubm9kZS5zdGFjay5mb3JtYXRBcm4oY29tcG9uZW50cyksIGR1bW15QXJuKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0Lm5vZGUuc3RhY2sucGFyc2VBcm4gZm9yd2FyZHMgdG8gdGhlIFN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBjb21wb25lbnRzOiBBcm5Db21wb25lbnRzID0geyBzZXJ2aWNlOiAndGVzdCcsIHJlc291cmNlOiAndGVzdCcgfTtcbiAgICBjb25zdCBkdW1teUFybiA9ICdhcm46OjpkdW1teSc7XG4gICAgc3RhY2sucGFyc2VBcm4gPSAoYXJuKSA9PiB7XG4gICAgICB0ZXN0LnNhbWUoYXJuLCBkdW1teUFybik7XG4gICAgICByZXR1cm4gY29tcG9uZW50cztcbiAgICB9O1xuXG4gICAgY29uc3QgY29uc3RydWN0ID0gbmV3IENvbnN0cnVjdChzdGFjaywgJ0NvbnN0cnVjdCcpO1xuICAgIHRlc3Quc2FtZShjb25zdHJ1Y3Qubm9kZS5zdGFjay5wYXJzZUFybihkdW1teUFybiksIGNvbXBvbmVudHMpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3QuZ2V0Q2hpbGRyZW4oKSByZXR1cm5zIGFuIGFycmF5IG9mIGFsbCBjaGlsZHJlbicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIGNvbnN0IGNoaWxkID0gbmV3IENvbnN0cnVjdChyb290LCAnQ2hpbGQxJyk7XG4gICAgbmV3IENvbnN0cnVjdChyb290LCAnQ2hpbGQyJyk7XG4gICAgdGVzdC5lcXVhbChjaGlsZC5ub2RlLmNoaWxkcmVuLmxlbmd0aCwgMCwgJ25vIGNoaWxkcmVuJyk7XG4gICAgdGVzdC5lcXVhbChyb290Lm5vZGUuY2hpbGRyZW4ubGVuZ3RoLCAyLCAndHdvIGNoaWxkcmVuIGFyZSBleHBlY3RlZCcpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3QuZmluZENoaWxkKG5hbWUpIGNhbiBiZSB1c2VkIHRvIHJldHJpZXZlIGEgY2hpbGQgZnJvbSBhIHBhcmVudCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIGNvbnN0IGNoaWxkID0gbmV3IENvbnN0cnVjdChyb290LCAnQ29udHJ1Y3QnKTtcbiAgICB0ZXN0LnN0cmljdEVxdWFsKHJvb3Qubm9kZS50cnlGaW5kQ2hpbGQoY2hpbGQubm9kZS5pZCksIGNoaWxkLCAnZmluZENoaWxkKG5hbWUpIGNhbiBiZSB1c2VkIHRvIHJldHJpZXZlIHRoZSBjaGlsZCBmcm9tIGEgcGFyZW50Jyk7XG4gICAgdGVzdC5vayghcm9vdC5ub2RlLnRyeUZpbmRDaGlsZCgnTm90Rm91bmQnKSwgJ2ZpbmRDaGlsZChuYW1lKSByZXR1cm5zIHVuZGVmaW5lZCBpZiB0aGUgY2hpbGQgaXMgbm90IGZvdW5kJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvbnN0cnVjdC5nZXRDaGlsZChuYW1lKSBjYW4gYmUgdXNlZCB0byByZXRyaWV2ZSBhIGNoaWxkIGZyb20gYSBwYXJlbnQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFJvb3QoKTtcbiAgICBjb25zdCBjaGlsZCA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ0NvbnRydWN0Jyk7XG4gICAgdGVzdC5zdHJpY3RFcXVhbChyb290Lm5vZGUuZmluZENoaWxkKGNoaWxkLm5vZGUuaWQpLCBjaGlsZCwgJ2dldENoaWxkKG5hbWUpIGNhbiBiZSB1c2VkIHRvIHJldHJpZXZlIHRoZSBjaGlsZCBmcm9tIGEgcGFyZW50Jyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgcm9vdC5ub2RlLmZpbmRDaGlsZCgnTm90Rm91bmQnKTtcbiAgICB9LCAnJywgJ2dldENoaWxkKG5hbWUpIHJldHVybnMgdW5kZWZpbmVkIGlmIHRoZSBjaGlsZCBpcyBub3QgZm91bmQnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0LnRvU3RyaW5nKCkgYW5kIGNvbnN0cnVjdC50b1RyZWVTdHJpbmcoKSBjYW4gYmUgdXNlZCBmb3IgZGlhZ25vc3RpY3MnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCB0ID0gY3JlYXRlVHJlZSgpO1xuXG4gICAgdGVzdC5lcXVhbCh0LnJvb3QudG9TdHJpbmcoKSwgJ0FwcCcpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDFfMV8xLnRvU3RyaW5nKCksICdDb25zdHJ1Y3QgW0NoaWxkMS9DaGlsZDExL0NoaWxkMTExXScpO1xuICAgIHRlc3QuZXF1YWwodC5jaGlsZDIudG9TdHJpbmcoKSwgJ0NvbnN0cnVjdCBbQ2hpbGQyXScpO1xuICAgIHRlc3QuZXF1YWwodC5yb290Lm5vZGUudG9UcmVlU3RyaW5nKCksICdBcHBcXG4gIENvbnN0cnVjdCBbQ2hpbGQxXVxcbiAgICBDb25zdHJ1Y3QgW0NoaWxkMTFdXFxuICAgICAgQ29uc3RydWN0IFtDaGlsZDExMV1cXG4gICAgQ29uc3RydWN0IFtDaGlsZDEyXVxcbiAgQ29uc3RydWN0IFtDaGlsZDJdXFxuICAgIENvbnN0cnVjdCBbQ2hpbGQyMV1cXG4nKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0LmdldENvbnRleHQoa2V5KSBjYW4gYmUgdXNlZCB0byByZWFkIGEgdmFsdWUgZnJvbSBjb250ZXh0IGRlZmluZWQgYXQgdGhlIHJvb3QgbGV2ZWwnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBjb250ZXh0ID0ge1xuICAgICAgY3R4MTogMTIsXG4gICAgICBjdHgyOiAnaGVsbG8nXG4gICAgfTtcblxuICAgIGNvbnN0IHQgPSBjcmVhdGVUcmVlKGNvbnRleHQpO1xuICAgIHRlc3QuZXF1YWwodC5yb290Lm5vZGUuZ2V0Q29udGV4dCgnY3R4MScpLCAxMik7XG4gICAgdGVzdC5lcXVhbCh0LmNoaWxkMV8xXzEubm9kZS5nZXRDb250ZXh0KCdjdHgyJyksICdoZWxsbycpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb25zdHJ1Y3Quc2V0Q29udGV4dChrLHYpIHNldHMgY29udGV4dCBhdCBzb21lIGxldmVsIGFuZCBjb25zdHJ1Y3QuZ2V0Q29udGV4dChrZXkpIHdpbGwgcmV0dXJuIHRoZSBsb3dlcm1vc3QgdmFsdWUgZGVmaW5lZCBpbiB0aGUgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFJvb3QoKTtcbiAgICByb290Lm5vZGUuc2V0Q29udGV4dCgnYzEnLCAncm9vdCcpO1xuICAgIHJvb3Qubm9kZS5zZXRDb250ZXh0KCdjMicsICdyb290Jyk7XG5cbiAgICBjb25zdCBjaGlsZDEgPSBuZXcgQ29uc3RydWN0KHJvb3QsICdjaGlsZDEnKTtcbiAgICBjaGlsZDEubm9kZS5zZXRDb250ZXh0KCdjMicsICdjaGlsZDEnKTtcbiAgICBjaGlsZDEubm9kZS5zZXRDb250ZXh0KCdjMycsICdjaGlsZDEnKTtcblxuICAgIGNvbnN0IGNoaWxkMiA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2NoaWxkMicpO1xuICAgIGNvbnN0IGNoaWxkMyA9IG5ldyBDb25zdHJ1Y3QoY2hpbGQxLCAnY2hpbGQxY2hpbGQxJyk7XG4gICAgY2hpbGQzLm5vZGUuc2V0Q29udGV4dCgnYzEnLCAnY2hpbGQzJyk7XG4gICAgY2hpbGQzLm5vZGUuc2V0Q29udGV4dCgnYzQnLCAnY2hpbGQzJyk7XG5cbiAgICB0ZXN0LmVxdWFsKHJvb3Qubm9kZS5nZXRDb250ZXh0KCdjMScpLCAncm9vdCcpO1xuICAgIHRlc3QuZXF1YWwocm9vdC5ub2RlLmdldENvbnRleHQoJ2MyJyksICdyb290Jyk7XG4gICAgdGVzdC5lcXVhbChyb290Lm5vZGUuZ2V0Q29udGV4dCgnYzMnKSwgdW5kZWZpbmVkKTtcblxuICAgIHRlc3QuZXF1YWwoY2hpbGQxLm5vZGUuZ2V0Q29udGV4dCgnYzEnKSwgJ3Jvb3QnKTtcbiAgICB0ZXN0LmVxdWFsKGNoaWxkMS5ub2RlLmdldENvbnRleHQoJ2MyJyksICdjaGlsZDEnKTtcbiAgICB0ZXN0LmVxdWFsKGNoaWxkMS5ub2RlLmdldENvbnRleHQoJ2MzJyksICdjaGlsZDEnKTtcblxuICAgIHRlc3QuZXF1YWwoY2hpbGQyLm5vZGUuZ2V0Q29udGV4dCgnYzEnKSwgJ3Jvb3QnKTtcbiAgICB0ZXN0LmVxdWFsKGNoaWxkMi5ub2RlLmdldENvbnRleHQoJ2MyJyksICdyb290Jyk7XG4gICAgdGVzdC5lcXVhbChjaGlsZDIubm9kZS5nZXRDb250ZXh0KCdjMycpLCB1bmRlZmluZWQpO1xuXG4gICAgdGVzdC5lcXVhbChjaGlsZDMubm9kZS5nZXRDb250ZXh0KCdjMScpLCAnY2hpbGQzJyk7XG4gICAgdGVzdC5lcXVhbChjaGlsZDMubm9kZS5nZXRDb250ZXh0KCdjMicpLCAnY2hpbGQxJyk7XG4gICAgdGVzdC5lcXVhbChjaGlsZDMubm9kZS5nZXRDb250ZXh0KCdjMycpLCAnY2hpbGQxJyk7XG4gICAgdGVzdC5lcXVhbChjaGlsZDMubm9kZS5nZXRDb250ZXh0KCdjNCcpLCAnY2hpbGQzJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0LnNldENvbnRleHQoa2V5LCB2YWx1ZSkgY2FuIG9ubHkgYmUgY2FsbGVkIGJlZm9yZSBhZGRpbmcgYW55IGNoaWxkcmVuJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG4gICAgbmV3IENvbnN0cnVjdChyb290LCAnY2hpbGQxJyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gcm9vdC5ub2RlLnNldENvbnRleHQoJ2snLCAndicpKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0LnBhdGhQYXJ0cyByZXR1cm5zIGFuIGFycmF5IG9mIHN0cmluZ3Mgb2YgYWxsIG5hbWVzIGZyb20gcm9vdCB0byBub2RlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgdHJlZSA9IGNyZWF0ZVRyZWUoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbCh0cmVlLnJvb3Qubm9kZS5wYXRoLCAnJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwodHJlZS5jaGlsZDFfMV8xLm5vZGUucGF0aCwgJ0NoaWxkMS9DaGlsZDExL0NoaWxkMTExJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwodHJlZS5jaGlsZDIubm9kZS5wYXRoLCAnQ2hpbGQyJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2lmIGEgcm9vdCBjb25zdHJ1Y3QgaGFzIGEgbmFtZSwgaXQgc2hvdWxkIGJlIGluY2x1ZGVkIGluIHRoZSBwYXRoJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgdHJlZSA9IGNyZWF0ZVRyZWUoe30pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHRyZWUucm9vdC5ub2RlLnBhdGgsICcnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbCh0cmVlLmNoaWxkMV8xXzEubm9kZS5wYXRoLCAnQ2hpbGQxL0NoaWxkMTEvQ2hpbGQxMTEnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0IGNhbiBub3QgYmUgY3JlYXRlZCB3aXRoIHRoZSBuYW1lIG9mIGEgc2libGluZycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ1NhbWVOYW1lJyk7XG5cbiAgICAvLyBUSEVOOiBUaGV5IGhhdmUgZGlmZmVyZW50IHBhdGhzXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgbmV3IENvbnN0cnVjdChyb290LCAnU2FtZU5hbWUnKTtcbiAgICB9LCAvVGhlcmUgaXMgYWxyZWFkeSBhIENvbnN0cnVjdCB3aXRoIG5hbWUgJ1NhbWVOYW1lJyBpbiBBcHAvKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBjMCA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ2MwJyk7XG4gICAgbmV3IENvbnN0cnVjdChjMCwgJ1NhbWVOYW1lJyk7XG5cbiAgICAvLyBUSEVOOiBUaGV5IGhhdmUgZGlmZmVyZW50IHBhdGhzXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgbmV3IENvbnN0cnVjdChjMCwgJ1NhbWVOYW1lJyk7XG4gICAgfSwgL1RoZXJlIGlzIGFscmVhZHkgYSBDb25zdHJ1Y3Qgd2l0aCBuYW1lICdTYW1lTmFtZScgaW4gQ29uc3RydWN0IFxcW2MwXFxdLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkTWV0YWRhdGEodHlwZSwgZGF0YSkgY2FuIGJlIHVzZWQgdG8gYXR0YWNoIG1ldGFkYXRhIHRvIGNvbnN0cnVjdHMgRklORF9NRScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIGNvbnN0IGNvbiA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ015Q29uc3RydWN0Jyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoY29uLm5vZGUubWV0YWRhdGEsIFtdLCAnc3RhcnRzIGVtcHR5Jyk7XG5cbiAgICBjb24ubm9kZS5hZGRNZXRhZGF0YSgna2V5JywgJ3ZhbHVlJyk7XG4gICAgY29uLm5vZGUuYWRkTWV0YWRhdGEoJ251bWJlcicsIDEwMyk7XG4gICAgY29uLm5vZGUuYWRkTWV0YWRhdGEoJ2FycmF5JywgWyAxMjMsIDQ1NiBdKTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKGNvbi5ub2RlLm1ldGFkYXRhWzBdLnR5cGUsICdrZXknKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjb24ubm9kZS5tZXRhZGF0YVswXS5kYXRhLCAndmFsdWUnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjb24ubm9kZS5tZXRhZGF0YVsxXS5kYXRhLCAxMDMpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGNvbi5ub2RlLm1ldGFkYXRhWzJdLmRhdGEsIFsgMTIzLCA0NTYgXSk7XG4gICAgdGVzdC5vayhjb24ubm9kZS5tZXRhZGF0YVswXS50cmFjZSAmJiBjb24ubm9kZS5tZXRhZGF0YVswXS50cmFjZVswXS5pbmRleE9mKCdGSU5EX01FJykgIT09IC0xLCAnRmlyc3Qgc3RhY2sgbGluZSBzaG91bGQgaW5jbHVkZSB0aGlzIGZ1bmN0aW9uXFxzIG5hbWUnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkTWV0YWRhdGEodHlwZSwgdW5kZWZpbmVkL251bGwpIGlzIGlnbm9yZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFJvb3QoKTtcbiAgICBjb25zdCBjb24gPSBuZXcgQ29uc3RydWN0KHJvb3QsICdGb28nKTtcbiAgICBjb24ubm9kZS5hZGRNZXRhZGF0YSgnTnVsbCcsIG51bGwpO1xuICAgIGNvbi5ub2RlLmFkZE1ldGFkYXRhKCdVbmRlZmluZWQnLCB1bmRlZmluZWQpO1xuICAgIGNvbi5ub2RlLmFkZE1ldGFkYXRhKCdUcnVlJywgdHJ1ZSk7XG4gICAgY29uLm5vZGUuYWRkTWV0YWRhdGEoJ0ZhbHNlJywgZmFsc2UpO1xuICAgIGNvbi5ub2RlLmFkZE1ldGFkYXRhKCdFbXB0eScsICcnKTtcblxuICAgIGNvbnN0IGV4aXN0cyA9IChrZXk6IHN0cmluZykgPT4gY29uLm5vZGUubWV0YWRhdGEuZmluZCh4ID0+IHgudHlwZSA9PT0ga2V5KTtcblxuICAgIHRlc3Qub2soIWV4aXN0cygnTnVsbCcpKTtcbiAgICB0ZXN0Lm9rKCFleGlzdHMoJ1VuZGVmaW5lZCcpKTtcbiAgICB0ZXN0Lm9rKGV4aXN0cygnVHJ1ZScpKTtcbiAgICB0ZXN0Lm9rKGV4aXN0cygnRmFsc2UnKSk7XG4gICAgdGVzdC5vayhleGlzdHMoJ0VtcHR5JykpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhZGRXYXJuaW5nKG1lc3NhZ2UpIGNhbiBiZSB1c2VkIHRvIGFkZCBhIFwiV0FSTklOR1wiIG1lc3NhZ2UgZW50cnkgdG8gdGhlIGNvbnN0cnVjdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIGNvbnN0IGNvbiA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ015Q29uc3RydWN0Jyk7XG4gICAgY29uLm5vZGUuYWRkV2FybmluZygnVGhpcyBjb25zdHJ1Y3QgaXMgZGVwcmVjYXRlZCwgdXNlIHRoZSBvdGhlciBvbmUgaW5zdGVhZCcpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGNvbi5ub2RlLm1ldGFkYXRhWzBdLnR5cGUsIGN4YXBpLldBUk5JTkdfTUVUQURBVEFfS0VZKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjb24ubm9kZS5tZXRhZGF0YVswXS5kYXRhLCAnVGhpcyBjb25zdHJ1Y3QgaXMgZGVwcmVjYXRlZCwgdXNlIHRoZSBvdGhlciBvbmUgaW5zdGVhZCcpO1xuICAgIHRlc3Qub2soY29uLm5vZGUubWV0YWRhdGFbMF0udHJhY2UgJiYgY29uLm5vZGUubWV0YWRhdGFbMF0udHJhY2UubGVuZ3RoID4gMCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2FkZEVycm9yKG1lc3NhZ2UpIGNhbiBiZSB1c2VkIHRvIGFkZCBhIFwiRVJST1JcIiBtZXNzYWdlIGVudHJ5IHRvIHRoZSBjb25zdHJ1Y3QnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFJvb3QoKTtcbiAgICBjb25zdCBjb24gPSBuZXcgQ29uc3RydWN0KHJvb3QsICdNeUNvbnN0cnVjdCcpO1xuICAgIGNvbi5ub2RlLmFkZEVycm9yKCdTdG9wIScpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGNvbi5ub2RlLm1ldGFkYXRhWzBdLnR5cGUsIGN4YXBpLkVSUk9SX01FVEFEQVRBX0tFWSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoY29uLm5vZGUubWV0YWRhdGFbMF0uZGF0YSwgJ1N0b3AhJyk7XG4gICAgdGVzdC5vayhjb24ubm9kZS5tZXRhZGF0YVswXS50cmFjZSAmJiBjb24ubm9kZS5tZXRhZGF0YVswXS50cmFjZS5sZW5ndGggPiAwKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkSW5mbyhtZXNzYWdlKSBjYW4gYmUgdXNlZCB0byBhZGQgYW4gXCJJTkZPXCIgbWVzc2FnZSBlbnRyeSB0byB0aGUgY29uc3RydWN0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG4gICAgY29uc3QgY29uID0gbmV3IENvbnN0cnVjdChyb290LCAnTXlDb25zdHJ1Y3QnKTtcbiAgICBjb24ubm9kZS5hZGRJbmZvKCdIZXkgdGhlcmUsIGhvdyBkbyB5b3UgZG8/Jyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoY29uLm5vZGUubWV0YWRhdGFbMF0udHlwZSwgY3hhcGkuSU5GT19NRVRBREFUQV9LRVkpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGNvbi5ub2RlLm1ldGFkYXRhWzBdLmRhdGEsICdIZXkgdGhlcmUsIGhvdyBkbyB5b3UgZG8/Jyk7XG4gICAgdGVzdC5vayhjb24ubm9kZS5tZXRhZGF0YVswXS50cmFjZSAmJiBjb24ubm9kZS5tZXRhZGF0YVswXS50cmFjZS5sZW5ndGggPiAwKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbXVsdGlwbGUgY2hpbGRyZW4gb2YgdGhlIHNhbWUgdHlwZSwgd2l0aCBleHBsaWNpdCBuYW1lcyBhcmUgd2VsY29tZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgUm9vdCgpO1xuICAgIG5ldyBNeUJlYXV0aWZ1bENvbnN0cnVjdChyb290LCAnbWJjMScpO1xuICAgIG5ldyBNeUJlYXV0aWZ1bENvbnN0cnVjdChyb290LCAnbWJjMicpO1xuICAgIG5ldyBNeUJlYXV0aWZ1bENvbnN0cnVjdChyb290LCAnbWJjMycpO1xuICAgIG5ldyBNeUJlYXV0aWZ1bENvbnN0cnVjdChyb290LCAnbWJjNCcpO1xuICAgIHRlc3QuZXF1YWwocm9vdC5ub2RlLmNoaWxkcmVuLmxlbmd0aCwgNCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvbnN0cnVjdC5yZXF1aXJlZChwcm9wcywgbmFtZSkgY2FuIGJlIHVzZWQgdG8gdmFsaWRhdGUgdGhhdCByZXF1aXJlZCBwcm9wZXJ0aWVzIGFyZSBkZWZpbmVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG5cbiAgICAvLyBzaG91bGQgYmUgb2tcbiAgICBjb25zdCBjID0gbmV3IENvbnN0cnVjdFdpdGhSZXF1aXJlZChyb290LCAnQ29uc3RydWN0JywgeyByZXF1aXJlZFByb3A6IDEyMywgYW5vdGhlclJlcXVpcmVkUHJvcDogdHJ1ZSB9KTtcbiAgICB0ZXN0LmVxdWFsKGMucmVxdWlyZWRQcm9wLCAxMjMpO1xuICAgIHRlc3QuZXF1YWwoYy5hbm90aGVyUmVxdWlyZWRQcm9wLCB0cnVlKTtcblxuICAgIC8vIHNob3VsZCB0aHJvd1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBDb25zdHJ1Y3RXaXRoUmVxdWlyZWQocm9vdCwgJ0MnLCB7IG9wdGlvbmFsUHJvcDogJ2hlbGxvJyB9IGFzIGFueSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgJ2NvbnN0cnVjdC52YWxpZGF0ZSgpIGNhbiBiZSBpbXBsZW1lbnRlZCB0byBwZXJmb3JtIHZhbGlkYXRpb24sIGNvbnN0cnVjdC52YWxpZGF0ZVRyZWUoKSB3aWxsIHJldHVybiBhbGwgZXJyb3JzIGZyb20gdGhlIHN1YnRyZWUgKERGUyknKHRlc3Q6IFRlc3QpIHtcblxuICAgIGNsYXNzIE15Q29uc3RydWN0IGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICAgICAgcmV0dXJuIFsgJ215LWVycm9yMScsICdteS1lcnJvcjInIF07XG4gICAgICB9XG4gICAgfVxuXG4gICAgY2xhc3MgWW91ckNvbnN0cnVjdCBleHRlbmRzIENvbnN0cnVjdCB7XG4gICAgICBwcm90ZWN0ZWQgdmFsaWRhdGUoKSB7XG4gICAgICAgIHJldHVybiBbICd5b3VyLWVycm9yMScgXTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjbGFzcyBUaGVpckNvbnN0cnVjdCBleHRlbmRzIENvbnN0cnVjdCB7XG4gICAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAgICAgbmV3IFlvdXJDb25zdHJ1Y3QodGhpcywgJ1lvdXJDb25zdHJ1Y3QnKTtcbiAgICAgIH1cblxuICAgICAgcHJvdGVjdGVkIHZhbGlkYXRlKCkge1xuICAgICAgICByZXR1cm4gWyAndGhlaXItZXJyb3InIF07XG4gICAgICB9XG4gICAgfVxuXG4gICAgY2xhc3MgVGVzdFN0YWNrIGV4dGVuZHMgUm9vdCB7XG4gICAgICBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgc3VwZXIoKTtcblxuICAgICAgICBuZXcgTXlDb25zdHJ1Y3QodGhpcywgJ015Q29uc3RydWN0Jyk7XG4gICAgICAgIG5ldyBUaGVpckNvbnN0cnVjdCh0aGlzLCAnVGhlaXJDb25zdHJ1Y3QnKTtcbiAgICAgIH1cblxuICAgICAgcHJvdGVjdGVkIHZhbGlkYXRlKCkge1xuICAgICAgICByZXR1cm4gIFsgJ3N0YWNrLWVycm9yJyBdO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFRlc3RTdGFjaygpO1xuXG4gICAgY29uc3QgZXJyb3JzID0gKHN0YWNrLm5vZGUudmFsaWRhdGVUcmVlKCkpLm1hcCh2ID0+ICh7IHBhdGg6IHYuc291cmNlLm5vZGUucGF0aCwgbWVzc2FnZTogdi5tZXNzYWdlIH0pKTtcblxuICAgIC8vIHZhbGlkYXRlIERGU1xuICAgIHRlc3QuZGVlcEVxdWFsKGVycm9ycywgW1xuICAgICAgeyBwYXRoOiAnTXlDb25zdHJ1Y3QnLCBtZXNzYWdlOiAnbXktZXJyb3IxJyB9LFxuICAgICAgeyBwYXRoOiAnTXlDb25zdHJ1Y3QnLCBtZXNzYWdlOiAnbXktZXJyb3IyJyB9LFxuICAgICAgeyBwYXRoOiAnVGhlaXJDb25zdHJ1Y3QvWW91ckNvbnN0cnVjdCcsIG1lc3NhZ2U6ICd5b3VyLWVycm9yMScgfSxcbiAgICAgIHsgcGF0aDogJ1RoZWlyQ29uc3RydWN0JywgbWVzc2FnZTogJ3RoZWlyLWVycm9yJyB9LFxuICAgICAgeyBwYXRoOiAnJywgbWVzc2FnZTogJ3N0YWNrLWVycm9yJyB9XG4gICAgXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29uc3RydWN0LmxvY2soKSBwcm90ZWN0cyBhZ2FpbnN0IGFkZGluZyBjaGlsZHJlbiBhbnl3aGVyZSB1bmRlciB0aGlzIGNvbnN0cnVjdCAoZGlyZWN0IG9yIGluZGlyZWN0KScodGVzdDogVGVzdCkge1xuXG4gICAgY2xhc3MgTG9ja2FibGVDb25zdHJ1Y3QgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgICAgcHVibGljIGxvY2tNZSgpIHtcbiAgICAgICAgdGhpcy5ub2RlLmxvY2soKTtcbiAgICAgIH1cblxuICAgICAgcHVibGljIHVubG9ja01lKCkge1xuICAgICAgICB0aGlzLm5vZGUudW5sb2NrKCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgUm9vdCgpO1xuXG4gICAgY29uc3QgYzBhID0gbmV3IExvY2thYmxlQ29uc3RydWN0KHN0YWNrLCAnYzBhJyk7XG4gICAgY29uc3QgYzBiID0gbmV3IENvbnN0cnVjdChzdGFjaywgJ2MwYicpO1xuXG4gICAgY29uc3QgYzFhID0gbmV3IENvbnN0cnVjdChjMGEsICdjMWEnKTtcbiAgICBjb25zdCBjMWIgPSBuZXcgQ29uc3RydWN0KGMwYSwgJ2MxYicpO1xuXG4gICAgYzBhLmxvY2tNZSgpO1xuXG4gICAgLy8gbm93IHdlIHNob3VsZCBzdGlsbCBiZSBhYmxlIHRvIGFkZCBjaGlsZHJlbiB0byBjMGIsIGJ1dCBub3QgdG8gYzBhIG9yIGFueSBpdHMgY2hpbGRyZW5cbiAgICBuZXcgQ29uc3RydWN0KGMwYiwgJ2MxYScpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBDb25zdHJ1Y3QoYzBhLCAnZmFpbDEnKSwgL0Nhbm5vdCBhZGQgY2hpbGRyZW4gdG8gXCJjMGFcIiBkdXJpbmcgc3ludGhlc2lzLyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IENvbnN0cnVjdChjMWEsICdmYWlsMicpLCAvQ2Fubm90IGFkZCBjaGlsZHJlbiB0byBcImMwYVxcL2MxYVwiIGR1cmluZyBzeW50aGVzaXMvKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgQ29uc3RydWN0KGMxYiwgJ2ZhaWwzJyksIC9DYW5ub3QgYWRkIGNoaWxkcmVuIHRvIFwiYzBhXFwvYzFiXCIgZHVyaW5nIHN5bnRoZXNpcy8pO1xuXG4gICAgYzBhLnVubG9ja01lKCk7XG5cbiAgICBuZXcgQ29uc3RydWN0KGMwYSwgJ2MwYVonKTtcbiAgICBuZXcgQ29uc3RydWN0KGMxYSwgJ2MxYVonKTtcbiAgICBuZXcgQ29uc3RydWN0KGMxYiwgJ2MxYlonKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdmaW5kQWxsIHJldHVybnMgYSBsaXN0IG9mIGFsbCBjaGlsZHJlbiBpbiBlaXRoZXIgREZTIG9yIEJGUycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYzEgPSBuZXcgQ29uc3RydWN0KHVuZGVmaW5lZCBhcyBhbnksICcxJyk7XG4gICAgY29uc3QgYzIgPSBuZXcgQ29uc3RydWN0KGMxLCAnMicpO1xuICAgIG5ldyBDb25zdHJ1Y3QoYzEsICczJyk7XG4gICAgbmV3IENvbnN0cnVjdChjMiwgJzQnKTtcbiAgICBuZXcgQ29uc3RydWN0KGMyLCAnNScpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKGMxLm5vZGUuZmluZEFsbCgpLm1hcCh4ID0+IHgubm9kZS5pZCksIGMxLm5vZGUuZmluZEFsbChDb25zdHJ1Y3RPcmRlci5QcmVPcmRlcikubWFwKHggPT4geC5ub2RlLmlkKSk7IC8vIGRlZmF1bHQgaXMgUHJlT3JkZXJcbiAgICB0ZXN0LmRlZXBFcXVhbChjMS5ub2RlLmZpbmRBbGwoQ29uc3RydWN0T3JkZXIuUHJlT3JkZXIpLm1hcCh4ID0+IHgubm9kZS5pZCksIFsgJzEnLCAnMicsICc0JywgJzUnLCAnMycgXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYzEubm9kZS5maW5kQWxsKENvbnN0cnVjdE9yZGVyLlBvc3RPcmRlcikubWFwKHggPT4geC5ub2RlLmlkKSwgWyAnNCcsICc1JywgJzInLCAnMycsICcxJyBdKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYW5jZXN0b3JzIHJldHVybnMgYSBsaXN0IG9mIHBhcmVudHMgdXAgdG8gcm9vdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHsgY2hpbGQxLCBjaGlsZDFfMV8xIH0gPSBjcmVhdGVUcmVlKCk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChjaGlsZDFfMV8xLm5vZGUuYW5jZXN0b3JzKCkubWFwKHggPT4geC5ub2RlLmlkKSwgWyAnJywgJ0NoaWxkMScsICdDaGlsZDExJywgJ0NoaWxkMTExJyBdKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjaGlsZDFfMV8xLm5vZGUuYW5jZXN0b3JzKGNoaWxkMSkubWFwKHggPT4geC5ub2RlLmlkKSwgWyAnQ2hpbGQxMScsICdDaGlsZDExMScgXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoY2hpbGQxXzFfMS5ub2RlLmFuY2VzdG9ycyhjaGlsZDFfMV8xKSwgWyBdKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnXCJyb290XCIgcmV0dXJucyB0aGUgcm9vdCBjb25zdHJ1Y3QnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCB7IGNoaWxkMSwgY2hpbGQyLCBjaGlsZDFfMV8xLCByb290IH0gPSBjcmVhdGVUcmVlKCk7XG4gICAgdGVzdC5vayhjaGlsZDEubm9kZS5yb290ID09PSByb290KTtcbiAgICB0ZXN0Lm9rKGNoaWxkMi5ub2RlLnJvb3QgPT09IHJvb3QpO1xuICAgIHRlc3Qub2soY2hpbGQxXzFfMS5ub2RlLnJvb3QgPT09IHJvb3QpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9XG59O1xuXG5mdW5jdGlvbiBjcmVhdGVUcmVlKGNvbnRleHQ/OiBhbnkpIHtcbiAgY29uc3Qgcm9vdCA9IG5ldyBSb290KCk7XG4gIGlmIChjb250ZXh0KSB7XG4gICAgT2JqZWN0LmtleXMoY29udGV4dCkuZm9yRWFjaChrZXkgPT4gcm9vdC5ub2RlLnNldENvbnRleHQoa2V5LCBjb250ZXh0W2tleV0pKTtcbiAgfVxuXG4gIGNvbnN0IGNoaWxkMSA9IG5ldyBDb25zdHJ1Y3Qocm9vdCwgJ0NoaWxkMScpO1xuICBjb25zdCBjaGlsZDIgPSBuZXcgQ29uc3RydWN0KHJvb3QsICdDaGlsZDInKTtcbiAgY29uc3QgY2hpbGQxXzEgPSBuZXcgQ29uc3RydWN0KGNoaWxkMSwgJ0NoaWxkMTEnKTtcbiAgY29uc3QgY2hpbGQxXzIgPSBuZXcgQ29uc3RydWN0KGNoaWxkMSwgJ0NoaWxkMTInKTtcbiAgY29uc3QgY2hpbGQxXzFfMSA9IG5ldyBDb25zdHJ1Y3QoY2hpbGQxXzEsICdDaGlsZDExMScpO1xuICBjb25zdCBjaGlsZDJfMSA9IG5ldyBDb25zdHJ1Y3QoY2hpbGQyLCAnQ2hpbGQyMScpO1xuXG4gIHJldHVybiB7XG4gICAgcm9vdCwgY2hpbGQxLCBjaGlsZDIsIGNoaWxkMV8xLCBjaGlsZDFfMiwgY2hpbGQxXzFfMSwgY2hpbGQyXzFcbiAgfTtcbn1cblxuY2xhc3MgTXlCZWF1dGlmdWxDb25zdHJ1Y3QgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgQ29uc3RydWN0V2l0aFJlcXVpcmVkUHJvcHMge1xuICBvcHRpb25hbFByb3A/OiBzdHJpbmc7XG4gIHJlcXVpcmVkUHJvcDogbnVtYmVyO1xuICBhbm90aGVyUmVxdWlyZWRQcm9wOiBib29sZWFuO1xufVxuXG5jbGFzcyBDb25zdHJ1Y3RXaXRoUmVxdWlyZWQgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgcmVxdWlyZWRQcm9wOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhbm90aGVyUmVxdWlyZWRQcm9wOiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDb25zdHJ1Y3RXaXRoUmVxdWlyZWRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnJlcXVpcmVkUHJvcCA9IHRoaXMubm9kZS5yZXF1aXJlZChwcm9wcywgJ3JlcXVpcmVkUHJvcCcpO1xuICAgIHRoaXMuYW5vdGhlclJlcXVpcmVkUHJvcCA9IHRoaXMubm9kZS5yZXF1aXJlZChwcm9wcywgJ2Fub3RoZXJSZXF1aXJlZFByb3AnKTtcbiAgfVxufVxuIl19