"use strict";
const lib_1 = require("../lib");
const evaluate_cfn_1 = require("./evaluate-cfn");
/**
 * Return two Tokens, one of which evaluates to a Token directly, one which evaluates to it lazily
 */
function tokensThatResolveTo(value) {
    return [
        new lib_1.Token(value),
        new lib_1.Token(() => value)
    ];
}
module.exports = {
    'string tokens can be JSONified and JSONification can be reversed'(test) {
        const stack = new lib_1.Stack();
        for (const token of tokensThatResolveTo('woof woof')) {
            // GIVEN
            const fido = { name: 'Fido', speaks: token };
            // WHEN
            const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON(fido));
            // THEN
            test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), '{"name":"Fido","speaks":"woof woof"}');
        }
        test.done();
    },
    'string tokens can be embedded while being JSONified'(test) {
        const stack = new lib_1.Stack();
        for (const token of tokensThatResolveTo('woof woof')) {
            // GIVEN
            const fido = { name: 'Fido', speaks: `deep ${token}` };
            // WHEN
            const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON(fido));
            // THEN
            test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), '{"name":"Fido","speaks":"deep woof woof"}');
        }
        test.done();
    },
    'constant string has correct amount of quotes applied'(test) {
        const stack = new lib_1.Stack();
        const inputString = 'Hello, "world"';
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON(inputString));
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), JSON.stringify(inputString));
        test.done();
    },
    'integer Tokens behave correctly in stringification and JSONification'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const num = new lib_1.Token(() => 1);
        const embedded = `the number is ${num}`;
        // WHEN
        test.equal(evaluate_cfn_1.evaluateCFN(stack.node.resolve(embedded)), "the number is 1");
        test.equal(evaluate_cfn_1.evaluateCFN(stack.node.resolve(lib_1.CloudFormationLang.toJSON({ embedded }))), "{\"embedded\":\"the number is 1\"}");
        test.equal(evaluate_cfn_1.evaluateCFN(stack.node.resolve(lib_1.CloudFormationLang.toJSON({ num }))), "{\"num\":1}");
        test.done();
    },
    'tokens in strings survive additional TokenJSON.stringification()'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        for (const token of tokensThatResolveTo('pong!')) {
            // WHEN
            const stringified = lib_1.CloudFormationLang.toJSON(`ping? ${token}`);
            // THEN
            test.equal(evaluate_cfn_1.evaluateCFN(stack.node.resolve(stringified)), '"ping? pong!"');
        }
        test.done();
    },
    'intrinsic Tokens embed correctly in JSONification'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bucketName = new lib_1.Token({ Ref: 'MyBucket' });
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON({ theBucket: bucketName }));
        // THEN
        const context = { MyBucket: 'TheName' };
        test.equal(evaluate_cfn_1.evaluateCFN(resolved, context), '{"theBucket":"TheName"}');
        test.done();
    },
    'fake intrinsics are serialized to objects'(test) {
        const stack = new lib_1.Stack();
        const fakeIntrinsics = new lib_1.Token(() => ({
            a: {
                'Fn::GetArtifactAtt': {
                    key: 'val',
                },
            },
            b: {
                'Fn::GetParam': [
                    'val1',
                    'val2',
                ],
            },
        }));
        const stringified = lib_1.CloudFormationLang.toJSON(fakeIntrinsics);
        test.equal(evaluate_cfn_1.evaluateCFN(stack.node.resolve(stringified)), '{"a":{"Fn::GetArtifactAtt":{"key":"val"}},"b":{"Fn::GetParam":["val1","val2"]}}');
        test.done();
    },
    'embedded string literals in intrinsics are escaped when calling TokenJSON.stringify()'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const token = lib_1.Fn.join('', ['Hello', 'This\nIs', 'Very "cool"']);
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON({
            literal: 'I can also "contain" quotes',
            token
        }));
        // THEN
        const expected = '{"literal":"I can also \\"contain\\" quotes","token":"HelloThis\\nIsVery \\"cool\\""}';
        test.equal(evaluate_cfn_1.evaluateCFN(resolved), expected);
        test.done();
    },
    'Tokens in Tokens are handled correctly'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bucketName = new lib_1.Token({ Ref: 'MyBucket' });
        const combinedName = lib_1.Fn.join('', ['The bucket name is ', bucketName.toString()]);
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON({ theBucket: combinedName }));
        // THEN
        const context = { MyBucket: 'TheName' };
        test.equal(evaluate_cfn_1.evaluateCFN(resolved, context), '{"theBucket":"The bucket name is TheName"}');
        test.done();
    },
    'Doubly nested strings evaluate correctly in JSON context'(test) {
        // WHEN
        const stack = new lib_1.Stack();
        const fidoSays = new lib_1.Token(() => 'woof');
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON({
            information: `Did you know that Fido says: ${fidoSays}`
        }));
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), '{"information":"Did you know that Fido says: woof"}');
        test.done();
    },
    'Doubly nested intrinsics evaluate correctly in JSON context'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const fidoSays = new lib_1.Token(() => ({ Ref: 'Something' }));
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON({
            information: `Did you know that Fido says: ${fidoSays}`
        }));
        // THEN
        const context = { Something: 'woof woof' };
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved, context), '{"information":"Did you know that Fido says: woof woof"}');
        test.done();
    },
    'Quoted strings in embedded JSON context are escaped'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const fidoSays = new lib_1.Token(() => '"woof"');
        // WHEN
        const resolved = stack.node.resolve(lib_1.CloudFormationLang.toJSON({
            information: `Did you know that Fido says: ${fidoSays}`
        }));
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), '{"information":"Did you know that Fido says: \\"woof\\""}');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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