"use strict";
const lib_1 = require("../lib");
module.exports = {
    'create from components with defaults'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'sqs',
            resource: 'myqueuename'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.node.resolve(arn), stack.node.resolve(`arn:${pseudo.partition}:sqs:${pseudo.region}:${pseudo.accountId}:myqueuename`));
        test.done();
    },
    'create from components with specific values for the various components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'dynamodb',
            resource: 'table',
            account: '123456789012',
            region: 'us-east-1',
            partition: 'aws-cn',
            resourceName: 'mytable/stream/label'
        });
        test.deepEqual(stack.node.resolve(arn), 'arn:aws-cn:dynamodb:us-east-1:123456789012:table/mytable/stream/label');
        test.done();
    },
    'allow empty string in components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 's3',
            resource: 'my-bucket',
            account: '',
            region: '',
            partition: 'aws-cn',
        });
        test.deepEqual(stack.node.resolve(arn), 'arn:aws-cn:s3:::my-bucket');
        test.done();
    },
    'resourcePathSep can be set to ":" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'codedeploy',
            resource: 'application',
            sep: ':',
            resourceName: 'WordPress_App'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.node.resolve(arn), stack.node.resolve(`arn:${pseudo.partition}:codedeploy:${pseudo.region}:${pseudo.accountId}:application:WordPress_App`));
        test.done();
    },
    'resourcePathSep can be set to "" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'ssm',
            resource: 'parameter',
            sep: '',
            resourceName: '/parameter-name'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.node.resolve(arn), stack.node.resolve(`arn:${pseudo.partition}:ssm:${pseudo.region}:${pseudo.accountId}:parameter/parameter-name`));
        test.done();
    },
    'fails if resourcePathSep is neither ":" nor "/"'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => stack.formatArn({
            service: 'foo',
            resource: 'bar',
            sep: 'x'
        }));
        test.done();
    },
    'Arn.parse(s)': {
        'fails': {
            'if doesn\'t start with "arn:"'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn("barn:foo:x:a:1:2"), /ARNs must start with "arn:": barn:foo/);
                test.done();
            },
            'if the ARN doesnt have enough components'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:is:too:short'), /ARNs must have at least 6 components: arn:is:too:short/);
                test.done();
            },
            'if "service" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws::4:5:6'), /The `service` component \(3rd component\) is required/);
                test.done();
            },
            'if "resource" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws:service:::'), /The `resource` component \(6th component\) is required/);
                test.done();
            }
        },
        'various successful parses'(test) {
            const stack = new lib_1.Stack();
            const tests = {
                'arn:aws:a4b:region:accountid:resourcetype/resource': {
                    partition: 'aws',
                    service: 'a4b',
                    region: 'region',
                    account: 'accountid',
                    resource: 'resourcetype',
                    resourceName: 'resource',
                    sep: '/'
                },
                'arn:aws:apigateway:us-east-1::a123456789012bc3de45678901f23a45:/test/mydemoresource/*': {
                    partition: 'aws',
                    service: 'apigateway',
                    region: 'us-east-1',
                    resource: 'a123456789012bc3de45678901f23a45',
                    sep: ':',
                    resourceName: '/test/mydemoresource/*'
                },
                'arn:aws-cn:cloud9::123456789012:environment:81e900317347585a0601e04c8d52eaEX': {
                    partition: 'aws-cn',
                    service: 'cloud9',
                    account: '123456789012',
                    resource: 'environment',
                    resourceName: '81e900317347585a0601e04c8d52eaEX',
                    sep: ':'
                },
                'arn::cognito-sync:::identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla': {
                    service: 'cognito-sync',
                    resource: 'identitypool',
                    resourceName: 'us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla',
                    sep: '/'
                },
                'arn:aws:s3:::my_corporate_bucket': {
                    partition: 'aws',
                    service: 's3',
                    resource: 'my_corporate_bucket'
                }
            };
            Object.keys(tests).forEach(arn => {
                const expected = tests[arn];
                test.deepEqual(stack.parseArn(arn), expected, arn);
            });
            test.done();
        },
        'a Token with : separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new lib_1.Token(() => theToken).toString(), ':');
            test.deepEqual(stack.node.resolve(parsed.partition), { 'Fn::Select': [1, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.node.resolve(parsed.service), { 'Fn::Select': [2, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.node.resolve(parsed.region), { 'Fn::Select': [3, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.node.resolve(parsed.account), { 'Fn::Select': [4, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.node.resolve(parsed.resource), { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.node.resolve(parsed.resourceName), { 'Fn::Select': [6, { 'Fn::Split': [':', theToken] }] });
            test.equal(parsed.sep, ':');
            test.done();
        },
        'a Token with / separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new lib_1.Token(() => theToken).toString());
            test.equal(parsed.sep, '/');
            // tslint:disable-next-line:max-line-length
            test.deepEqual(stack.node.resolve(parsed.resource), { 'Fn::Select': [0, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            // tslint:disable-next-line:max-line-length
            test.deepEqual(stack.node.resolve(parsed.resourceName), { 'Fn::Select': [1, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            test.done();
        }
    },
    'can use a fully specified ARN from a different stack without incurring an import'(test) {
        // GIVEN
        const stack1 = new lib_1.Stack(undefined, 'Stack1', { env: { account: '12345678', region: 'us-turbo-5' } });
        const stack2 = new lib_1.Stack(undefined, 'Stack2', { env: { account: '87654321', region: 'us-turbo-1' } });
        // WHEN
        const arn = stack1.formatArn({
            // No partition specified here
            service: 'bla',
            resource: 'thing',
            resourceName: 'thong',
        });
        new lib_1.CfnOutput(stack2, 'SomeValue', { value: arn });
        // THEN
        test.deepEqual(stack2.node.resolve(stack2._toCloudFormation()), {
            Outputs: {
                SomeValue: {
                    Value: {
                        // Look ma, no Fn::ImportValue!
                        'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':bla:us-turbo-5:12345678:thing/thong']]
                    }
                }
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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