"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("./token");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    obj = construct.node.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without undefined values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x !== undefined).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value === undefined) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends token_1.Token {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
//# sourceMappingURL=data:application/json;base64,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