"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./encoding");
const stack_trace_1 = require("./stack-trace");
const token_map_1 = require("./token-map");
/**
 * If objects has a function property by this name, they will be considered tokens, and this
 * function will be called to resolve the value for this object.
 */
exports.RESOLVE_METHOD = 'resolve';
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Creates a token that resolves to `value`.
     *
     * If value is a function, the function is evaluated upon resolution and
     * the value it returns will be used as the token's value.
     *
     * displayName is used to represent the Token when it's embedded into a string; it
     * will look something like this:
     *
     *    "embedded in a larger string is ${Token[DISPLAY_NAME.123]}"
     *
     * This value is used as a hint to humans what the meaning of the Token is,
     * and does not have any effect on the evaluation.
     *
     * Must contain only alphanumeric and simple separator characters (_.:-).
     *
     * @param valueOrFunction What this token will evaluate to, literal or function.
     * @param displayName A human-readable display hint for this Token
     */
    constructor(valueOrFunction, displayName) {
        this.valueOrFunction = valueOrFunction;
        this.displayName = displayName;
        this.trace = stack_trace_1.createStackTrace();
    }
    /**
     * @deprecated use `Token.isToken`
     */
    static unresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Returns true if obj is a token (i.e. has the resolve() method or is a
     * string or array which includes token markers).
     *
     * @param obj The object to test.
     */
    static isToken(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * @returns The resolved value for this token.
     */
    resolve(context) {
        let value = this.valueOrFunction;
        if (typeof (value) === 'function') {
            value = value(context);
        }
        return value;
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    toString() {
        const valueType = typeof this.valueOrFunction;
        // Optimization: if we can immediately resolve this, don't bother
        // registering a Token.
        if (valueType === 'string' || valueType === 'number' || valueType === 'boolean') {
            return this.valueOrFunction.toString();
        }
        if (this.tokenStringification === undefined) {
            this.tokenStringification = token_map_1.TokenMap.instance().registerString(this, this.displayName);
        }
        return this.tokenStringification;
    }
    /**
     * Turn this Token into JSON
     *
     * This gets called by JSON.stringify(). We want to prohibit this, because
     * it's not possible to do this properly, so we just throw an error here.
     */
    toJSON() {
        // We can't do the right work here because in case we contain a function, we
        // won't know the type of value that function represents (in the simplest
        // case, string or number), and we can't know that without an
        // IResolveContext to actually do the resolution, which we don't have.
        // We used to throw an error, but since JSON.stringify() is often used in
        // error messages to produce a readable representation of an object, if we
        // throw here we'll obfuscate that descriptive error with something worse.
        // So return a string representation that indicates this thing is a token
        // and needs resolving.
        return JSON.stringify(`<unresolved-token:${this.displayName || 'TOKEN'}>`);
    }
    /**
     * Return a string list representation of this token
     *
     * Call this if the Token intrinsically evaluates to a list of strings.
     * If so, you can represent the Token in a similar way in the type
     * system.
     *
     * Note that even though the Token is represented as a list of strings, you
     * still cannot do any operations on it such as concatenation, indexing,
     * or taking its length. The only useful operations you can do to these lists
     * is constructing a `FnJoin` or a `FnSelect` on it.
     */
    toList() {
        const valueType = typeof this.valueOrFunction;
        if (valueType === 'string' || valueType === 'number' || valueType === 'boolean') {
            throw this.newError('Got a literal Token value; only intrinsics can ever evaluate to lists.');
        }
        if (this.tokenListification === undefined) {
            this.tokenListification = token_map_1.TokenMap.instance().registerList(this, this.displayName);
        }
        return this.tokenListification;
    }
    /**
     * Return a floating point representation of this Token
     *
     * Call this if the Token intrinsically resolves to something that represents
     * a number, and you need to pass it into an API that expects a number.
     *
     * You may not do any operations on the returned value; any arithmetic or
     * other operations can and probably will destroy the token-ness of the value.
     */
    toNumber() {
        if (this.tokenNumberification === undefined) {
            const valueType = typeof this.valueOrFunction;
            // Optimization: if we can immediately resolve this, don't bother
            // registering a Token.
            if (valueType === 'number') {
                return this.valueOrFunction;
            }
            if (valueType !== 'function') {
                throw this.newError(`Token value is not number or lazy, can't represent as number: ${this.valueOrFunction}`);
            }
            this.tokenNumberification = token_map_1.TokenMap.instance().registerNumber(this);
        }
        return this.tokenNumberification;
    }
    /**
     * Creates a throwable Error object that contains the token creation stack trace.
     * @param message Error message
     */
    newError(message) {
        return new Error(`${message}\nToken created:\n    at ${this.trace.join('\n    at ')}\nError thrown:`);
    }
}
exports.Token = Token;
/**
 * Whether the given object is an `IResolvedValuePostProcessor`
 */
function isResolvedValuePostProcessor(x) {
    return x.postProcess !== undefined;
}
exports.isResolvedValuePostProcessor = isResolvedValuePostProcessor;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9rZW4uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0b2tlbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLHlDQUF3QztBQUN4QywrQ0FBaUQ7QUFDakQsMkNBQXVDO0FBRXZDOzs7R0FHRztBQUNVLFFBQUEsY0FBYyxHQUFHLFNBQVMsQ0FBQztBQUV4Qzs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFhLEtBQUs7SUEyQmhCOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSCxZQUE2QixlQUFxQixFQUFtQixXQUFvQjtRQUE1RCxvQkFBZSxHQUFmLGVBQWUsQ0FBTTtRQUFtQixnQkFBVyxHQUFYLFdBQVcsQ0FBUztRQUN2RixJQUFJLENBQUMsS0FBSyxHQUFHLDhCQUFnQixFQUFFLENBQUM7SUFDbEMsQ0FBQztJQS9DRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBUTtRQUMvQixPQUFPLHFCQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFRO1FBQzVCLE9BQU8scUJBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QixDQUFDO0lBa0NEOztPQUVHO0lBQ0ksT0FBTyxDQUFDLE9BQXdCO1FBQ3JDLElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUM7UUFDakMsSUFBSSxPQUFNLENBQUMsS0FBSyxDQUFDLEtBQUssVUFBVSxFQUFFO1lBQ2hDLEtBQUssR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDeEI7UUFFRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUTtRQUNiLE1BQU0sU0FBUyxHQUFHLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQztRQUM5QyxpRUFBaUU7UUFDakUsdUJBQXVCO1FBQ3ZCLElBQUksU0FBUyxLQUFLLFFBQVEsSUFBSSxTQUFTLEtBQUssUUFBUSxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUU7WUFDL0UsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVEsRUFBRSxDQUFDO1NBQ3hDO1FBRUQsSUFBSSxJQUFJLENBQUMsb0JBQW9CLEtBQUssU0FBUyxFQUFFO1lBQzNDLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ3hGO1FBQ0QsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTTtRQUNYLDRFQUE0RTtRQUM1RSx5RUFBeUU7UUFDekUsNkRBQTZEO1FBQzdELHNFQUFzRTtRQUV0RSx5RUFBeUU7UUFDekUsMEVBQTBFO1FBQzFFLDBFQUEwRTtRQUMxRSx5RUFBeUU7UUFDekUsdUJBQXVCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxxQkFBcUIsSUFBSSxDQUFDLFdBQVcsSUFBSSxPQUFPLEdBQUcsQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLE1BQU07UUFDWCxNQUFNLFNBQVMsR0FBRyxPQUFPLElBQUksQ0FBQyxlQUFlLENBQUM7UUFDOUMsSUFBSSxTQUFTLEtBQUssUUFBUSxJQUFJLFNBQVMsS0FBSyxRQUFRLElBQUksU0FBUyxLQUFLLFNBQVMsRUFBRTtZQUMvRSxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUMvRjtRQUVELElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLFNBQVMsRUFBRTtZQUN6QyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsb0JBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNwRjtRQUNELE9BQU8sSUFBSSxDQUFDLGtCQUFrQixDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLFFBQVE7UUFDYixJQUFJLElBQUksQ0FBQyxvQkFBb0IsS0FBSyxTQUFTLEVBQUU7WUFDM0MsTUFBTSxTQUFTLEdBQUcsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUFDO1lBQzlDLGlFQUFpRTtZQUNqRSx1QkFBdUI7WUFDdkIsSUFBSSxTQUFTLEtBQUssUUFBUSxFQUFFO2dCQUFFLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQzthQUFFO1lBQzVELElBQUksU0FBUyxLQUFLLFVBQVUsRUFBRTtnQkFDNUIsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLGlFQUFpRSxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQzthQUM5RztZQUNELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN0RTtRQUVELE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDO0lBQ25DLENBQUM7SUFFRDs7O09BR0c7SUFDTyxRQUFRLENBQUMsT0FBZTtRQUNoQyxPQUFPLElBQUksS0FBSyxDQUFDLEdBQUcsT0FBTyw0QkFBNEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDeEcsQ0FBQztDQUNGO0FBbEtELHNCQWtLQztBQTJCRDs7R0FFRztBQUNILFNBQWdCLDRCQUE0QixDQUFDLENBQU07SUFDakQsT0FBTyxDQUFDLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQztBQUNyQyxDQUFDO0FBRkQsb0VBRUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSBcIi4vY29uc3RydWN0XCI7XG5pbXBvcnQgeyB1bnJlc29sdmVkIH0gZnJvbSBcIi4vZW5jb2RpbmdcIjtcbmltcG9ydCB7IGNyZWF0ZVN0YWNrVHJhY2UgfSBmcm9tICcuL3N0YWNrLXRyYWNlJztcbmltcG9ydCB7IFRva2VuTWFwIH0gZnJvbSBcIi4vdG9rZW4tbWFwXCI7XG5cbi8qKlxuICogSWYgb2JqZWN0cyBoYXMgYSBmdW5jdGlvbiBwcm9wZXJ0eSBieSB0aGlzIG5hbWUsIHRoZXkgd2lsbCBiZSBjb25zaWRlcmVkIHRva2VucywgYW5kIHRoaXNcbiAqIGZ1bmN0aW9uIHdpbGwgYmUgY2FsbGVkIHRvIHJlc29sdmUgdGhlIHZhbHVlIGZvciB0aGlzIG9iamVjdC5cbiAqL1xuZXhwb3J0IGNvbnN0IFJFU09MVkVfTUVUSE9EID0gJ3Jlc29sdmUnO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBzcGVjaWFsIG9yIGxhemlseS1ldmFsdWF0ZWQgdmFsdWUuXG4gKlxuICogQ2FuIGJlIHVzZWQgdG8gZGVsYXkgZXZhbHVhdGlvbiBvZiBhIGNlcnRhaW4gdmFsdWUgaW4gY2FzZSwgZm9yIGV4YW1wbGUsXG4gKiB0aGF0IGl0IHJlcXVpcmVzIHNvbWUgY29udGV4dCBvciBsYXRlLWJvdW5kIGRhdGEuIENhbiBhbHNvIGJlIHVzZWQgdG9cbiAqIG1hcmsgdmFsdWVzIHRoYXQgbmVlZCBzcGVjaWFsIHByb2Nlc3NpbmcgYXQgZG9jdW1lbnQgcmVuZGVyaW5nIHRpbWUuXG4gKlxuICogVG9rZW5zIGNhbiBiZSBlbWJlZGRlZCBpbnRvIHN0cmluZ3Mgd2hpbGUgcmV0YWluaW5nIHRoZWlyIG9yaWdpbmFsXG4gKiBzZW1hbnRpY3MuXG4gKi9cbmV4cG9ydCBjbGFzcyBUb2tlbiB7XG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYFRva2VuLmlzVG9rZW5gXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHVucmVzb2x2ZWQob2JqOiBhbnkpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdW5yZXNvbHZlZChvYmopO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdHJ1ZSBpZiBvYmogaXMgYSB0b2tlbiAoaS5lLiBoYXMgdGhlIHJlc29sdmUoKSBtZXRob2Qgb3IgaXMgYVxuICAgKiBzdHJpbmcgb3IgYXJyYXkgd2hpY2ggaW5jbHVkZXMgdG9rZW4gbWFya2VycykuXG4gICAqXG4gICAqIEBwYXJhbSBvYmogVGhlIG9iamVjdCB0byB0ZXN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpc1Rva2VuKG9iajogYW55KTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHVucmVzb2x2ZWQob2JqKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY2FwdHVyZWQgc3RhY2sgdHJhY2Ugd2hpY2ggcmVwcmVzZW50cyB0aGUgbG9jYXRpb24gaW4gd2hpY2ggdGhpcyB0b2tlbiB3YXMgY3JlYXRlZC5cbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSB0cmFjZTogc3RyaW5nW107XG5cbiAgcHJpdmF0ZSB0b2tlblN0cmluZ2lmaWNhdGlvbj86IHN0cmluZztcbiAgcHJpdmF0ZSB0b2tlbkxpc3RpZmljYXRpb24/OiBzdHJpbmdbXTtcbiAgcHJpdmF0ZSB0b2tlbk51bWJlcmlmaWNhdGlvbj86IG51bWJlcjtcblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHRva2VuIHRoYXQgcmVzb2x2ZXMgdG8gYHZhbHVlYC5cbiAgICpcbiAgICogSWYgdmFsdWUgaXMgYSBmdW5jdGlvbiwgdGhlIGZ1bmN0aW9uIGlzIGV2YWx1YXRlZCB1cG9uIHJlc29sdXRpb24gYW5kXG4gICAqIHRoZSB2YWx1ZSBpdCByZXR1cm5zIHdpbGwgYmUgdXNlZCBhcyB0aGUgdG9rZW4ncyB2YWx1ZS5cbiAgICpcbiAgICogZGlzcGxheU5hbWUgaXMgdXNlZCB0byByZXByZXNlbnQgdGhlIFRva2VuIHdoZW4gaXQncyBlbWJlZGRlZCBpbnRvIGEgc3RyaW5nOyBpdFxuICAgKiB3aWxsIGxvb2sgc29tZXRoaW5nIGxpa2UgdGhpczpcbiAgICpcbiAgICogICAgXCJlbWJlZGRlZCBpbiBhIGxhcmdlciBzdHJpbmcgaXMgJHtUb2tlbltESVNQTEFZX05BTUUuMTIzXX1cIlxuICAgKlxuICAgKiBUaGlzIHZhbHVlIGlzIHVzZWQgYXMgYSBoaW50IHRvIGh1bWFucyB3aGF0IHRoZSBtZWFuaW5nIG9mIHRoZSBUb2tlbiBpcyxcbiAgICogYW5kIGRvZXMgbm90IGhhdmUgYW55IGVmZmVjdCBvbiB0aGUgZXZhbHVhdGlvbi5cbiAgICpcbiAgICogTXVzdCBjb250YWluIG9ubHkgYWxwaGFudW1lcmljIGFuZCBzaW1wbGUgc2VwYXJhdG9yIGNoYXJhY3RlcnMgKF8uOi0pLlxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWVPckZ1bmN0aW9uIFdoYXQgdGhpcyB0b2tlbiB3aWxsIGV2YWx1YXRlIHRvLCBsaXRlcmFsIG9yIGZ1bmN0aW9uLlxuICAgKiBAcGFyYW0gZGlzcGxheU5hbWUgQSBodW1hbi1yZWFkYWJsZSBkaXNwbGF5IGhpbnQgZm9yIHRoaXMgVG9rZW5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgdmFsdWVPckZ1bmN0aW9uPzogYW55LCBwcml2YXRlIHJlYWRvbmx5IGRpc3BsYXlOYW1lPzogc3RyaW5nKSB7XG4gICAgdGhpcy50cmFjZSA9IGNyZWF0ZVN0YWNrVHJhY2UoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBUaGUgcmVzb2x2ZWQgdmFsdWUgZm9yIHRoaXMgdG9rZW4uXG4gICAqL1xuICBwdWJsaWMgcmVzb2x2ZShjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpOiBhbnkge1xuICAgIGxldCB2YWx1ZSA9IHRoaXMudmFsdWVPckZ1bmN0aW9uO1xuICAgIGlmICh0eXBlb2YodmFsdWUpID09PSAnZnVuY3Rpb24nKSB7XG4gICAgICB2YWx1ZSA9IHZhbHVlKGNvbnRleHQpO1xuICAgIH1cblxuICAgIHJldHVybiB2YWx1ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIHN0cmluZyByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqXG4gICAqIElmIHRoZSBUb2tlbiBpcyBpbml0aWFsaXplZCB3aXRoIGEgbGl0ZXJhbCwgdGhlIHN0cmluZ2lmaWVkIHZhbHVlIG9mIHRoZVxuICAgKiBsaXRlcmFsIGlzIHJldHVybmVkLiBPdGhlcndpc2UsIGEgc3BlY2lhbCBxdW90ZWQgc3RyaW5nIHJlcHJlc2VudGF0aW9uXG4gICAqIG9mIHRoZSBUb2tlbiBpcyByZXR1cm5lZCB0aGF0IGNhbiBiZSBlbWJlZGRlZCBpbnRvIG90aGVyIHN0cmluZ3MuXG4gICAqXG4gICAqIFN0cmluZ3Mgd2l0aCBxdW90ZWQgVG9rZW5zIGluIHRoZW0gY2FuIGJlIHJlc3RvcmVkIGJhY2sgaW50b1xuICAgKiBjb21wbGV4IHZhbHVlcyB3aXRoIHRoZSBUb2tlbnMgcmVzdG9yZWQgYnkgY2FsbGluZyBgcmVzb2x2ZSgpYFxuICAgKiBvbiB0aGUgc3RyaW5nLlxuICAgKi9cbiAgcHVibGljIHRvU3RyaW5nKCk6IHN0cmluZyB7XG4gICAgY29uc3QgdmFsdWVUeXBlID0gdHlwZW9mIHRoaXMudmFsdWVPckZ1bmN0aW9uO1xuICAgIC8vIE9wdGltaXphdGlvbjogaWYgd2UgY2FuIGltbWVkaWF0ZWx5IHJlc29sdmUgdGhpcywgZG9uJ3QgYm90aGVyXG4gICAgLy8gcmVnaXN0ZXJpbmcgYSBUb2tlbi5cbiAgICBpZiAodmFsdWVUeXBlID09PSAnc3RyaW5nJyB8fCB2YWx1ZVR5cGUgPT09ICdudW1iZXInIHx8IHZhbHVlVHlwZSA9PT0gJ2Jvb2xlYW4nKSB7XG4gICAgICByZXR1cm4gdGhpcy52YWx1ZU9yRnVuY3Rpb24udG9TdHJpbmcoKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy50b2tlblN0cmluZ2lmaWNhdGlvbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLnRva2VuU3RyaW5naWZpY2F0aW9uID0gVG9rZW5NYXAuaW5zdGFuY2UoKS5yZWdpc3RlclN0cmluZyh0aGlzLCB0aGlzLmRpc3BsYXlOYW1lKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMudG9rZW5TdHJpbmdpZmljYXRpb247XG4gIH1cblxuICAvKipcbiAgICogVHVybiB0aGlzIFRva2VuIGludG8gSlNPTlxuICAgKlxuICAgKiBUaGlzIGdldHMgY2FsbGVkIGJ5IEpTT04uc3RyaW5naWZ5KCkuIFdlIHdhbnQgdG8gcHJvaGliaXQgdGhpcywgYmVjYXVzZVxuICAgKiBpdCdzIG5vdCBwb3NzaWJsZSB0byBkbyB0aGlzIHByb3Blcmx5LCBzbyB3ZSBqdXN0IHRocm93IGFuIGVycm9yIGhlcmUuXG4gICAqL1xuICBwdWJsaWMgdG9KU09OKCk6IGFueSB7XG4gICAgLy8gV2UgY2FuJ3QgZG8gdGhlIHJpZ2h0IHdvcmsgaGVyZSBiZWNhdXNlIGluIGNhc2Ugd2UgY29udGFpbiBhIGZ1bmN0aW9uLCB3ZVxuICAgIC8vIHdvbid0IGtub3cgdGhlIHR5cGUgb2YgdmFsdWUgdGhhdCBmdW5jdGlvbiByZXByZXNlbnRzIChpbiB0aGUgc2ltcGxlc3RcbiAgICAvLyBjYXNlLCBzdHJpbmcgb3IgbnVtYmVyKSwgYW5kIHdlIGNhbid0IGtub3cgdGhhdCB3aXRob3V0IGFuXG4gICAgLy8gSVJlc29sdmVDb250ZXh0IHRvIGFjdHVhbGx5IGRvIHRoZSByZXNvbHV0aW9uLCB3aGljaCB3ZSBkb24ndCBoYXZlLlxuXG4gICAgLy8gV2UgdXNlZCB0byB0aHJvdyBhbiBlcnJvciwgYnV0IHNpbmNlIEpTT04uc3RyaW5naWZ5KCkgaXMgb2Z0ZW4gdXNlZCBpblxuICAgIC8vIGVycm9yIG1lc3NhZ2VzIHRvIHByb2R1Y2UgYSByZWFkYWJsZSByZXByZXNlbnRhdGlvbiBvZiBhbiBvYmplY3QsIGlmIHdlXG4gICAgLy8gdGhyb3cgaGVyZSB3ZSdsbCBvYmZ1c2NhdGUgdGhhdCBkZXNjcmlwdGl2ZSBlcnJvciB3aXRoIHNvbWV0aGluZyB3b3JzZS5cbiAgICAvLyBTbyByZXR1cm4gYSBzdHJpbmcgcmVwcmVzZW50YXRpb24gdGhhdCBpbmRpY2F0ZXMgdGhpcyB0aGluZyBpcyBhIHRva2VuXG4gICAgLy8gYW5kIG5lZWRzIHJlc29sdmluZy5cbiAgICByZXR1cm4gSlNPTi5zdHJpbmdpZnkoYDx1bnJlc29sdmVkLXRva2VuOiR7dGhpcy5kaXNwbGF5TmFtZSB8fCAnVE9LRU4nfT5gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSBzdHJpbmcgbGlzdCByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqXG4gICAqIENhbGwgdGhpcyBpZiB0aGUgVG9rZW4gaW50cmluc2ljYWxseSBldmFsdWF0ZXMgdG8gYSBsaXN0IG9mIHN0cmluZ3MuXG4gICAqIElmIHNvLCB5b3UgY2FuIHJlcHJlc2VudCB0aGUgVG9rZW4gaW4gYSBzaW1pbGFyIHdheSBpbiB0aGUgdHlwZVxuICAgKiBzeXN0ZW0uXG4gICAqXG4gICAqIE5vdGUgdGhhdCBldmVuIHRob3VnaCB0aGUgVG9rZW4gaXMgcmVwcmVzZW50ZWQgYXMgYSBsaXN0IG9mIHN0cmluZ3MsIHlvdVxuICAgKiBzdGlsbCBjYW5ub3QgZG8gYW55IG9wZXJhdGlvbnMgb24gaXQgc3VjaCBhcyBjb25jYXRlbmF0aW9uLCBpbmRleGluZyxcbiAgICogb3IgdGFraW5nIGl0cyBsZW5ndGguIFRoZSBvbmx5IHVzZWZ1bCBvcGVyYXRpb25zIHlvdSBjYW4gZG8gdG8gdGhlc2UgbGlzdHNcbiAgICogaXMgY29uc3RydWN0aW5nIGEgYEZuSm9pbmAgb3IgYSBgRm5TZWxlY3RgIG9uIGl0LlxuICAgKi9cbiAgcHVibGljIHRvTGlzdCgpOiBzdHJpbmdbXSB7XG4gICAgY29uc3QgdmFsdWVUeXBlID0gdHlwZW9mIHRoaXMudmFsdWVPckZ1bmN0aW9uO1xuICAgIGlmICh2YWx1ZVR5cGUgPT09ICdzdHJpbmcnIHx8IHZhbHVlVHlwZSA9PT0gJ251bWJlcicgfHwgdmFsdWVUeXBlID09PSAnYm9vbGVhbicpIHtcbiAgICAgIHRocm93IHRoaXMubmV3RXJyb3IoJ0dvdCBhIGxpdGVyYWwgVG9rZW4gdmFsdWU7IG9ubHkgaW50cmluc2ljcyBjYW4gZXZlciBldmFsdWF0ZSB0byBsaXN0cy4nKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy50b2tlbkxpc3RpZmljYXRpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy50b2tlbkxpc3RpZmljYXRpb24gPSBUb2tlbk1hcC5pbnN0YW5jZSgpLnJlZ2lzdGVyTGlzdCh0aGlzLCB0aGlzLmRpc3BsYXlOYW1lKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMudG9rZW5MaXN0aWZpY2F0aW9uO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIGZsb2F0aW5nIHBvaW50IHJlcHJlc2VudGF0aW9uIG9mIHRoaXMgVG9rZW5cbiAgICpcbiAgICogQ2FsbCB0aGlzIGlmIHRoZSBUb2tlbiBpbnRyaW5zaWNhbGx5IHJlc29sdmVzIHRvIHNvbWV0aGluZyB0aGF0IHJlcHJlc2VudHNcbiAgICogYSBudW1iZXIsIGFuZCB5b3UgbmVlZCB0byBwYXNzIGl0IGludG8gYW4gQVBJIHRoYXQgZXhwZWN0cyBhIG51bWJlci5cbiAgICpcbiAgICogWW91IG1heSBub3QgZG8gYW55IG9wZXJhdGlvbnMgb24gdGhlIHJldHVybmVkIHZhbHVlOyBhbnkgYXJpdGhtZXRpYyBvclxuICAgKiBvdGhlciBvcGVyYXRpb25zIGNhbiBhbmQgcHJvYmFibHkgd2lsbCBkZXN0cm95IHRoZSB0b2tlbi1uZXNzIG9mIHRoZSB2YWx1ZS5cbiAgICovXG4gIHB1YmxpYyB0b051bWJlcigpOiBudW1iZXIge1xuICAgIGlmICh0aGlzLnRva2VuTnVtYmVyaWZpY2F0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGNvbnN0IHZhbHVlVHlwZSA9IHR5cGVvZiB0aGlzLnZhbHVlT3JGdW5jdGlvbjtcbiAgICAgIC8vIE9wdGltaXphdGlvbjogaWYgd2UgY2FuIGltbWVkaWF0ZWx5IHJlc29sdmUgdGhpcywgZG9uJ3QgYm90aGVyXG4gICAgICAvLyByZWdpc3RlcmluZyBhIFRva2VuLlxuICAgICAgaWYgKHZhbHVlVHlwZSA9PT0gJ251bWJlcicpIHsgcmV0dXJuIHRoaXMudmFsdWVPckZ1bmN0aW9uOyB9XG4gICAgICBpZiAodmFsdWVUeXBlICE9PSAnZnVuY3Rpb24nKSB7XG4gICAgICAgIHRocm93IHRoaXMubmV3RXJyb3IoYFRva2VuIHZhbHVlIGlzIG5vdCBudW1iZXIgb3IgbGF6eSwgY2FuJ3QgcmVwcmVzZW50IGFzIG51bWJlcjogJHt0aGlzLnZhbHVlT3JGdW5jdGlvbn1gKTtcbiAgICAgIH1cbiAgICAgIHRoaXMudG9rZW5OdW1iZXJpZmljYXRpb24gPSBUb2tlbk1hcC5pbnN0YW5jZSgpLnJlZ2lzdGVyTnVtYmVyKHRoaXMpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLnRva2VuTnVtYmVyaWZpY2F0aW9uO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB0aHJvd2FibGUgRXJyb3Igb2JqZWN0IHRoYXQgY29udGFpbnMgdGhlIHRva2VuIGNyZWF0aW9uIHN0YWNrIHRyYWNlLlxuICAgKiBAcGFyYW0gbWVzc2FnZSBFcnJvciBtZXNzYWdlXG4gICAqL1xuICBwcm90ZWN0ZWQgbmV3RXJyb3IobWVzc2FnZTogc3RyaW5nKTogYW55IHtcbiAgICByZXR1cm4gbmV3IEVycm9yKGAke21lc3NhZ2V9XFxuVG9rZW4gY3JlYXRlZDpcXG4gICAgYXQgJHt0aGlzLnRyYWNlLmpvaW4oJ1xcbiAgICBhdCAnKX1cXG5FcnJvciB0aHJvd246YCk7XG4gIH1cbn1cblxuLyoqXG4gKiBDdXJyZW50IHJlc29sdXRpb24gY29udGV4dCBmb3IgdG9rZW5zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVJlc29sdmVDb250ZXh0IHtcbiAgLyoqXG4gICAqIFRoZSBzY29wZSBmcm9tIHdoaWNoIHJlc29sdXRpb24gaGFzIGJlZW4gaW5pdGlhdGVkXG4gICAqL1xuICByZWFkb25seSBzY29wZTogSUNvbnN0cnVjdDtcblxuICAvKipcbiAgICogUmVzb2x2ZSBhbiBpbm5lciBvYmplY3RcbiAgICovXG4gIHJlc29sdmUoeDogYW55KTogYW55O1xufVxuXG4vKipcbiAqIEEgVG9rZW4gdGhhdCBjYW4gcG9zdC1wcm9jZXNzIHRoZSBjb21wbGV0ZSByZXNvbHZlZCB2YWx1ZSwgYWZ0ZXIgcmVzb2x2ZSgpIGhhcyByZWN1cnNlZCBvdmVyIGl0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVJlc29sdmVkVmFsdWVQb3N0UHJvY2Vzc29yIHtcbiAgLyoqXG4gICAqIFByb2Nlc3MgdGhlIGNvbXBsZXRlbHkgcmVzb2x2ZWQgdmFsdWUsIGFmdGVyIGZ1bGwgcmVjdXJzaW9uL3Jlc29sdXRpb24gaGFzIGhhcHBlbmVkXG4gICAqL1xuICBwb3N0UHJvY2VzcyhpbnB1dDogYW55LCBjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpOiBhbnk7XG59XG5cbi8qKlxuICogV2hldGhlciB0aGUgZ2l2ZW4gb2JqZWN0IGlzIGFuIGBJUmVzb2x2ZWRWYWx1ZVBvc3RQcm9jZXNzb3JgXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc1Jlc29sdmVkVmFsdWVQb3N0UHJvY2Vzc29yKHg6IGFueSk6IHggaXMgSVJlc29sdmVkVmFsdWVQb3N0UHJvY2Vzc29yIHtcbiAgcmV0dXJuIHgucG9zdFByb2Nlc3MgIT09IHVuZGVmaW5lZDtcbn0iXX0=