import { IConstruct } from "./construct";
/**
 * If objects has a function property by this name, they will be considered tokens, and this
 * function will be called to resolve the value for this object.
 */
export declare const RESOLVE_METHOD = "resolve";
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
export declare class Token {
    private readonly valueOrFunction?;
    private readonly displayName?;
    /**
     * @deprecated use `Token.isToken`
     */
    static unresolved(obj: any): boolean;
    /**
     * Returns true if obj is a token (i.e. has the resolve() method or is a
     * string or array which includes token markers).
     *
     * @param obj The object to test.
     */
    static isToken(obj: any): boolean;
    /**
     * The captured stack trace which represents the location in which this token was created.
     */
    protected readonly trace: string[];
    private tokenStringification?;
    private tokenListification?;
    private tokenNumberification?;
    /**
     * Creates a token that resolves to `value`.
     *
     * If value is a function, the function is evaluated upon resolution and
     * the value it returns will be used as the token's value.
     *
     * displayName is used to represent the Token when it's embedded into a string; it
     * will look something like this:
     *
     *    "embedded in a larger string is ${Token[DISPLAY_NAME.123]}"
     *
     * This value is used as a hint to humans what the meaning of the Token is,
     * and does not have any effect on the evaluation.
     *
     * Must contain only alphanumeric and simple separator characters (_.:-).
     *
     * @param valueOrFunction What this token will evaluate to, literal or function.
     * @param displayName A human-readable display hint for this Token
     */
    constructor(valueOrFunction?: any, displayName?: string | undefined);
    /**
     * @returns The resolved value for this token.
     */
    resolve(context: IResolveContext): any;
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    toString(): string;
    /**
     * Turn this Token into JSON
     *
     * This gets called by JSON.stringify(). We want to prohibit this, because
     * it's not possible to do this properly, so we just throw an error here.
     */
    toJSON(): any;
    /**
     * Return a string list representation of this token
     *
     * Call this if the Token intrinsically evaluates to a list of strings.
     * If so, you can represent the Token in a similar way in the type
     * system.
     *
     * Note that even though the Token is represented as a list of strings, you
     * still cannot do any operations on it such as concatenation, indexing,
     * or taking its length. The only useful operations you can do to these lists
     * is constructing a `FnJoin` or a `FnSelect` on it.
     */
    toList(): string[];
    /**
     * Return a floating point representation of this Token
     *
     * Call this if the Token intrinsically resolves to something that represents
     * a number, and you need to pass it into an API that expects a number.
     *
     * You may not do any operations on the returned value; any arithmetic or
     * other operations can and probably will destroy the token-ness of the value.
     */
    toNumber(): number;
    /**
     * Creates a throwable Error object that contains the token creation stack trace.
     * @param message Error message
     */
    protected newError(message: string): any;
}
/**
 * Current resolution context for tokens
 */
export interface IResolveContext {
    /**
     * The scope from which resolution has been initiated
     */
    readonly scope: IConstruct;
    /**
     * Resolve an inner object
     */
    resolve(x: any): any;
}
/**
 * A Token that can post-process the complete resolved value, after resolve() has recursed over it
 */
export interface IResolvedValuePostProcessor {
    /**
     * Process the completely resolved value, after full recursion/resolution has happened
     */
    postProcess(input: any, context: IResolveContext): any;
}
/**
 * Whether the given object is an `IResolvedValuePostProcessor`
 */
export declare function isResolvedValuePostProcessor(x: any): x is IResolvedValuePostProcessor;
