"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./encoding");
const token_1 = require("./token");
const glob = global;
/**
 * Central place where we keep a mapping from Tokens to their String representation
 *
 * The string representation is used to embed token into strings,
 * and stored to be able to reverse that mapping.
 *
 * All instances of TokenStringMap share the same storage, so that this process
 * works even when different copies of the library are loaded.
 */
class TokenMap {
    constructor() {
        this.stringTokenMap = new Map();
        this.numberTokenMap = new Map();
        this.tokenCounter = 0;
    }
    /**
     * Singleton instance of the token string map
     */
    static instance() {
        if (!glob.__cdkTokenMap) {
            glob.__cdkTokenMap = new TokenMap();
        }
        return glob.__cdkTokenMap;
    }
    /**
     * Generate a unique string for this Token, returning a key
     *
     * Every call for the same Token will produce a new unique string, no
     * attempt is made to deduplicate. Token objects should cache the
     * value themselves, if required.
     *
     * The token can choose (part of) its own representation string with a
     * hint. This may be used to produce aesthetically pleasing and
     * recognizable token representations for humans.
     */
    registerString(token, representationHint) {
        const key = this.register(token, representationHint);
        return `${encoding_1.BEGIN_STRING_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`;
    }
    /**
     * Generate a unique string for this Token, returning a key
     */
    registerList(token, representationHint) {
        const key = this.register(token, representationHint);
        return [`${encoding_1.BEGIN_LIST_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`];
    }
    /**
     * Lookup a token from an encoded value
     */
    tokenFromEncoding(x) {
        if (typeof 'x' === 'string') {
            return this.lookupString(x);
        }
        if (Array.isArray(x)) {
            return this.lookupList(x);
        }
        if (typeof x === 'object' && x !== null && token_1.Token.isToken(x)) {
            return x;
        }
        return undefined;
    }
    /**
     * Create a unique number representation for this Token and return it
     */
    registerNumber(token) {
        const tokenIndex = this.tokenCounter++;
        this.numberTokenMap.set(tokenIndex, token);
        return encoding_1.createTokenDouble(tokenIndex);
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupString(s) {
        const str = encoding_1.TokenString.forStringToken(s);
        const fragments = str.split(this.lookupToken.bind(this));
        if (fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupList(xs) {
        if (xs.length !== 1) {
            return undefined;
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const fragments = str.split(this.lookupToken.bind(this));
        if (fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Reverse a number encoding into a Token, or undefined if the number wasn't a Token
     */
    lookupNumberToken(x) {
        const tokenIndex = encoding_1.extractTokenDouble(x);
        if (tokenIndex === undefined) {
            return undefined;
        }
        const t = this.numberTokenMap.get(tokenIndex);
        if (t === undefined) {
            throw new Error('Encoded representation of unknown number Token found');
        }
        return t;
    }
    /**
     * Find a Token by key.
     *
     * This excludes the token markers.
     */
    lookupToken(key) {
        const token = this.stringTokenMap.get(key);
        if (!token) {
            throw new Error(`Unrecognized token key: ${key}`);
        }
        return token;
    }
    register(token, representationHint) {
        const counter = this.tokenCounter++;
        const representation = (representationHint || `TOKEN`).replace(new RegExp(`[^${encoding_1.VALID_KEY_CHARS}]`, 'g'), '.');
        const key = `${representation}.${counter}`;
        this.stringTokenMap.set(key, token);
        return key;
    }
}
exports.TokenMap = TokenMap;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9rZW4tbWFwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidG9rZW4tbWFwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEseUNBQ3lGO0FBQ3pGLG1DQUFnQztBQUVoQyxNQUFNLElBQUksR0FBRyxNQUFhLENBQUM7QUFFM0I7Ozs7Ozs7O0dBUUc7QUFDSCxNQUFhLFFBQVE7SUFBckI7UUFXbUIsbUJBQWMsR0FBRyxJQUFJLEdBQUcsRUFBaUIsQ0FBQztRQUMxQyxtQkFBYyxHQUFHLElBQUksR0FBRyxFQUFpQixDQUFDO1FBQ25ELGlCQUFZLEdBQUcsQ0FBQyxDQUFDO0lBdUczQixDQUFDO0lBbkhDOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFFBQVE7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLFFBQVEsRUFBRSxDQUFDO1NBQ3JDO1FBQ0QsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDO0lBQzVCLENBQUM7SUFNRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksY0FBYyxDQUFDLEtBQVksRUFBRSxrQkFBMkI7UUFDN0QsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUNyRCxPQUFPLEdBQUcsb0NBQXlCLEdBQUcsR0FBRyxHQUFHLDJCQUFnQixFQUFFLENBQUM7SUFDakUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksWUFBWSxDQUFDLEtBQVksRUFBRSxrQkFBMkI7UUFDM0QsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUNyRCxPQUFPLENBQUMsR0FBRyxrQ0FBdUIsR0FBRyxHQUFHLEdBQUcsMkJBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7T0FFRztJQUNJLGlCQUFpQixDQUFDLENBQU07UUFDN0IsSUFBSSxPQUFPLEdBQUcsS0FBSyxRQUFRLEVBQUU7WUFBRSxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FBRTtRQUM3RCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFBRSxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FBRTtRQUNwRCxJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJLGFBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDM0QsT0FBTyxDQUFVLENBQUM7U0FDbkI7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQ7O09BRUc7SUFDSSxjQUFjLENBQUMsS0FBWTtRQUNoQyxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDdkMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQzNDLE9BQU8sNEJBQWlCLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksWUFBWSxDQUFDLENBQVM7UUFDM0IsTUFBTSxHQUFHLEdBQUcsc0JBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDMUMsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ3pELElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDMUIsT0FBTyxTQUFTLENBQUMsVUFBVSxDQUFDO1NBQzdCO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksVUFBVSxDQUFDLEVBQVk7UUFDNUIsSUFBSSxFQUFFLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUFFLE9BQU8sU0FBUyxDQUFDO1NBQUU7UUFDMUMsTUFBTSxHQUFHLEdBQUcsc0JBQVcsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUMsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ3pELElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDMUIsT0FBTyxTQUFTLENBQUMsVUFBVSxDQUFDO1NBQzdCO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksaUJBQWlCLENBQUMsQ0FBUztRQUNoQyxNQUFNLFVBQVUsR0FBRyw2QkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN6QyxJQUFJLFVBQVUsS0FBSyxTQUFTLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBQ25ELE1BQU0sQ0FBQyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxLQUFLLFNBQVMsRUFBRTtZQUFFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztTQUFFO1FBQ2pHLE9BQU8sQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsR0FBVztRQUM1QixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsR0FBRyxFQUFFLENBQUMsQ0FBQztTQUNuRDtRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVPLFFBQVEsQ0FBQyxLQUFZLEVBQUUsa0JBQTJCO1FBQ3hELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUNwQyxNQUFNLGNBQWMsR0FBRyxDQUFDLGtCQUFrQixJQUFJLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxLQUFLLDBCQUFlLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUM5RyxNQUFNLEdBQUcsR0FBRyxHQUFHLGNBQWMsSUFBSSxPQUFPLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDcEMsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUFwSEQsNEJBb0hDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQkVHSU5fTElTVF9UT0tFTl9NQVJLRVIsIEJFR0lOX1NUUklOR19UT0tFTl9NQVJLRVIsIGNyZWF0ZVRva2VuRG91YmxlLFxuICBFTkRfVE9LRU5fTUFSS0VSLCBleHRyYWN0VG9rZW5Eb3VibGUsIFRva2VuU3RyaW5nLCBWQUxJRF9LRVlfQ0hBUlMgfSBmcm9tIFwiLi9lbmNvZGluZ1wiO1xuaW1wb3J0IHsgVG9rZW4gfSBmcm9tIFwiLi90b2tlblwiO1xuXG5jb25zdCBnbG9iID0gZ2xvYmFsIGFzIGFueTtcblxuLyoqXG4gKiBDZW50cmFsIHBsYWNlIHdoZXJlIHdlIGtlZXAgYSBtYXBwaW5nIGZyb20gVG9rZW5zIHRvIHRoZWlyIFN0cmluZyByZXByZXNlbnRhdGlvblxuICpcbiAqIFRoZSBzdHJpbmcgcmVwcmVzZW50YXRpb24gaXMgdXNlZCB0byBlbWJlZCB0b2tlbiBpbnRvIHN0cmluZ3MsXG4gKiBhbmQgc3RvcmVkIHRvIGJlIGFibGUgdG8gcmV2ZXJzZSB0aGF0IG1hcHBpbmcuXG4gKlxuICogQWxsIGluc3RhbmNlcyBvZiBUb2tlblN0cmluZ01hcCBzaGFyZSB0aGUgc2FtZSBzdG9yYWdlLCBzbyB0aGF0IHRoaXMgcHJvY2Vzc1xuICogd29ya3MgZXZlbiB3aGVuIGRpZmZlcmVudCBjb3BpZXMgb2YgdGhlIGxpYnJhcnkgYXJlIGxvYWRlZC5cbiAqL1xuZXhwb3J0IGNsYXNzIFRva2VuTWFwIHtcbiAgLyoqXG4gICAqIFNpbmdsZXRvbiBpbnN0YW5jZSBvZiB0aGUgdG9rZW4gc3RyaW5nIG1hcFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpbnN0YW5jZSgpOiBUb2tlbk1hcCB7XG4gICAgaWYgKCFnbG9iLl9fY2RrVG9rZW5NYXApIHtcbiAgICAgIGdsb2IuX19jZGtUb2tlbk1hcCA9IG5ldyBUb2tlbk1hcCgpO1xuICAgIH1cbiAgICByZXR1cm4gZ2xvYi5fX2Nka1Rva2VuTWFwO1xuICB9XG5cbiAgcHJpdmF0ZSByZWFkb25seSBzdHJpbmdUb2tlbk1hcCA9IG5ldyBNYXA8c3RyaW5nLCBUb2tlbj4oKTtcbiAgcHJpdmF0ZSByZWFkb25seSBudW1iZXJUb2tlbk1hcCA9IG5ldyBNYXA8bnVtYmVyLCBUb2tlbj4oKTtcbiAgcHJpdmF0ZSB0b2tlbkNvdW50ZXIgPSAwO1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhIHVuaXF1ZSBzdHJpbmcgZm9yIHRoaXMgVG9rZW4sIHJldHVybmluZyBhIGtleVxuICAgKlxuICAgKiBFdmVyeSBjYWxsIGZvciB0aGUgc2FtZSBUb2tlbiB3aWxsIHByb2R1Y2UgYSBuZXcgdW5pcXVlIHN0cmluZywgbm9cbiAgICogYXR0ZW1wdCBpcyBtYWRlIHRvIGRlZHVwbGljYXRlLiBUb2tlbiBvYmplY3RzIHNob3VsZCBjYWNoZSB0aGVcbiAgICogdmFsdWUgdGhlbXNlbHZlcywgaWYgcmVxdWlyZWQuXG4gICAqXG4gICAqIFRoZSB0b2tlbiBjYW4gY2hvb3NlIChwYXJ0IG9mKSBpdHMgb3duIHJlcHJlc2VudGF0aW9uIHN0cmluZyB3aXRoIGFcbiAgICogaGludC4gVGhpcyBtYXkgYmUgdXNlZCB0byBwcm9kdWNlIGFlc3RoZXRpY2FsbHkgcGxlYXNpbmcgYW5kXG4gICAqIHJlY29nbml6YWJsZSB0b2tlbiByZXByZXNlbnRhdGlvbnMgZm9yIGh1bWFucy5cbiAgICovXG4gIHB1YmxpYyByZWdpc3RlclN0cmluZyh0b2tlbjogVG9rZW4sIHJlcHJlc2VudGF0aW9uSGludD86IHN0cmluZyk6IHN0cmluZyB7XG4gICAgY29uc3Qga2V5ID0gdGhpcy5yZWdpc3Rlcih0b2tlbiwgcmVwcmVzZW50YXRpb25IaW50KTtcbiAgICByZXR1cm4gYCR7QkVHSU5fU1RSSU5HX1RPS0VOX01BUktFUn0ke2tleX0ke0VORF9UT0tFTl9NQVJLRVJ9YDtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhIHVuaXF1ZSBzdHJpbmcgZm9yIHRoaXMgVG9rZW4sIHJldHVybmluZyBhIGtleVxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyTGlzdCh0b2tlbjogVG9rZW4sIHJlcHJlc2VudGF0aW9uSGludD86IHN0cmluZyk6IHN0cmluZ1tdIHtcbiAgICBjb25zdCBrZXkgPSB0aGlzLnJlZ2lzdGVyKHRva2VuLCByZXByZXNlbnRhdGlvbkhpbnQpO1xuICAgIHJldHVybiBbYCR7QkVHSU5fTElTVF9UT0tFTl9NQVJLRVJ9JHtrZXl9JHtFTkRfVE9LRU5fTUFSS0VSfWBdO1xuICB9XG5cbiAgLyoqXG4gICAqIExvb2t1cCBhIHRva2VuIGZyb20gYW4gZW5jb2RlZCB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHRva2VuRnJvbUVuY29kaW5nKHg6IGFueSk6IFRva2VuIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAodHlwZW9mICd4JyA9PT0gJ3N0cmluZycpIHsgcmV0dXJuIHRoaXMubG9va3VwU3RyaW5nKHgpOyB9XG4gICAgaWYgKEFycmF5LmlzQXJyYXkoeCkpIHsgcmV0dXJuIHRoaXMubG9va3VwTGlzdCh4KTsgfVxuICAgIGlmICh0eXBlb2YgeCA9PT0gJ29iamVjdCcgJiYgeCAhPT0gbnVsbCAmJiBUb2tlbi5pc1Rva2VuKHgpKSB7XG4gICAgICByZXR1cm4geCBhcyBUb2tlbjtcbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSB1bmlxdWUgbnVtYmVyIHJlcHJlc2VudGF0aW9uIGZvciB0aGlzIFRva2VuIGFuZCByZXR1cm4gaXRcbiAgICovXG4gIHB1YmxpYyByZWdpc3Rlck51bWJlcih0b2tlbjogVG9rZW4pOiBudW1iZXIge1xuICAgIGNvbnN0IHRva2VuSW5kZXggPSB0aGlzLnRva2VuQ291bnRlcisrO1xuICAgIHRoaXMubnVtYmVyVG9rZW5NYXAuc2V0KHRva2VuSW5kZXgsIHRva2VuKTtcbiAgICByZXR1cm4gY3JlYXRlVG9rZW5Eb3VibGUodG9rZW5JbmRleCk7XG4gIH1cblxuICAvKipcbiAgICogUmV2ZXJzZSBhIHN0cmluZyByZXByZXNlbnRhdGlvbiBpbnRvIGEgVG9rZW4gb2JqZWN0XG4gICAqL1xuICBwdWJsaWMgbG9va3VwU3RyaW5nKHM6IHN0cmluZyk6IFRva2VuIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBzdHIgPSBUb2tlblN0cmluZy5mb3JTdHJpbmdUb2tlbihzKTtcbiAgICBjb25zdCBmcmFnbWVudHMgPSBzdHIuc3BsaXQodGhpcy5sb29rdXBUb2tlbi5iaW5kKHRoaXMpKTtcbiAgICBpZiAoZnJhZ21lbnRzLmxlbmd0aCA9PT0gMSkge1xuICAgICAgcmV0dXJuIGZyYWdtZW50cy5maXJzdFRva2VuO1xuICAgIH1cbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldmVyc2UgYSBzdHJpbmcgcmVwcmVzZW50YXRpb24gaW50byBhIFRva2VuIG9iamVjdFxuICAgKi9cbiAgcHVibGljIGxvb2t1cExpc3QoeHM6IHN0cmluZ1tdKTogVG9rZW4gfCB1bmRlZmluZWQge1xuICAgIGlmICh4cy5sZW5ndGggIT09IDEpIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuICAgIGNvbnN0IHN0ciA9IFRva2VuU3RyaW5nLmZvckxpc3RUb2tlbih4c1swXSk7XG4gICAgY29uc3QgZnJhZ21lbnRzID0gc3RyLnNwbGl0KHRoaXMubG9va3VwVG9rZW4uYmluZCh0aGlzKSk7XG4gICAgaWYgKGZyYWdtZW50cy5sZW5ndGggPT09IDEpIHtcbiAgICAgIHJldHVybiBmcmFnbWVudHMuZmlyc3RUb2tlbjtcbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXZlcnNlIGEgbnVtYmVyIGVuY29kaW5nIGludG8gYSBUb2tlbiwgb3IgdW5kZWZpbmVkIGlmIHRoZSBudW1iZXIgd2Fzbid0IGEgVG9rZW5cbiAgICovXG4gIHB1YmxpYyBsb29rdXBOdW1iZXJUb2tlbih4OiBudW1iZXIpOiBUb2tlbiB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgdG9rZW5JbmRleCA9IGV4dHJhY3RUb2tlbkRvdWJsZSh4KTtcbiAgICBpZiAodG9rZW5JbmRleCA9PT0gdW5kZWZpbmVkKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgICBjb25zdCB0ID0gdGhpcy5udW1iZXJUb2tlbk1hcC5nZXQodG9rZW5JbmRleCk7XG4gICAgaWYgKHQgPT09IHVuZGVmaW5lZCkgeyB0aHJvdyBuZXcgRXJyb3IoJ0VuY29kZWQgcmVwcmVzZW50YXRpb24gb2YgdW5rbm93biBudW1iZXIgVG9rZW4gZm91bmQnKTsgfVxuICAgIHJldHVybiB0O1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbmQgYSBUb2tlbiBieSBrZXkuXG4gICAqXG4gICAqIFRoaXMgZXhjbHVkZXMgdGhlIHRva2VuIG1hcmtlcnMuXG4gICAqL1xuICBwdWJsaWMgbG9va3VwVG9rZW4oa2V5OiBzdHJpbmcpOiBUb2tlbiB7XG4gICAgY29uc3QgdG9rZW4gPSB0aGlzLnN0cmluZ1Rva2VuTWFwLmdldChrZXkpO1xuICAgIGlmICghdG9rZW4pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5yZWNvZ25pemVkIHRva2VuIGtleTogJHtrZXl9YCk7XG4gICAgfVxuICAgIHJldHVybiB0b2tlbjtcbiAgfVxuXG4gIHByaXZhdGUgcmVnaXN0ZXIodG9rZW46IFRva2VuLCByZXByZXNlbnRhdGlvbkhpbnQ/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGNvbnN0IGNvdW50ZXIgPSB0aGlzLnRva2VuQ291bnRlcisrO1xuICAgIGNvbnN0IHJlcHJlc2VudGF0aW9uID0gKHJlcHJlc2VudGF0aW9uSGludCB8fCBgVE9LRU5gKS5yZXBsYWNlKG5ldyBSZWdFeHAoYFteJHtWQUxJRF9LRVlfQ0hBUlN9XWAsICdnJyksICcuJyk7XG4gICAgY29uc3Qga2V5ID0gYCR7cmVwcmVzZW50YXRpb259LiR7Y291bnRlcn1gO1xuICAgIHRoaXMuc3RyaW5nVG9rZW5NYXAuc2V0KGtleSwgdG9rZW4pO1xuICAgIHJldHVybiBrZXk7XG4gIH1cbn0iXX0=