"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const app_1 = require("./app");
const construct_1 = require("./construct");
const logical_id_1 = require("./logical-id");
const uniqueid_1 = require("./uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/cdk.Stack');
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually a Program instance.
     * @param name The name of the CloudFormation stack. Defaults to "Stack".
     * @param props Stack properties.
     */
    constructor(scope, name, props = {}) {
        // For unit test convenience parents are optional, so bypass the type check when calling the parent.
        super(scope, name);
        /**
         * Lists all missing contextual information.
         * This is returned when the stack is synthesized under the 'missing' attribute
         * and allows tooling to obtain the context and re-synthesize.
         */
        this.missingContext = {};
        /**
         * Options for CloudFormation template (like version, transform, description).
         */
        this.templateOptions = {};
        /**
         * Other stacks this stack depends on
         */
        this.stackDependencies = new Set();
        /**
         * Values set for parameters in cloud assembly.
         */
        this.parameterValues = {};
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this.configuredEnv = props.env || {};
        this.env = this.parseEnvironment(props.env);
        this.logicalIds = new logical_id_1.LogicalIDs(props && props.namingScheme ? props.namingScheme : new logical_id_1.HashedAddressingScheme());
        this.name = props.stackName !== undefined ? props.stackName : this.calculateStackName();
        this.autoDeploy = props && props.autoDeploy === false ? false : true;
        if (!Stack.VALID_STACK_NAME_REGEX.test(this.name)) {
            throw new Error(`Stack name must match the regular expression: ${Stack.VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Adds a metadata annotation "aws:cdk:physical-name" to the construct if physicalName
     * is non-null. This can be used later by tools and aspects to determine if resources
     * have been created with physical names.
     */
    static annotatePhysicalName(construct, physicalName) {
        if (physicalName == null) {
            return;
        }
        construct.node.addMetadata('aws:cdk:physical-name', physicalName);
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(obj) {
        return obj[STACK_SYMBOL] === true;
    }
    /**
     * Returns the environment specification for this stack (aws://account/region).
     */
    get environment() {
        const account = this.env.account || 'unknown-account';
        const region = this.env.region || 'unknown-region';
        return cxapi.EnvironmentUtils.format(account, region);
    }
    /**
     * Looks up a resource by path.
     *
     * @returns The Resource or undefined if not found
     */
    findResource(constructPath) {
        const r = this.node.findChild(constructPath);
        if (!r) {
            return undefined;
        }
        // found an element, check if it's a resource (duck-type)
        if (!('resourceType' in r)) {
            throw new Error(`Found a stack element for ${constructPath} but it is not a resource: ${r.toString()}`);
        }
        return r;
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        // before we begin synthesis, we shall lock this stack, so children cannot be added
        this.node.lock();
        try {
            const template = {
                Description: this.templateOptions.description,
                Transform: this.templateOptions.transform,
                AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
                Metadata: this.templateOptions.metadata
            };
            const elements = cfnElements(this);
            const fragments = elements.map(e => this.node.resolve(e._toCloudFormation()));
            // merge in all CloudFormation fragments collected from the tree
            for (const fragment of fragments) {
                merge(template, fragment);
            }
            // resolve all tokens and remove all empties
            const ret = this.node.resolve(template) || {};
            this.logicalIds.assertAllRenamesApplied();
            return ret;
        }
        finally {
            // allow mutations after synthesis is finished.
            this.node.unlock();
        }
    }
    /**
     * @param why more information about why region is required.
     * @returns The region in which this stack is deployed. Throws if region is not defined.
     */
    requireRegion(why) {
        if (!this.env.region) {
            throw new Error(`${why ? why + '. ' : ''}Stack requires region information. It can be either supplied via the "env" property, ` +
                `via the "${cxapi.DEFAULT_REGION_CONTEXT_KEY}" context parameters or using "aws configure"`);
        }
        return this.env.region;
    }
    /**
     * Returns the AWS account ID of this Stack,
     * or throws an exception if the account ID is not set in the environment.
     *
     * @param why more information about why is the account ID required
     * @returns the AWS account ID of this Stack
     */
    requireAccountId(why) {
        if (!this.env.account) {
            throw new Error(`${why ? why + '. ' : ''}Stack requires account information. ` +
                'It can be supplied either via the "env" property when creating the Stack, or by using "aws configure"');
        }
        return this.env.account;
    }
    parentApp() {
        const parent = this.node.scope;
        return parent instanceof app_1.App
            ? parent
            : undefined;
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions on how the key should be supplied.
     * @param key Key that uniquely identifies this missing context.
     * @param details The set of parameters needed to obtain the context (specific to context provider).
     */
    reportMissingContext(key, details) {
        this.missingContext[key] = details;
    }
    /**
     * Rename a generated logical identities
     */
    renameLogical(oldId, newId) {
        if (this.node.children.length > 0) {
            throw new Error("All renames must be set up before adding elements to the stack");
        }
        this.logicalIds.renameLogical(oldId, newId);
    }
    /**
     * Add a dependency between this stack and another stack
     */
    addDependency(stack, reason) {
        if (stack === this) {
            return;
        } // Can ignore a dependency on self
        reason = reason || 'dependency added using stack.addDependency()';
        const dep = stack.stackDependencyReasons(this);
        if (dep !== undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`'${stack.node.path}' depends on '${this.node.path}' (${dep.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        this.stackDependencies.add({ stack, reason });
    }
    /**
     * Return the stacks this stack depends on
     */
    dependencies() {
        return Array.from(this.stackDependencies.values()).map(d => d.stack);
    }
    /**
     * The account in which this stack is defined
     *
     * Either returns the literal account for this stack if it was specified
     * literally upon Stack construction, or a symbolic value that will evaluate
     * to the correct account at deployment time.
     */
    get accountId() {
        if (this.configuredEnv.account) {
            return this.configuredEnv.account;
        }
        // Does not need to be scoped, the only situation in which
        // Export/Fn::ImportValue would work if { Ref: "AWS::AccountId" } is the
        // same for provider and consumer anyway.
        return pseudo_1.Aws.accountId;
    }
    /**
     * The region in which this stack is defined
     *
     * Either returns the literal region for this stack if it was specified
     * literally upon Stack construction, or a symbolic value that will evaluate
     * to the correct region at deployment time.
     */
    get region() {
        if (this.configuredEnv.region) {
            return this.configuredEnv.region;
        }
        // Does not need to be scoped, the only situation in which
        // Export/Fn::ImportValue would work if { Ref: "AWS::AccountId" } is the
        // same for provider and consumer anyway.
        return pseudo_1.Aws.region;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return pseudo_1.Aws.partition;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        return new pseudo_1.ScopedAws(this).urlSuffix;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * The name of the stack currently being deployed
     *
     * Only available at deployment time; this will always return an unresolved value.
     */
    get stackName() {
        return new pseudo_1.ScopedAws(this).stackName;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.arnFromComponents(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.parseArn(arn, sepIfToken, hasName);
    }
    /**
     * Sets the value of a CloudFormation parameter.
     * @param parameter The parameter to set the value for
     * @param value The value, can use `${}` notation to reference other assembly block attributes.
     */
    setParameterValue(parameter, value) {
        this.parameterValues[parameter.logicalId] = value;
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !Stack.VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${Stack.VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Prepare stack
     *
     * Find all CloudFormation references and tell them we're consuming them.
     *
     * Find all dependencies as well and add the appropriate DependsOn fields.
     */
    prepare() {
        // References
        for (const ref of this.node.findReferences()) {
            if (cfn_reference_1.CfnReference.isCfnReference(ref.reference)) {
                ref.reference.consumeFromStack(this, ref.source);
            }
        }
        // Resource dependencies
        for (const dependency of this.node.findDependencies()) {
            const theirStack = dependency.target.node.stack;
            if (theirStack !== undefined && theirStack !== this) {
                this.addDependency(theirStack);
            }
            else {
                for (const target of findResources([dependency.target])) {
                    for (const source of findResources([dependency.source])) {
                        source.addDependsOn(target);
                    }
                }
            }
        }
    }
    synthesize(builder) {
        const template = `${this.name}.template.json`;
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, template);
        fs.writeFileSync(outPath, JSON.stringify(this._toCloudFormation(), undefined, 2));
        const deps = this.dependencies().map(s => s.name);
        const meta = this.collectMetadata();
        const properties = {
            templateFile: template,
            parameters: Object.keys(this.parameterValues).length > 0 ? this.node.resolve(this.parameterValues) : undefined
        };
        // add an artifact that represents this stack
        builder.addArtifact(this.name, {
            type: cxapi.ArtifactType.AwsCloudFormationStack,
            environment: this.environment,
            properties,
            autoDeploy: this.autoDeploy ? undefined : false,
            dependencies: deps.length > 0 ? deps : undefined,
            metadata: Object.keys(meta).length > 0 ? meta : undefined,
            missing: this.missingContext && Object.keys(this.missingContext).length > 0 ? this.missingContext : undefined
        });
    }
    /**
     * Applied defaults to environment attributes.
     */
    parseEnvironment(env = {}) {
        return {
            account: env.account ? env.account : this.node.getContext(cxapi.DEFAULT_ACCOUNT_CONTEXT_KEY),
            region: env.region ? env.region : this.node.getContext(cxapi.DEFAULT_REGION_CONTEXT_KEY)
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of this.stackDependencies) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [dep.reason].concat(ret);
            }
        }
        return undefined;
    }
    collectMetadata() {
        const output = {};
        visit(this);
        const app = this.parentApp();
        if (app && app.node.metadata.length > 0) {
            output[construct_1.PATH_SEP] = app.node.metadata;
        }
        return output;
        function visit(node) {
            if (node.node.metadata.length > 0) {
                // Make the path absolute
                output[construct_1.PATH_SEP + node.node.path] = node.node.metadata.map(md => node.node.resolve(md));
            }
            for (const child of node.node.children) {
                visit(child);
            }
        }
    }
    /**
     * Calculcate the stack name based on the construct path
     */
    calculateStackName() {
        // In tests, it's possible for this stack to be the root object, in which case
        // we need to use it as part of the root path.
        const rootPath = this.node.scope !== undefined ? this.node.ancestors().slice(1) : [this];
        const ids = rootPath.map(c => c.node.id);
        // Special case, if rootPath is length 1 then just use ID (backwards compatibility)
        // otherwise use a unique stack name (including hash). This logic is already
        // in makeUniqueId, *however* makeUniqueId will also strip dashes from the name,
        // which *are* allowed and also used, so we short-circuit it.
        if (ids.length === 1) {
            // Could be empty in a unit test, so just pretend it's named "Stack" then
            return ids[0] || 'Stack';
        }
        return uniqueid_1.makeUniqueId(ids);
    }
}
Stack.VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
exports.Stack = Stack;
function merge(template, part) {
    for (const section of Object.keys(part)) {
        const src = part[section];
        // create top-level section if it doesn't exist
        let dest = template[section];
        if (!dest) {
            template[section] = dest = src;
        }
        else {
            // add all entities from source section to destination section
            for (const id of Object.keys(src)) {
                if (id in dest) {
                    throw new Error(`section '${section}' already contains '${id}'`);
                }
                dest[id] = src[id];
            }
        }
    }
}
/**
 * Collect all CfnElements from a Stack
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
// These imports have to be at the end to prevent circular imports
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./cfn-reference");
const cfn_resource_1 = require("./cfn-resource");
const pseudo_1 = require("./pseudo");
/**
 * Find all resources in a set of constructs
 */
function findResources(roots) {
    const ret = new Array();
    for (const root of roots) {
        ret.push(...root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource));
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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