import cxapi = require('@aws-cdk/cx-api');
import { App } from './app';
import { CfnParameter } from './cfn-parameter';
import { Construct } from './construct';
import { Environment } from './environment';
import { IAddressingScheme, LogicalIDs } from './logical-id';
export interface StackProps {
    /**
     * The AWS environment (account/region) where this stack will be deployed.
     *
     * @default - The `default-account` and `default-region` context parameters will be
     * used. If they are undefined, it will not be possible to deploy the stack.
     */
    readonly env?: Environment;
    /**
     * Name to deploy the stack with
     *
     * @default - Derived from construct path.
     */
    readonly stackName?: string;
    /**
     * Strategy for logical ID generation
     *
     * @default - The HashedNamingScheme will be used.
     */
    readonly namingScheme?: IAddressingScheme;
    /**
     * Should the Stack be deployed when running `cdk deploy` without arguments
     * (and listed when running `cdk synth` without arguments).
     * Setting this to `false` is useful when you have a Stack in your CDK app
     * that you don't want to deploy using the CDK toolkit -
     * for example, because you're planning on deploying it through CodePipeline.
     *
     * @default true
     */
    readonly autoDeploy?: boolean;
}
/**
 * A root construct which represents a single CloudFormation stack.
 */
export declare class Stack extends Construct {
    /**
     * Adds a metadata annotation "aws:cdk:physical-name" to the construct if physicalName
     * is non-null. This can be used later by tools and aspects to determine if resources
     * have been created with physical names.
     */
    static annotatePhysicalName(construct: Construct, physicalName?: string): void;
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(obj: any): obj is Stack;
    private static readonly VALID_STACK_NAME_REGEX;
    /**
     * Lists all missing contextual information.
     * This is returned when the stack is synthesized under the 'missing' attribute
     * and allows tooling to obtain the context and re-synthesize.
     */
    readonly missingContext: {
        [key: string]: cxapi.MissingContext;
    };
    /**
     * The environment in which this stack is deployed.
     */
    readonly env: Environment;
    /**
     * Logical ID generation strategy
     */
    readonly logicalIds: LogicalIDs;
    /**
     * Options for CloudFormation template (like version, transform, description).
     */
    readonly templateOptions: ITemplateOptions;
    /**
     * The CloudFormation stack name.
     *
     * This is the stack name either configuration via the `stackName` property
     * or automatically derived from the construct path.
     */
    readonly name: string;
    /**
     * Should the Stack be deployed when running `cdk deploy` without arguments
     * (and listed when running `cdk synth` without arguments).
     * Setting this to `false` is useful when you have a Stack in your CDK app
     * that you don't want to deploy using the CDK toolkit -
     * for example, because you're planning on deploying it through CodePipeline.
     *
     * By default, this is `true`.
     */
    readonly autoDeploy: boolean;
    /**
     * Other stacks this stack depends on
     */
    private readonly stackDependencies;
    /**
     * Values set for parameters in cloud assembly.
     */
    private readonly parameterValues;
    /**
     * Environment as configured via props
     *
     * (Both on Stack and inherited from App)
     */
    private readonly configuredEnv;
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually a Program instance.
     * @param name The name of the CloudFormation stack. Defaults to "Stack".
     * @param props Stack properties.
     */
    constructor(scope?: Construct, name?: string, props?: StackProps);
    /**
     * Returns the environment specification for this stack (aws://account/region).
     */
    readonly environment: string;
    /**
     * Looks up a resource by path.
     *
     * @returns The Resource or undefined if not found
     */
    findResource(constructPath: string): CfnResource | undefined;
    /**
     * @param why more information about why region is required.
     * @returns The region in which this stack is deployed. Throws if region is not defined.
     */
    requireRegion(why?: string): string;
    /**
     * Returns the AWS account ID of this Stack,
     * or throws an exception if the account ID is not set in the environment.
     *
     * @param why more information about why is the account ID required
     * @returns the AWS account ID of this Stack
     */
    requireAccountId(why?: string): string;
    parentApp(): App | undefined;
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions on how the key should be supplied.
     * @param key Key that uniquely identifies this missing context.
     * @param details The set of parameters needed to obtain the context (specific to context provider).
     */
    reportMissingContext(key: string, details: cxapi.MissingContext): void;
    /**
     * Rename a generated logical identities
     */
    renameLogical(oldId: string, newId: string): void;
    /**
     * Add a dependency between this stack and another stack
     */
    addDependency(stack: Stack, reason?: string): void;
    /**
     * Return the stacks this stack depends on
     */
    dependencies(): Stack[];
    /**
     * The account in which this stack is defined
     *
     * Either returns the literal account for this stack if it was specified
     * literally upon Stack construction, or a symbolic value that will evaluate
     * to the correct account at deployment time.
     */
    readonly accountId: string;
    /**
     * The region in which this stack is defined
     *
     * Either returns the literal region for this stack if it was specified
     * literally upon Stack construction, or a symbolic value that will evaluate
     * to the correct region at deployment time.
     */
    readonly region: string;
    /**
     * The partition in which this stack is defined
     */
    readonly partition: string;
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    readonly urlSuffix: string;
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    readonly stackId: string;
    /**
     * The name of the stack currently being deployed
     *
     * Only available at deployment time; this will always return an unresolved value.
     */
    readonly stackName: string;
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    readonly notificationArns: string[];
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components: ArnComponents): string;
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn: string, sepIfToken?: string, hasName?: boolean): ArnComponents;
    /**
     * Sets the value of a CloudFormation parameter.
     * @param parameter The parameter to set the value for
     * @param value The value, can use `${}` notation to reference other assembly block attributes.
     */
    setParameterValue(parameter: CfnParameter, value: string): void;
    /**
     * Prepare stack
     *
     * Find all CloudFormation references and tell them we're consuming them.
     *
     * Find all dependencies as well and add the appropriate DependsOn fields.
     */
    protected prepare(): void;
    protected synthesize(builder: cxapi.CloudAssemblyBuilder): void;
    /**
     * Applied defaults to environment attributes.
     */
    private parseEnvironment;
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    private stackDependencyReasons;
    private collectMetadata;
    /**
     * Calculcate the stack name based on the construct path
     */
    private calculateStackName;
}
/**
 * CloudFormation template options for a stack.
 */
export interface ITemplateOptions {
    /**
     * Gets or sets the description of this stack.
     * If provided, it will be included in the CloudFormation template's "Description" attribute.
     */
    description?: string;
    /**
     * Gets or sets the AWSTemplateFormatVersion field of the CloudFormation template.
     */
    templateFormatVersion?: string;
    /**
     * Gets or sets the top-level template transform for this stack (e.g. "AWS::Serverless-2016-10-31").
     */
    transform?: string;
    /**
     * Metadata associated with the CloudFormation template.
     */
    metadata?: {
        [key: string]: any;
    };
}
import { ArnComponents } from './arn';
import { CfnResource } from './cfn-resource';
