"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const token_1 = require("./token");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainTex()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends token_1.Token {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error(`secretId cannot be empty`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || ''
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SecretsManager, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SsmSecure, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(() => ref.toString());
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error(`CloudFormation parameter must be configured with "NoEcho"`);
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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