"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./encoding");
const token_1 = require("./token");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        return {
            scope: options.scope,
            resolve(x) { return resolve(x, { ...options, prefix: newPrefix }); }
        };
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        const str = encoding_1.TokenString.forStringToken(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return options.resolver.resolveString(fragments, makeContext());
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return resolveNumberToken(obj, makeContext());
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext());
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`).resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        return options.resolver.resolveToken(obj, makeContext());
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext().resolve(key);
        if (typeof (resolvedKey) !== 'string') {
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}`);
        }
        const value = makeContext(key).resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context) {
        let resolved = t[token_1.RESOLVE_METHOD](context);
        // The token might have returned more values that need resolving, recurse
        resolved = context.resolve(resolved);
        if (token_1.isResolvedValuePostProcessor(t)) {
            resolved = t.postProcess(resolved, context);
        }
        return resolved;
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new StringConcat());
    resolve(fn(), { scope, prefix: [], resolver });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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