import { IConstruct } from './construct';
import { TokenizedStringFragments } from './string-fragments';
import { IResolveContext, Token } from "./token";
/**
 * Options to the resolve() operation
 *
 * NOT the same as the ResolveContext; ResolveContext is exposed to Token
 * implementors and resolution hooks, whereas this struct is just to bundle
 * a number of things that would otherwise be arguments to resolve() in a
 * readable way.
 */
export interface IResolveOptions {
    scope: IConstruct;
    resolver: ITokenResolver;
    prefix?: string[];
}
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
export declare function resolve(obj: any, options: IResolveOptions): any;
/**
 * How to resolve tokens
 */
export interface ITokenResolver {
    /**
     * Resolve a single token
     */
    resolveToken(t: Token, context: IResolveContext): any;
    /**
     * Resolve a string with at least one stringified token in it
     *
     * (May use concatenation)
     */
    resolveString(s: TokenizedStringFragments, context: IResolveContext): any;
    /**
     * Resolve a tokenized list
     */
    resolveList(l: string[], context: IResolveContext): any;
}
/**
 * Function used to concatenate symbols in the target document language
 *
 * Interface so it could potentially be exposed over jsii.
 */
export interface IFragmentConcatenator {
    /**
     * Join the fragment on the left and on the right
     */
    join(left: any | undefined, right: any | undefined): any;
}
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
export declare class StringConcat implements IFragmentConcatenator {
    join(left: any | undefined, right: any | undefined): any;
}
/**
 * Default resolver implementation
 */
export declare class DefaultTokenResolver implements ITokenResolver {
    private readonly concat;
    constructor(concat: IFragmentConcatenator);
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t: Token, context: IResolveContext): any;
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments: TokenizedStringFragments, context: IResolveContext): any;
    resolveList(xs: string[], context: IResolveContext): any;
}
/**
 * Find all Tokens that are used in the given structure
 */
export declare function findTokens(scope: IConstruct, fn: () => any): Token[];
/**
 * Remember all Tokens encountered while resolving
 */
export declare class RememberingTokenResolver extends DefaultTokenResolver {
    private readonly tokensSeen;
    resolveToken(t: Token, context: IResolveContext): any;
    resolveString(s: TokenizedStringFragments, context: IResolveContext): any;
    readonly tokens: Token[];
}
