"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const resource_policy_1 = require("./resource-policy");
var RemovalPolicy;
(function (RemovalPolicy) {
    /**
     * This is the default removal policy for most resources. It means that when the resource
     * is removed from the app, it will be physically destroyed.
     */
    RemovalPolicy[RemovalPolicy["Destroy"] = 0] = "Destroy";
    /**
     * This uses the 'Retain' DeletionPolicy, which will cause the resource to be retained
     * in the account, but orphaned from the stack.
     */
    RemovalPolicy[RemovalPolicy["Orphan"] = 1] = "Orphan";
    /**
     * This will apply the 'Retain' DeletionPolicy and also add metadata for the toolkit
     * to apply a CloudFormation stack policy which forbids the deletion of resource.
     */
    RemovalPolicy[RemovalPolicy["Forbid"] = 2] = "Forbid";
})(RemovalPolicy = exports.RemovalPolicy || (exports.RemovalPolicy = {}));
function applyRemovalPolicy(resource, removalPolicy) {
    if (removalPolicy === RemovalPolicy.Orphan || removalPolicy === RemovalPolicy.Forbid) {
        resource.options.deletionPolicy = resource_policy_1.DeletionPolicy.Retain;
    }
    // attach metadata that will tell the toolkit to protect this resource by
    // applying an appropriate stack update policy.
    if (removalPolicy === RemovalPolicy.Forbid) {
        resource.node.addMetadata('aws:cdk:protected', true);
    }
}
exports.applyRemovalPolicy = applyRemovalPolicy;
//# sourceMappingURL=data:application/json;base64,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