"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const uniqueid_1 = require("./uniqueid");
const PATH_SEP = '/';
/**
 * Renders a hashed ID for a resource.
 *
 * In order to make sure logical IDs are unique and stable, we hash the resource
 * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
 * a suffix to the path components joined without a separator (CloudFormation
 * IDs only allow alphanumeric characters).
 *
 * The result will be:
 *
 *   <path.join('')><md5(path.join('/')>
 *     "human"      "hash"
 *
 * If the "human" part of the ID exceeds 240 characters, we simply trim it so
 * the total ID doesn't exceed CloudFormation's 255 character limit.
 *
 * We only take 8 characters from the md5 hash (0.000005 chance of collision).
 *
 * Special cases:
 *
 * - If the path only contains a single component (i.e. it's a top-level
 *   resource), we won't add the hash to it. The hash is not needed for
 *   disamiguation and also, it allows for a more straightforward migration an
 *   existing CloudFormation template to a CDK stack without logical ID changes
 *   (or renames).
 * - For aesthetic reasons, if the last components of the path are the same
 *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
 *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
 *   instead of `L1L2PipelinePipeline<HASH>`
 * - If a component is named "Default" it will be omitted from the path. This
 *   allows refactoring higher level abstractions around constructs without affecting
 *   the IDs of already deployed resources.
 * - If a component is named "Resource" it will be omitted from the user-visible
 *   path, but included in the hash. This reduces visual noise in the human readable
 *   part of the identifier.
 */
class HashedAddressingScheme {
    allocateAddress(addressComponents) {
        return uniqueid_1.makeUniqueId(addressComponents);
    }
}
exports.HashedAddressingScheme = HashedAddressingScheme;
/**
 * Class that keeps track of the logical IDs that are assigned to resources
 *
 * Supports renaming the generated IDs.
 */
class LogicalIDs {
    constructor(namingScheme) {
        this.namingScheme = namingScheme;
        /**
         * The rename table (old to new)
         */
        this.renames = {};
        /**
         * All assigned names (new to old, may be identical)
         *
         * This is used to ensure that:
         *
         * - No 2 resources end up with the same final logical ID, unless they were the same to begin with.
         * - All renames have been used at the end of renaming.
         */
        this.reverse = {};
    }
    /**
     * Rename a logical ID from an old ID to a new ID
     */
    renameLogical(oldId, newId) {
        if (oldId in this.renames) {
            throw new Error(`A rename has already been registered for '${oldId}'`);
        }
        this.renames[oldId] = newId;
    }
    /**
     * Return the logical ID for the given stack element
     */
    getLogicalId(cfnElement) {
        const path = cfnElement.stackPath.split(PATH_SEP);
        const generatedId = this.namingScheme.allocateAddress(path);
        const finalId = this.applyRename(generatedId);
        validateLogicalId(finalId);
        return finalId;
    }
    /**
     * Throw an error if not all renames have been used
     *
     * This is to assure that users didn't make typoes when registering renames.
     */
    assertAllRenamesApplied() {
        const keys = new Set();
        Object.keys(this.renames).forEach(keys.add.bind(keys));
        Object.keys(this.reverse).map(newId => {
            keys.delete(this.reverse[newId]);
        });
        if (keys.size !== 0) {
            const unusedRenames = Array.from(keys.values());
            throw new Error(`The following Logical IDs were attempted to be renamed, but not found: ${unusedRenames.join(', ')}`);
        }
    }
    /**
     * Return the renamed version of an ID, if applicable
     */
    applyRename(oldId) {
        let newId = oldId;
        if (oldId in this.renames) {
            newId = this.renames[oldId];
        }
        // If this newId has already been used, it must have been with the same oldId
        if (newId in this.reverse && this.reverse[newId] !== oldId) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Two objects have been assigned the same Logical ID: '${this.reverse[newId]}' and '${oldId}' are now both named '${newId}'.`);
        }
        this.reverse[newId] = oldId;
        return newId;
    }
}
exports.LogicalIDs = LogicalIDs;
const VALID_LOGICALID_REGEX = /^[A-Za-z][A-Za-z0-9]{1,254}$/;
/**
 * Validate logical ID is valid for CloudFormation
 */
function validateLogicalId(logicalId) {
    if (!VALID_LOGICALID_REGEX.test(logicalId)) {
        throw new Error(`Logical ID must adhere to the regular expression: ${VALID_LOGICALID_REGEX.toString()}, got '${logicalId}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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