"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const instrinsics_1 = require("./instrinsics");
const token_1 = require("./token");
// tslint:disable:max-line-length
/**
 * CloudFormation intrinsic functions.
 * http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html
 */
class Fn {
    /**
     * The ``Fn::GetAtt`` intrinsic function returns the value of an attribute
     * from a resource in the template.
     * @param logicalNameOfResource The logical name (also called logical ID) of
     * the resource that contains the attribute that you want.
     * @param attributeName The name of the resource-specific attribute whose
     * value you want. See the resource's reference page for details about the
     * attributes available for that resource type.
     * @returns a CloudFormationToken object
     */
    static getAtt(logicalNameOfResource, attributeName) {
        return new FnGetAtt(logicalNameOfResource, attributeName);
    }
    /**
     * The intrinsic function ``Fn::Join`` appends a set of values into a single
     * value, separated by the specified delimiter. If a delimiter is the empty
     * string, the set of values are concatenated with no delimiter.
     * @param delimiter The value you want to occur between fragments. The
     * delimiter will occur between fragments only. It will not terminate the
     * final value.
     * @param listOfValues The list of values you want combined.
     * @returns a token represented as a string
     */
    static join(delimiter, listOfValues) {
        if (listOfValues.length === 0) {
            throw new Error(`FnJoin requires at least one value to be provided`);
        }
        return new FnJoin(delimiter, listOfValues).toString();
    }
    /**
     * To split a string into a list of string values so that you can select an element from the
     * resulting string list, use the ``Fn::Split`` intrinsic function. Specify the location of splits
     * with a delimiter, such as , (a comma). After you split a string, use the ``Fn::Select`` function
     * to pick a specific element.
     * @param delimiter A string value that determines where the source string is divided.
     * @param source The string value that you want to split.
     * @returns a token represented as a string array
     */
    static split(delimiter, source) {
        // short-circut if source is not a token
        if (!token_1.Token.isToken(source)) {
            return source.split(delimiter);
        }
        return new FnSplit(delimiter, source).toList();
    }
    /**
     * The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.
     * @param index The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
     * @param array The list of objects to select from. This list must not be null, nor can it have null entries.
     * @returns a token represented as a string
     */
    static select(index, array) {
        if (!token_1.Token.isToken(array)) {
            return array[index];
        }
        return new FnSelect(index, array).toString();
    }
    /**
     * The intrinsic function ``Fn::Sub`` substitutes variables in an input string
     * with values that you specify. In your templates, you can use this function
     * to construct commands or outputs that include values that aren't available
     * until you create or update a stack.
     * @param body A string with variables that AWS CloudFormation substitutes
     * with their associated values at runtime. Write variables as ${MyVarName}.
     * Variables can be template parameter names, resource logical IDs, resource
     * attributes, or a variable in a key-value map. If you specify only template
     * parameter names, resource logical IDs, and resource attributes, don't
     * specify a key-value map.
     * @param variables The name of a variable that you included in the String
     * parameter. The value that AWS CloudFormation substitutes for the associated
     * variable name at runtime.
     * @returns a token represented as a string
     */
    static sub(body, variables) {
        return new FnSub(body, variables).toString();
    }
    /**
     * The intrinsic function ``Fn::Base64`` returns the Base64 representation of
     * the input string. This function is typically used to pass encoded data to
     * Amazon EC2 instances by way of the UserData property.
     * @param data The string value you want to convert to Base64.
     * @returns a token represented as a string
     */
    static base64(data) {
        return new FnBase64(data).toString();
    }
    /**
     * The intrinsic function ``Fn::Cidr`` returns the specified Cidr address block.
     * @param ipBlock  The user-specified default Cidr address block.
     * @param count  The number of subnets' Cidr block wanted. Count can be 1 to 256.
     * @param sizeMask The digit covered in the subnet.
     * @returns a token represented as a string
     */
    static cidr(ipBlock, count, sizeMask) {
        return new FnCidr(ipBlock, count, sizeMask).toList();
    }
    /**
     * The intrinsic function ``Fn::GetAZs`` returns an array that lists
     * Availability Zones for a specified region. Because customers have access to
     * different Availability Zones, the intrinsic function ``Fn::GetAZs`` enables
     * template authors to write templates that adapt to the calling user's
     * access. That way you don't have to hard-code a full list of Availability
     * Zones for a specified region.
     * @param region The name of the region for which you want to get the
     * Availability Zones. You can use the AWS::Region pseudo parameter to specify
     * the region in which the stack is created. Specifying an empty string is
     * equivalent to specifying AWS::Region.
     * @returns a token represented as a string array
     */
    static getAZs(region) {
        return new FnGetAZs(region).toList();
    }
    /**
     * The intrinsic function ``Fn::ImportValue`` returns the value of an output
     * exported by another stack. You typically use this function to create
     * cross-stack references. In the following example template snippets, Stack A
     * exports VPC security group values and Stack B imports them.
     * @param sharedValueToImport The stack output value that you want to import.
     * @returns a token represented as a string
     */
    static importValue(sharedValueToImport) {
        return new FnImportValue(sharedValueToImport).toString();
    }
    /**
     * The intrinsic function ``Fn::FindInMap`` returns the value corresponding to
     * keys in a two-level map that is declared in the Mappings section.
     * @returns a token represented as a string
     */
    static findInMap(mapName, topLevelKey, secondLevelKey) {
        return new FnFindInMap(mapName, topLevelKey, secondLevelKey).toString();
    }
    /**
     * Returns true if all the specified conditions evaluate to true, or returns
     * false if any one of the conditions evaluates to false. ``Fn::And`` acts as
     * an AND operator. The minimum number of conditions that you can include is
     * 2, and the maximum is 10.
     * @param conditions conditions to AND
     * @returns an FnCondition token
     */
    static conditionAnd(...conditions) {
        return new FnAnd(...conditions);
    }
    /**
     * Compares if two values are equal. Returns true if the two values are equal
     * or false if they aren't.
     * @param lhs A value of any type that you want to compare.
     * @param rhs A value of any type that you want to compare.
     * @returns an FnCondition token
     */
    static conditionEquals(lhs, rhs) {
        return new FnEquals(lhs, rhs);
    }
    /**
     * Returns one value if the specified condition evaluates to true and another
     * value if the specified condition evaluates to false. Currently, AWS
     * CloudFormation supports the ``Fn::If`` intrinsic function in the metadata
     * attribute, update policy attribute, and property values in the Resources
     * section and Outputs sections of a template. You can use the AWS::NoValue
     * pseudo parameter as a return value to remove the corresponding property.
     * @param conditionId A reference to a condition in the Conditions section. Use
     * the condition's name to reference it.
     * @param valueIfTrue A value to be returned if the specified condition
     * evaluates to true.
     * @param valueIfFalse A value to be returned if the specified condition
     * evaluates to false.
     * @returns an FnCondition token
     */
    static conditionIf(conditionId, valueIfTrue, valueIfFalse) {
        return new FnIf(conditionId, valueIfTrue, valueIfFalse);
    }
    /**
     * Returns true for a condition that evaluates to false or returns false for a
     * condition that evaluates to true. ``Fn::Not`` acts as a NOT operator.
     * @param condition A condition such as ``Fn::Equals`` that evaluates to true
     * or false.
     * @returns an FnCondition token
     */
    static conditionNot(condition) {
        return new FnNot(condition);
    }
    /**
     * Returns true if any one of the specified conditions evaluate to true, or
     * returns false if all of the conditions evaluates to false. ``Fn::Or`` acts
     * as an OR operator. The minimum number of conditions that you can include is
     * 2, and the maximum is 10.
     * @param conditions conditions that evaluates to true or false.
     * @returns an FnCondition token
     */
    static conditionOr(...conditions) {
        return new FnOr(...conditions);
    }
    /**
     * Returns true if a specified string matches at least one value in a list of
     * strings.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     * @returns an FnCondition token
     */
    static conditionContains(listOfStrings, value) {
        return new FnContains(listOfStrings, value);
    }
    /**
     * Returns true if a specified string matches all values in a list.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list
     * of strings.
     * @returns an FnCondition token
     */
    conditionEachMemberEquals(listOfStrings, value) {
        return new FnEachMemberEquals(listOfStrings, value);
    }
    /**
     * Returns true if each member in a list of strings matches at least one value
     * in a second list of strings.
     * @param stringsToCheck A list of strings, such as "A", "B", "C". AWS
     * CloudFormation checks whether each member in the strings_to_check parameter
     * is in the strings_to_match parameter.
     * @param stringsToMatch A list of strings, such as "A", "B", "C". Each member
     * in the strings_to_match parameter is compared against the members of the
     * strings_to_check parameter.
     * @returns an FnCondition token
     */
    conditionEachMemberIn(stringsToCheck, stringsToMatch) {
        return new FnEachMemberIn(stringsToCheck, stringsToMatch);
    }
    /**
     * Returns all values for a specified parameter type.
     * @param parameterType An AWS-specific parameter type, such as
     * AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see
     * Parameters in the AWS CloudFormation User Guide.
     * @returns a token represented as a string array
     */
    refAll(parameterType) {
        return new FnRefAll(parameterType).toList();
    }
    /**
     * Returns an attribute value or list of values for a specific parameter and
     * attribute.
     * @param parameterOrLogicalId The name of a parameter for which you want to
     * retrieve attribute values. The parameter must be declared in the Parameters
     * section of the template.
     * @param attribute The name of an attribute from which you want to retrieve a
     * value.
     * @returns a token represented as a string
     */
    valueOf(parameterOrLogicalId, attribute) {
        return new FnValueOf(parameterOrLogicalId, attribute).toString();
    }
    /**
     * Returns a list of all attribute values for a given parameter type and
     * attribute.
     * @param parameterType An AWS-specific parameter type, such as
     * AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see
     * Parameters in the AWS CloudFormation User Guide.
     * @param attribute The name of an attribute from which you want to retrieve a
     * value. For more information about attributes, see Supported Attributes.
     * @returns a token represented as a string array
     */
    valueOfAll(parameterType, attribute) {
        return new FnValueOfAll(parameterType, attribute).toList();
    }
}
exports.Fn = Fn;
/**
 * Base class for tokens that represent CloudFormation intrinsic functions.
 */
class FnBase extends token_1.Token {
    constructor(name, value) {
        super({ [name]: value });
    }
}
/**
 * The intrinsic function ``Fn::FindInMap`` returns the value corresponding to keys in a two-level
 * map that is declared in the Mappings section.
 */
class FnFindInMap extends FnBase {
    /**
     * Creates an ``Fn::FindInMap`` function.
     * @param mapName The logical name of a mapping declared in the Mappings section that contains the keys and values.
     * @param topLevelKey The top-level key name. Its value is a list of key-value pairs.
     * @param secondLevelKey The second-level key name, which is set to one of the keys from the list assigned to TopLevelKey.
     */
    constructor(mapName, topLevelKey, secondLevelKey) {
        super('Fn::FindInMap', [mapName, topLevelKey, secondLevelKey]);
    }
}
/**
 * The ``Fn::GetAtt`` intrinsic function returns the value of an attribute from a resource in the template.
 */
class FnGetAtt extends FnBase {
    /**
     * Creates a ``Fn::GetAtt`` function.
     * @param logicalNameOfResource The logical name (also called logical ID) of the resource that contains the attribute that you want.
     * @param attributeName The name of the resource-specific attribute whose value you want. See the resource's reference page for details about the attributes available for that resource type.
     */
    constructor(logicalNameOfResource, attributeName) {
        super('Fn::GetAtt', [logicalNameOfResource, attributeName]);
    }
}
/**
 * The intrinsic function ``Fn::GetAZs`` returns an array that lists Availability Zones for a
 * specified region. Because customers have access to different Availability Zones, the intrinsic
 * function ``Fn::GetAZs`` enables template authors to write templates that adapt to the calling
 * user's access. That way you don't have to hard-code a full list of Availability Zones for a
 * specified region.
 */
class FnGetAZs extends FnBase {
    /**
     * Creates an ``Fn::GetAZs`` function.
     * @param region The name of the region for which you want to get the Availability Zones.
     *         You can use the AWS::Region pseudo parameter to specify the region in
     *         which the stack is created. Specifying an empty string is equivalent to
     *         specifying AWS::Region.
     */
    constructor(region) {
        super('Fn::GetAZs', region || '');
    }
}
/**
 * The intrinsic function ``Fn::ImportValue`` returns the value of an output exported by another stack.
 * You typically use this function to create cross-stack references. In the following example
 * template snippets, Stack A exports VPC security group values and Stack B imports them.
 */
class FnImportValue extends FnBase {
    /**
     * Creates an ``Fn::ImportValue`` function.
     * @param sharedValueToImport The stack output value that you want to import.
     */
    constructor(sharedValueToImport) {
        super('Fn::ImportValue', sharedValueToImport);
    }
}
/**
 * The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.
 */
class FnSelect extends FnBase {
    /**
     * Creates an ``Fn::Select`` function.
     * @param index The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
     * @param array The list of objects to select from. This list must not be null, nor can it have null entries.
     */
    constructor(index, array) {
        super('Fn::Select', [index, array]);
    }
}
/**
 * To split a string into a list of string values so that you can select an element from the
 * resulting string list, use the ``Fn::Split`` intrinsic function. Specify the location of splits
 * with a delimiter, such as , (a comma). After you split a string, use the ``Fn::Select`` function
 * to pick a specific element.
 */
class FnSplit extends FnBase {
    /**
     * Create an ``Fn::Split`` function.
     * @param delimiter A string value that determines where the source string is divided.
     * @param source The string value that you want to split.
     */
    constructor(delimiter, source) {
        super('Fn::Split', [delimiter, source]);
    }
}
/**
 * The intrinsic function ``Fn::Sub`` substitutes variables in an input string with values that
 * you specify. In your templates, you can use this function to construct commands or outputs
 * that include values that aren't available until you create or update a stack.
 */
class FnSub extends FnBase {
    /**
     * Creates an ``Fn::Sub`` function.
     * @param body A string with variables that AWS CloudFormation substitutes with their
     *       associated values at runtime. Write variables as ${MyVarName}. Variables
     *       can be template parameter names, resource logical IDs, resource attributes,
     *       or a variable in a key-value map. If you specify only template parameter names,
     *       resource logical IDs, and resource attributes, don't specify a key-value map.
     * @param variables The name of a variable that you included in the String parameter.
     *          The value that AWS CloudFormation substitutes for the associated variable name at runtime.
     */
    constructor(body, variables) {
        super('Fn::Sub', variables ? [body, variables] : body);
    }
}
/**
 * The intrinsic function ``Fn::Base64`` returns the Base64 representation of the input string.
 * This function is typically used to pass encoded data to Amazon EC2 instances by way of
 * the UserData property.
 */
class FnBase64 extends FnBase {
    /**
     * Creates an ``Fn::Base64`` function.
     * @param data The string value you want to convert to Base64.
     */
    constructor(data) {
        super('Fn::Base64', data);
    }
}
/**
 * The intrinsic function ``Fn::Cidr`` returns the specified Cidr address block.
 */
class FnCidr extends FnBase {
    /**
     * Creates an ``Fn::Cidr`` function.
     * @param ipBlock  The user-specified default Cidr address block.
     * @param count  The number of subnets' Cidr block wanted. Count can be 1 to 256.
     * @param sizeMask The digit covered in the subnet.
     */
    constructor(ipBlock, count, sizeMask) {
        if (count < 1 || count > 256) {
            throw new Error(`Fn::Cidr's count attribute must be betwen 1 and 256, ${count} was provided.`);
        }
        super('Fn::Cidr', [ipBlock, count, sizeMask]);
    }
}
class FnConditionBase extends token_1.Token {
    constructor(type, value) {
        super({ [type]: value });
    }
}
/**
 * Returns true if all the specified conditions evaluate to true, or returns false if any one
 *  of the conditions evaluates to false. ``Fn::And`` acts as an AND operator. The minimum number of
 * conditions that you can include is 2, and the maximum is 10.
 */
class FnAnd extends FnConditionBase {
    constructor(...condition) {
        super('Fn::And', condition);
    }
}
/**
 * Compares if two values are equal. Returns true if the two values are equal or false
 * if they aren't.
 */
class FnEquals extends FnConditionBase {
    /**
     * Creates an ``Fn::Equals`` condition function.
     * @param lhs A value of any type that you want to compare.
     * @param rhs A value of any type that you want to compare.
     */
    constructor(lhs, rhs) {
        super('Fn::Equals', [lhs, rhs]);
    }
}
/**
 * Returns one value if the specified condition evaluates to true and another value if the
 * specified condition evaluates to false. Currently, AWS CloudFormation supports the ``Fn::If``
 * intrinsic function in the metadata attribute, update policy attribute, and property values
 * in the Resources section and Outputs sections of a template. You can use the AWS::NoValue
 * pseudo parameter as a return value to remove the corresponding property.
 */
class FnIf extends FnConditionBase {
    /**
     * Creates an ``Fn::If`` condition function.
     * @param condition A reference to a condition in the Conditions section. Use the condition's name to reference it.
     * @param valueIfTrue A value to be returned if the specified condition evaluates to true.
     * @param valueIfFalse A value to be returned if the specified condition evaluates to false.
     */
    constructor(condition, valueIfTrue, valueIfFalse) {
        super('Fn::If', [condition, valueIfTrue, valueIfFalse]);
    }
}
/**
 * Returns true for a condition that evaluates to false or returns false for a condition that evaluates to true.
 * ``Fn::Not`` acts as a NOT operator.
 */
class FnNot extends FnConditionBase {
    /**
     * Creates an ``Fn::Not`` condition function.
     * @param condition A condition such as ``Fn::Equals`` that evaluates to true or false.
     */
    constructor(condition) {
        super('Fn::Not', [condition]);
    }
}
/**
 * Returns true if any one of the specified conditions evaluate to true, or returns false if
 * all of the conditions evaluates to false. ``Fn::Or`` acts as an OR operator. The minimum number
 * of conditions that you can include is 2, and the maximum is 10.
 */
class FnOr extends FnConditionBase {
    /**
     * Creates an ``Fn::Or`` condition function.
     * @param condition A condition that evaluates to true or false.
     */
    constructor(...condition) {
        super('Fn::Or', condition);
    }
}
/**
 * Returns true if a specified string matches at least one value in a list of strings.
 */
class FnContains extends FnConditionBase {
    /**
     * Creates an ``Fn::Contains`` function.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     */
    constructor(listOfStrings, value) {
        super('Fn::Contains', [listOfStrings, value]);
    }
}
/**
 * Returns true if a specified string matches all values in a list.
 */
class FnEachMemberEquals extends FnConditionBase {
    /**
     * Creates an ``Fn::EachMemberEquals`` function.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     */
    constructor(listOfStrings, value) {
        super('Fn::EachMemberEquals', [listOfStrings, value]);
    }
}
/**
 * Returns true if each member in a list of strings matches at least one value in a second
 * list of strings.
 */
class FnEachMemberIn extends FnConditionBase {
    /**
     * Creates an ``Fn::EachMemberIn`` function.
     * @param stringsToCheck A list of strings, such as "A", "B", "C". AWS CloudFormation checks whether each member in the strings_to_check parameter is in the strings_to_match parameter.
     * @param stringsToMatch A list of strings, such as "A", "B", "C". Each member in the strings_to_match parameter is compared against the members of the strings_to_check parameter.
     */
    constructor(stringsToCheck, stringsToMatch) {
        super('Fn::EachMemberIn', [[stringsToCheck], stringsToMatch]);
    }
}
/**
 * Returns all values for a specified parameter type.
 */
class FnRefAll extends FnBase {
    /**
     * Creates an ``Fn::RefAll`` function.
     * @param parameterType An AWS-specific parameter type, such as AWS::EC2::SecurityGroup::Id or
     *            AWS::EC2::VPC::Id. For more information, see Parameters in the AWS
     *            CloudFormation User Guide.
     */
    constructor(parameterType) {
        super('Fn::RefAll', parameterType);
    }
}
/**
 * Returns an attribute value or list of values for a specific parameter and attribute.
 */
class FnValueOf extends FnBase {
    /**
     * Creates an ``Fn::ValueOf`` function.
     * @param parameterOrLogicalId The name of a parameter for which you want to retrieve attribute values. The parameter must be declared in the Parameters section of the template.
     * @param attribute The name of an attribute from which you want to retrieve a value.
     */
    constructor(parameterOrLogicalId, attribute) {
        super('Fn::ValueOf', [parameterOrLogicalId, attribute]);
    }
}
/**
 * Returns a list of all attribute values for a given parameter type and attribute.
 */
class FnValueOfAll extends FnBase {
    /**
     * Creates an ``Fn::ValueOfAll`` function.
     * @param parameterType An AWS-specific parameter type, such as AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see Parameters in the AWS CloudFormation User Guide.
     * @param attribute The name of an attribute from which you want to retrieve a value. For more information about attributes, see Supported Attributes.
     */
    constructor(parameterType, attribute) {
        super('Fn::ValueOfAll', [parameterType, attribute]);
    }
}
/**
 * The intrinsic function ``Fn::Join`` appends a set of values into a single value, separated by
 * the specified delimiter. If a delimiter is the empty string, the set of values are concatenated
 * with no delimiter.
 */
class FnJoin extends token_1.Token {
    /**
     * Creates an ``Fn::Join`` function.
     * @param delimiter The value you want to occur between fragments. The delimiter will occur between fragments only.
     *          It will not terminate the final value.
     * @param listOfValues The list of values you want combined.
     */
    constructor(delimiter, listOfValues) {
        if (listOfValues.length === 0) {
            throw new Error(`FnJoin requires at least one value to be provided`);
        }
        super();
        this.delimiter = delimiter;
        this.listOfValues = listOfValues;
    }
    resolve(context) {
        if (token_1.Token.isToken(this.listOfValues)) {
            // This is a list token, don't try to do smart things with it.
            return { 'Fn::Join': [this.delimiter, this.listOfValues] };
        }
        const resolved = this.resolveValues(context);
        if (resolved.length === 1) {
            return resolved[0];
        }
        return { 'Fn::Join': [this.delimiter, resolved] };
    }
    /**
     * Optimization: if an Fn::Join is nested in another one and they share the same delimiter, then flatten it up. Also,
     * if two concatenated elements are literal strings (not tokens), then pre-concatenate them with the delimiter, to
     * generate shorter output.
     */
    resolveValues(context) {
        if (this._resolvedValues) {
            return this._resolvedValues;
        }
        const resolvedValues = this.listOfValues.map(context.resolve);
        return this._resolvedValues = instrinsics_1.minimalCloudFormationJoin(this.delimiter, resolvedValues);
    }
}
//# sourceMappingURL=data:application/json;base64,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