"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() { return self._dependencyRoots; },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        DependableTrait.traitMap.set(instance, trait);
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = DependableTrait.traitMap.get(instance);
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
DependableTrait.traitMap = new WeakMap();
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,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