"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 */
class ContextProvider {
    constructor(context, provider, props = {}) {
        this.context = context;
        this.provider = provider;
        this.props = {
            account: context.node.stack.env.account,
            region: context.node.stack.env.region,
            ...props,
        };
    }
    get key() {
        const propStrings = propsToArray(this.props);
        return `${this.provider}:${propStrings.join(':')}`;
    }
    /**
     * Read a provider value and verify it is not `null`
     */
    getValue(defaultValue) {
        // if account or region is not defined this is probably a test mode, so we just
        // return the default value
        if (!this.props.account || !this.props.region) {
            this.context.node.addError(formatMissingScopeError(this.provider, this.props));
            return defaultValue;
        }
        const value = this.context.node.getContext(this.key);
        if (value != null) {
            return value;
        }
        this.context.node.stack.reportMissingContext(this.key, {
            provider: this.provider,
            props: this.props,
        });
        return defaultValue;
    }
    /**
     * Read a provider value, verifying it's a string
     * @param defaultValue The value to return if there is no value defined for this context key
     */
    getStringValue(defaultValue) {
        // if scope is undefined, this is probably a test mode, so we just
        // return the default value
        if (!this.props.account || !this.props.region) {
            this.context.node.addError(formatMissingScopeError(this.provider, this.props));
            return defaultValue;
        }
        const value = this.context.node.getContext(this.key);
        if (value != null) {
            if (typeof value !== 'string') {
                throw new TypeError(`Expected context parameter '${this.key}' to be a string, but got '${JSON.stringify(value)}'`);
            }
            return value;
        }
        this.context.node.stack.reportMissingContext(this.key, {
            provider: this.provider,
            props: this.props,
        });
        return defaultValue;
    }
    /**
     * Read a provider value, verifying it's a list
     * @param defaultValue The value to return if there is no value defined for this context key
     */
    getStringListValue(defaultValue) {
        // if scope is undefined, this is probably a test mode, so we just
        // return the default value and report an error so this in not accidentally used
        // in the toolkit
        if (!this.props.account || !this.props.region) {
            this.context.node.addError(formatMissingScopeError(this.provider, this.props));
            return defaultValue;
        }
        const value = this.context.node.getContext(this.key);
        if (value != null) {
            if (!value.map) {
                throw new Error(`Context value '${this.key}' is supposed to be a list, got '${JSON.stringify(value)}'`);
            }
            return value;
        }
        this.context.node.stack.reportMissingContext(this.key, {
            provider: this.provider,
            props: this.props,
        });
        return defaultValue;
    }
}
exports.ContextProvider = ContextProvider;
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
/**
 * Context provider that will return the availability zones for the current account and region
 */
class AvailabilityZoneProvider {
    constructor(context) {
        this.provider = new ContextProvider(context, cxapi.AVAILABILITY_ZONE_PROVIDER);
    }
    /**
     * Return the list of AZs for the current account and region
     */
    get availabilityZones() {
        return this.provider.getStringListValue(['dummy1a', 'dummy1b', 'dummy1c']);
    }
}
exports.AvailabilityZoneProvider = AvailabilityZoneProvider;
/**
 * Context provider that will read values from the SSM parameter store in the indicated account and region
 */
class SSMParameterProvider {
    constructor(context, props) {
        this.provider = new ContextProvider(context, cxapi.SSM_PARAMETER_PROVIDER, props);
    }
    /**
     * Return the SSM parameter string with the indicated key
     */
    parameterValue(defaultValue = 'dummy') {
        return this.provider.getStringValue(defaultValue);
    }
}
exports.SSMParameterProvider = SSMParameterProvider;
function formatMissingScopeError(provider, props) {
    let s = `Cannot determine scope for context provider ${provider}`;
    const propsString = Object.keys(props).map(key => (`${key}=${props[key]}`));
    s += ` with props: ${propsString}.`;
    s += '\n';
    s += 'This usually happens when AWS credentials are not available and the default account/region cannot be determined.';
    return s;
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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