"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const cloudformation_lang_1 = require("./cloudformation-lang");
const dependency_1 = require("./dependency");
const resolve_1 = require("./resolve");
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
const uniqueid_1 = require("./uniqueid");
exports.PATH_SEP = '/';
/**
 * Represents the construct node in the scope tree.
 */
class ConstructNode {
    constructor(host, scope, id) {
        this.host = host;
        /**
         * An array of aspects applied to this node
         */
        this.aspects = [];
        /**
         * List of children and their names
         */
        this._children = {};
        this.context = {};
        this._metadata = new Array();
        this.references = new Set();
        this.dependencies = new Set();
        /**
         * If this is set to 'true'. addChild() calls for this construct and any child
         * will fail. This is used to prevent tree mutations during synthesis.
         */
        this._locked = false;
        this.invokedAspects = [];
        id = id || ''; // if undefined, convert to empty string
        this.id = id;
        this.scope = scope;
        // We say that scope is required, but root scopes will bypass the type
        // checks and actually pass in 'undefined'.
        if (scope != null) {
            if (id === '') {
                throw new Error('Only root constructs may have an empty name');
            }
            // Has side effect so must be very last thing in constructor
            scope.node.addChild(host, this.id);
        }
        else {
            // This is a root construct.
            this.id = id;
        }
        // escape any path separators so they don't wreck havoc
        this.id = this._escapePathSeparator(this.id);
        if (token_1.Token.isToken(id)) {
            throw new Error(`Cannot use tokens in construct ID: ${id}`);
        }
    }
    /**
     * The stack the construct is a part of.
     */
    get stack() {
        // Lazy import to break cyclic import
        const stack = require('./stack');
        return this._stack || (this._stack = _lookStackUp(this));
        function _lookStackUp(_this) {
            if (stack.Stack.isStack(_this.host)) {
                return _this.host;
            }
            if (!_this.scope) {
                throw new Error(`No stack could be identified for the construct at path ${_this.path}`);
            }
            return _this.scope.node.stack;
        }
    }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() {
        const components = this.ancestors().slice(1).map(c => c.node.id);
        return components.join(exports.PATH_SEP);
    }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() {
        const components = this.ancestors().slice(1).map(c => c.node.id);
        return components.length > 0 ? uniqueid_1.makeUniqueId(components) : '';
    }
    /**
     * Returns a string with a tree representation of this construct and it's children.
     */
    toTreeString(depth = 0) {
        let out = '';
        for (let i = 0; i < depth; ++i) {
            out += '  ';
        }
        const name = this.id || '';
        out += `${this.typename}${name.length > 0 ? ' [' + name + ']' : ''}\n`;
        for (const child of this.children) {
            out += child.node.toTreeString(depth + 1);
        }
        return out;
    }
    /**
     * Return a descendant by path, or undefined
     *
     * Note that if the original ID of the construct you are looking for contained
     * a '/', then it would have been replaced by '--'.
     *
     * @param path Relative path of a direct or indirect child
     * @returns a child by path or undefined if not found.
     */
    tryFindChild(path) {
        if (path.startsWith(exports.PATH_SEP)) {
            throw new Error('Path must be relative');
        }
        const parts = path.split(exports.PATH_SEP);
        let curr = this.host;
        while (curr != null && parts.length > 0) {
            curr = curr.node._children[parts.shift()];
        }
        return curr;
    }
    /**
     * Return a descendant by path
     *
     * Throws an exception if the descendant is not found.
     *
     * Note that if the original ID of the construct you are looking for contained
     * a '/', then it would have been replaced by '--'.
     *
     * @param path Relative path of a direct or indirect child
     * @returns Child with the given path.
     */
    findChild(path) {
        const ret = this.tryFindChild(path);
        if (!ret) {
            throw new Error(`No child with path: '${path}'`);
        }
        return ret;
    }
    /**
     * All direct children of this construct.
     */
    get children() {
        return Object.values(this._children);
    }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PreOrder) {
        const ret = new Array();
        visit(this.host);
        return ret;
        function visit(node) {
            if (order === ConstructOrder.PreOrder) {
                ret.push(node);
            }
            for (const child of node.node.children) {
                visit(child);
            }
            if (order === ConstructOrder.PostOrder) {
                ret.push(node);
            }
        }
    }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) {
        if (this.children.length > 0) {
            const names = this.children.map(c => c.node.id);
            throw new Error('Cannot set context after children have been added: ' + names.join(','));
        }
        this.context[key] = value;
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or undefined
     */
    getContext(key) {
        const value = this.context[key];
        if (value !== undefined) {
            return value;
        }
        return this.scope && this.scope.node.getContext(key);
    }
    /**
     * Retrieve a value from tree-global context
     *
     * It is an error if the context object is not available.
     */
    requireContext(key) {
        const value = this.getContext(key);
        if (value == null) {
            throw new Error(`You must supply a context value named '${key}'`);
        }
        return value;
    }
    /**
     * An array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() {
        return this._metadata;
    }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param from a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, from) {
        if (data == null) {
            return;
        }
        const trace = this.getContext(cxapi.DISABLE_METADATA_STACK_TRACE) ? undefined : stack_trace_1.createStackTrace(from || this.addMetadata);
        this._metadata.push({ type, data, trace });
    }
    /**
     * Adds a { "aws:cdk:info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     */
    addInfo(message) {
        this.addMetadata(cxapi.INFO_METADATA_KEY, message);
    }
    /**
     * Adds a { warning: <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMetadata(cxapi.WARNING_METADATA_KEY, message);
    }
    /**
     * Adds an { error: <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMetadata(cxapi.ERROR_METADATA_KEY, message);
    }
    /**
     * Invokes 'validate' on all child constructs and then on this construct (depth-first).
     * @returns A list of validation errors. If the list is empty, all constructs are valid.
     */
    validateTree() {
        let errors = new Array();
        for (const child of this.children) {
            errors = errors.concat(child.node.validateTree());
        }
        const localErrors = this.host.validate();
        return errors.concat(localErrors.map(msg => new ValidationError(this.host, msg)));
    }
    /**
     * Run 'prepare()' on all constructs in the tree
     */
    prepareTree() {
        const constructs = this.host.node.findAll(ConstructOrder.PreOrder);
        // Aspects are applied root to leaf
        for (const construct of constructs) {
            construct.node.invokeAspects();
        }
        // Use .reverse() to achieve post-order traversal
        for (const construct of constructs.reverse()) {
            if (Construct.isConstruct(construct)) {
                construct.prepare();
            }
        }
    }
    /**
     * Applies the aspect to this Constructs node
     */
    apply(aspect) {
        this.aspects.push(aspect);
        return;
    }
    /**
     * Return the ancestors (including self) of this Construct up until and
     * excluding the indicated component
     *
     * @param upTo The construct to return the path components relative to, or the
     * entire list of ancestors (including root) if omitted. This construct will
     * not be included in the returned list.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be `this` and the first element is the oldest scope (if `upTo` is not set,
     * it will be the root of the construct tree).
     */
    ancestors(upTo) {
        const ret = new Array();
        let curr = this.host;
        while (curr && curr !== upTo) {
            ret.unshift(curr);
            curr = curr.node && curr.node.scope;
        }
        return ret;
    }
    /**
     * @returns The root of the construct tree.
     */
    get root() {
        return this.ancestors()[0];
    }
    /**
     * Throws if the `props` bag doesn't include the property `name`.
     * In the future we can add some type-checking here, maybe even auto-generate during compilation.
     * @param props The props bag.
     * @param name The name of the required property.
     *
     * @deprecated use ``requireProperty`` from ``@aws-cdk/runtime`` instead.
     */
    required(props, name) {
        if (!(name in props)) {
            throw new Error(`Construct of type ${this.typename} is missing required property: ${name}`);
        }
        const value = props[name];
        return value;
    }
    /**
     * @returns The type name of this node.
     */
    get typename() {
        const ctor = this.host.constructor;
        return ctor.name || 'Construct';
    }
    /**
     * Adds a child construct to this node.
     *
     * @param child The child construct
     * @param childName The type name of the child construct.
     * @returns The resolved path part name of the child
     */
    addChild(child, childName) {
        if (this.locked) {
            // special error if root is locked
            if (!this.path) {
                throw new Error('Cannot add children during synthesis');
            }
            throw new Error(`Cannot add children to "${this.path}" during synthesis`);
        }
        if (childName in this._children) {
            const name = this.id || '';
            throw new Error(`There is already a Construct with name '${childName}' in ${this.typename}${name.length > 0 ? ' [' + name + ']' : ''}`);
        }
        this._children[childName] = child;
    }
    /**
     * Locks this construct from allowing more children to be added. After this
     * call, no more children can be added to this construct or to any children.
     */
    lock() {
        this._locked = true;
    }
    /**
     * Unlocks this costruct and allows mutations (adding children).
     */
    unlock() {
        this._locked = false;
    }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() {
        if (this._locked) {
            return true;
        }
        if (this.scope && this.scope.node.locked) {
            return true;
        }
        return false;
    }
    /**
     * Resolve a tokenized value in the context of the current Construct
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this.host,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    stringifyJson(obj) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj).toString();
    }
    /**
     * Record a reference originating from this construct node
     */
    recordReference(...refs) {
        for (const ref of refs) {
            if (reference_1.Reference.isReference(ref)) {
                this.references.add(ref);
            }
        }
    }
    /**
     * Return all references of the given type originating from this node or any of its children
     */
    findReferences() {
        const ret = new Set();
        function recurse(node) {
            for (const reference of node.references) {
                ret.add({ source: node.host, reference });
            }
            for (const child of node.children) {
                recurse(child.node);
            }
        }
        recurse(this);
        return Array.from(ret);
    }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) {
        for (const dependency of dependencies) {
            this.dependencies.add(dependency);
        }
    }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    findDependencies() {
        const found = new Map(); // Deduplication map
        const ret = new Array();
        for (const source of this.findAll()) {
            for (const dependable of source.node.dependencies) {
                for (const target of dependency_1.DependableTrait.get(dependable).dependencyRoots) {
                    let foundTargets = found.get(source);
                    if (!foundTargets) {
                        found.set(source, foundTargets = new Set());
                    }
                    if (!foundTargets.has(target)) {
                        ret.push({ source, target });
                        foundTargets.add(target);
                    }
                }
            }
        }
        return ret;
    }
    /**
     * Triggers each aspect to invoke visit
     */
    invokeAspects() {
        const descendants = this.findAll();
        for (const aspect of this.aspects) {
            if (this.invokedAspects.includes(aspect)) {
                continue;
            }
            descendants.forEach(member => aspect.visit(member));
            this.invokedAspects.push(aspect);
        }
    }
    /**
     * If the construct ID contains a path separator, it is replaced by double dash (`--`).
     */
    _escapePathSeparator(id) {
        if (!id) {
            return id;
        }
        return id.split(exports.PATH_SEP).join('--');
    }
}
exports.ConstructNode = ConstructNode;
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct {
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x) {
        return x.prepare !== undefined && x.validate !== undefined;
    }
    /**
     * Creates a new construct node.
     *
     * @param scope The scope in which to define this construct
     * @param id The scoped construct ID. Must be unique amongst siblings. If
     * the ID includes a path separator (`/`), then it will be replaced by double
     * dash `--`.
     */
    constructor(scope, id) {
        this.node = new ConstructNode(this, scope, id);
        // Implement IDependable privately
        const self = this;
        dependency_1.DependableTrait.implement(this, {
            get dependencyRoots() { return [self]; },
        });
    }
    /**
     * Returns a string representation of this construct.
     */
    toString() {
        const path = this.node.path;
        return this.node.typename + (path.length > 0 ? ` [${path}]` : '');
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     */
    validate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    prepare() {
        return;
    }
}
exports.Construct = Construct;
class ValidationError {
    constructor(source, message) {
        this.source = source;
        this.message = message;
    }
}
exports.ValidationError = ValidationError;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PreOrder"] = 0] = "PreOrder";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["PostOrder"] = 1] = "PostOrder";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
// Import this _after_ everything else to help node work the classes out in the correct order...
const reference_1 = require("./reference");
//# sourceMappingURL=data:application/json;base64,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