import cxapi = require('@aws-cdk/cx-api');
import { IAspect } from './aspect';
import { IDependable } from './dependency';
import { Token } from './token';
export declare const PATH_SEP = "/";
/**
 * Represents a construct.
 */
export interface IConstruct extends IDependable {
    /**
     * The construct node in the scope tree.
     */
    readonly node: ConstructNode;
}
/**
 * Represents the construct node in the scope tree.
 */
export declare class ConstructNode {
    private readonly host;
    /**
     * Returns the scope in which this construct is defined.
     */
    readonly scope?: IConstruct;
    /**
     * The scoped construct ID
     * This ID is unique amongst all constructs defined in the same scope.
     * To obtain a global unique id for this construct, use `uniqueId`.
     */
    readonly id: string;
    /**
     * An array of aspects applied to this node
     */
    readonly aspects: IAspect[];
    /**
     * List of children and their names
     */
    private readonly _children;
    private readonly context;
    private readonly _metadata;
    private readonly references;
    private readonly dependencies;
    /** Will be used to cache the value of ``this.stack``. */
    private _stack?;
    /**
     * If this is set to 'true'. addChild() calls for this construct and any child
     * will fail. This is used to prevent tree mutations during synthesis.
     */
    private _locked;
    private invokedAspects;
    constructor(host: Construct, scope: IConstruct, id: string);
    /**
     * The stack the construct is a part of.
     */
    readonly stack: import('./stack').Stack;
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    readonly path: string;
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    readonly uniqueId: string;
    /**
     * Returns a string with a tree representation of this construct and it's children.
     */
    toTreeString(depth?: number): string;
    /**
     * Return a descendant by path, or undefined
     *
     * Note that if the original ID of the construct you are looking for contained
     * a '/', then it would have been replaced by '--'.
     *
     * @param path Relative path of a direct or indirect child
     * @returns a child by path or undefined if not found.
     */
    tryFindChild(path: string): IConstruct | undefined;
    /**
     * Return a descendant by path
     *
     * Throws an exception if the descendant is not found.
     *
     * Note that if the original ID of the construct you are looking for contained
     * a '/', then it would have been replaced by '--'.
     *
     * @param path Relative path of a direct or indirect child
     * @returns Child with the given path.
     */
    findChild(path: string): IConstruct;
    /**
     * All direct children of this construct.
     */
    readonly children: IConstruct[];
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order?: ConstructOrder): IConstruct[];
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key: string, value: any): void;
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or undefined
     */
    getContext(key: string): any;
    /**
     * Retrieve a value from tree-global context
     *
     * It is an error if the context object is not available.
     */
    requireContext(key: string): any;
    /**
     * An array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    readonly metadata: cxapi.MetadataEntry[];
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param from a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type: string, data: any, from?: any): void;
    /**
     * Adds a { "aws:cdk:info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     */
    addInfo(message: string): void;
    /**
     * Adds a { warning: <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     */
    addWarning(message: string): void;
    /**
     * Adds an { error: <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message: string): void;
    /**
     * Invokes 'validate' on all child constructs and then on this construct (depth-first).
     * @returns A list of validation errors. If the list is empty, all constructs are valid.
     */
    validateTree(): ValidationError[];
    /**
     * Run 'prepare()' on all constructs in the tree
     */
    prepareTree(): void;
    /**
     * Applies the aspect to this Constructs node
     */
    apply(aspect: IAspect): void;
    /**
     * Return the ancestors (including self) of this Construct up until and
     * excluding the indicated component
     *
     * @param upTo The construct to return the path components relative to, or the
     * entire list of ancestors (including root) if omitted. This construct will
     * not be included in the returned list.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be `this` and the first element is the oldest scope (if `upTo` is not set,
     * it will be the root of the construct tree).
     */
    ancestors(upTo?: Construct): IConstruct[];
    /**
     * @returns The root of the construct tree.
     */
    readonly root: IConstruct;
    /**
     * Throws if the `props` bag doesn't include the property `name`.
     * In the future we can add some type-checking here, maybe even auto-generate during compilation.
     * @param props The props bag.
     * @param name The name of the required property.
     *
     * @deprecated use ``requireProperty`` from ``@aws-cdk/runtime`` instead.
     */
    required(props: any, name: string): any;
    /**
     * @returns The type name of this node.
     */
    readonly typename: string;
    /**
     * Adds a child construct to this node.
     *
     * @param child The child construct
     * @param childName The type name of the child construct.
     * @returns The resolved path part name of the child
     */
    addChild(child: IConstruct, childName: string): void;
    /**
     * Locks this construct from allowing more children to be added. After this
     * call, no more children can be added to this construct or to any children.
     */
    lock(): void;
    /**
     * Unlocks this costruct and allows mutations (adding children).
     */
    unlock(): void;
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    readonly locked: boolean;
    /**
     * Resolve a tokenized value in the context of the current Construct
     */
    resolve(obj: any): any;
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    stringifyJson(obj: any): string;
    /**
     * Record a reference originating from this construct node
     */
    recordReference(...refs: Token[]): void;
    /**
     * Return all references of the given type originating from this node or any of its children
     */
    findReferences(): OutgoingReference[];
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies: IDependable[]): void;
    /**
     * Return all dependencies registered on this node or any of its children
     */
    findDependencies(): Dependency[];
    /**
     * Triggers each aspect to invoke visit
     */
    private invokeAspects;
    /**
     * If the construct ID contains a path separator, it is replaced by double dash (`--`).
     */
    private _escapePathSeparator;
}
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
export declare class Construct implements IConstruct {
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x: IConstruct): x is Construct;
    /**
     * Construct node.
     */
    readonly node: ConstructNode;
    /**
     * Creates a new construct node.
     *
     * @param scope The scope in which to define this construct
     * @param id The scoped construct ID. Must be unique amongst siblings. If
     * the ID includes a path separator (`/`), then it will be replaced by double
     * dash `--`.
     */
    constructor(scope: Construct, id: string);
    /**
     * Returns a string representation of this construct.
     */
    toString(): string;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     */
    protected validate(): string[];
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    protected prepare(): void;
}
export declare class ValidationError {
    readonly source: IConstruct;
    readonly message: string;
    constructor(source: IConstruct, message: string);
}
/**
 * In what order to return constructs
 */
export declare enum ConstructOrder {
    /**
     * Depth-first, pre-order
     */
    PreOrder = 0,
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    PostOrder = 1
}
/**
 * A single dependency
 */
export interface Dependency {
    /**
     * Source the dependency
     */
    readonly source: IConstruct;
    /**
     * Target of the dependency
     */
    readonly target: IConstruct;
}
/**
 * A single dependency
 */
export interface Dependency {
    /**
     * Source the dependency
     */
    readonly source: IConstruct;
    /**
     * Target of the dependency
     */
    readonly target: IConstruct;
}
export interface OutgoingReference {
    readonly source: IConstruct;
    readonly reference: Reference;
}
import { Reference } from './reference';
