"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const instrinsics_1 = require("./instrinsics");
const resolve_1 = require("./resolve");
const token_1 = require("./token");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     */
    static toJSON(obj) {
        // This works in two stages:
        //
        // First, resolve everything. This gets rid of the lazy evaluations, evaluation
        // to the real types of things (for example, would a function return a string, an
        // intrinsic, or a number? We have to resolve to know).
        //
        // We then to through the returned result, identify things that evaluated to
        // CloudFormation intrinsics, and re-wrap those in Tokens that have a
        // toJSON() method returning their string representation. If we then call
        // JSON.stringify() on that result, that gives us essentially the same
        // string that we started with, except with the non-token characters quoted.
        //
        //    {"field": "${TOKEN}"} --> {\"field\": \"${TOKEN}\"}
        //
        // A final resolve() on that string (done by the framework) will yield the string
        // we're after.
        //
        // Resolving and wrapping are done in go using the resolver framework.
        class IntrinsincWrapper extends resolve_1.DefaultTokenResolver {
            constructor() {
                super(CLOUDFORMATION_CONCAT);
            }
            resolveToken(t, context) {
                return wrap(super.resolveToken(t, context));
            }
            resolveString(fragments, context) {
                return wrap(super.resolveString(fragments, context));
            }
            resolveList(l, context) {
                return wrap(super.resolveList(l, context));
            }
        }
        // We need a ResolveContext to get started so return a Token
        return new token_1.Token((ctx) => {
            return JSON.stringify(resolve_1.resolve(obj, {
                scope: ctx.scope,
                resolver: new IntrinsincWrapper()
            }));
        }).toString();
        function wrap(value) {
            return instrinsics_1.isIntrinsic(value) ? new IntrinsicToken(() => deepQuoteStringsForJSON(value)) : value;
        }
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && typeof parts[0] === 'string' && typeof parts[1] === 'string') {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return { 'Fn::Join': ['', instrinsics_1.minimalCloudFormationJoin('', parts)] };
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Token that also stringifies in the toJSON() operation.
 */
class IntrinsicToken extends token_1.Token {
    /**
     * Special handler that gets called when JSON.stringify() is used.
     */
    toJSON() {
        return this.toString();
    }
}
/**
 * Deep escape strings for use in a JSON context
 */
function deepQuoteStringsForJSON(x) {
    if (typeof x === 'string') {
        // Whenever we escape a string we strip off the outermost quotes
        // since we're already in a quoted context.
        const stringified = JSON.stringify(x);
        return stringified.substring(1, stringified.length - 1);
    }
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringsForJSON);
    }
    if (typeof x === 'object') {
        for (const key of Object.keys(x)) {
            x[key] = deepQuoteStringsForJSON(x[key]);
        }
    }
    return x;
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    }
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolve_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
//# sourceMappingURL=data:application/json;base64,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