"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_element_1 = require("./cfn-element");
const util_1 = require("./util");
/**
 * The Rules that define template constraints in an AWS Service Catalog portfolio describe when
 * end users can use the template and which values they can specify for parameters that are declared
 * in the AWS CloudFormation template used to create the product they are attempting to use. Rules
 * are useful for preventing end users from inadvertently specifying an incorrect value.
 * For example, you can add a rule to verify whether end users specified a valid subnet in a
 * given VPC or used m1.small instance types for test environments. AWS CloudFormation uses
 * rules to validate parameter values before it creates the resources for the product.
 *
 * A rule can include a RuleCondition property and must include an Assertions property.
 * For each rule, you can define only one rule condition; you can define one or more asserts within the Assertions property.
 * You define a rule condition and assertions by using rule-specific intrinsic functions.
 *
 * @link https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html
 */
class CfnRule extends cfn_element_1.CfnRefElement {
    /**
     * Creates and adds a rule.
     * @param scope The parent construct.
     * @param props The rule props.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.ruleCondition = props && props.ruleCondition;
        this.assertions = props && props.assertions;
    }
    /**
     * Adds an assertion to the rule.
     * @param condition The expression to evaluation.
     * @param description The description of the assertion.
     */
    addAssertion(condition, description) {
        if (!this.assertions) {
            this.assertions = [];
        }
        this.assertions.push({
            assert: condition,
            assertDescription: description
        });
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Rules: {
                [this.logicalId]: {
                    RuleCondition: this.ruleCondition,
                    Assertions: util_1.capitalizePropertyNames(this, this.assertions)
                }
            }
        };
    }
}
exports.CfnRule = CfnRule;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLXJ1bGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4tcnVsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLCtDQUE4QztBQUU5QyxpQ0FBaUQ7QUF3Q2pEOzs7Ozs7Ozs7Ozs7OztHQWNHO0FBQ0gsTUFBYSxPQUFRLFNBQVEsMkJBQWE7SUFZeEM7Ozs7T0FJRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0I7UUFDNUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQ2xELElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxZQUFZLENBQUMsU0FBa0MsRUFBRSxXQUFtQjtRQUN6RSxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNwQixJQUFJLENBQUMsVUFBVSxHQUFHLEVBQUUsQ0FBQztTQUN0QjtRQUVELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxTQUFTO1lBQ2pCLGlCQUFpQixFQUFFLFdBQVc7U0FDL0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU87WUFDTCxLQUFLLEVBQUU7Z0JBQ0wsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUU7b0JBQ2hCLGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYTtvQkFDakMsVUFBVSxFQUFFLDhCQUF1QixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDO2lCQUMzRDthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXJERCwwQkFxREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQ2ZuQ29uZGl0aW9uRXhwcmVzc2lvbiB9IGZyb20gJy4vY2ZuLWNvbmRpdGlvbic7XG5pbXBvcnQgeyBDZm5SZWZFbGVtZW50IH0gZnJvbSAnLi9jZm4tZWxlbWVudCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuL2NvbnN0cnVjdCc7XG5pbXBvcnQgeyBjYXBpdGFsaXplUHJvcGVydHlOYW1lcyB9IGZyb20gJy4vdXRpbCc7XG5cbi8qKlxuICogQSBydWxlIGNhbiBpbmNsdWRlIGEgUnVsZUNvbmRpdGlvbiBwcm9wZXJ0eSBhbmQgbXVzdCBpbmNsdWRlIGFuIEFzc2VydGlvbnMgcHJvcGVydHkuXG4gKiBGb3IgZWFjaCBydWxlLCB5b3UgY2FuIGRlZmluZSBvbmx5IG9uZSBydWxlIGNvbmRpdGlvbjsgeW91IGNhbiBkZWZpbmUgb25lIG9yIG1vcmUgYXNzZXJ0cyB3aXRoaW4gdGhlIEFzc2VydGlvbnMgcHJvcGVydHkuXG4gKiBZb3UgZGVmaW5lIGEgcnVsZSBjb25kaXRpb24gYW5kIGFzc2VydGlvbnMgYnkgdXNpbmcgcnVsZS1zcGVjaWZpYyBpbnRyaW5zaWMgZnVuY3Rpb25zLlxuICpcbiAqIFlvdSBjYW4gdXNlIHRoZSBmb2xsb3dpbmcgcnVsZS1zcGVjaWZpYyBpbnRyaW5zaWMgZnVuY3Rpb25zIHRvIGRlZmluZSBydWxlIGNvbmRpdGlvbnMgYW5kIGFzc2VydGlvbnM6XG4gKlxuICogIEZuOjpBbmRcbiAqICBGbjo6Q29udGFpbnNcbiAqICBGbjo6RWFjaE1lbWJlckVxdWFsc1xuICogIEZuOjpFYWNoTWVtYmVySW5cbiAqICBGbjo6RXF1YWxzXG4gKiAgRm46OklmXG4gKiAgRm46Ok5vdFxuICogIEZuOjpPclxuICogIEZuOjpSZWZBbGxcbiAqICBGbjo6VmFsdWVPZlxuICogIEZuOjpWYWx1ZU9mQWxsXG4gKlxuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VjYXRhbG9nL2xhdGVzdC9hZG1pbmd1aWRlL3JlZmVyZW5jZS10ZW1wbGF0ZV9jb25zdHJhaW50X3J1bGVzLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDZm5SdWxlUHJvcHMge1xuICAvKipcbiAgICogSWYgdGhlIHJ1bGUgY29uZGl0aW9uIGV2YWx1YXRlcyB0byBmYWxzZSwgdGhlIHJ1bGUgZG9lc24ndCB0YWtlIGVmZmVjdC5cbiAgICogSWYgdGhlIGZ1bmN0aW9uIGluIHRoZSBydWxlIGNvbmRpdGlvbiBldmFsdWF0ZXMgdG8gdHJ1ZSwgZXhwcmVzc2lvbnMgaW4gZWFjaCBhc3NlcnQgYXJlIGV2YWx1YXRlZCBhbmQgYXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBSdWxlJ3MgYXNzZXJ0aW9ucyB3aWxsIGFsd2F5cyB0YWtlIGVmZmVjdC5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bGVDb25kaXRpb24/OiBJQ2ZuQ29uZGl0aW9uRXhwcmVzc2lvbjtcblxuICAvKipcbiAgICogQXNzZXJ0aW9ucyB3aGljaCBkZWZpbmUgdGhlIHJ1bGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYXNzZXJ0aW9ucyBmb3IgdGhlIHJ1bGUuXG4gICAqL1xuICByZWFkb25seSBhc3NlcnRpb25zPzogUnVsZUFzc2VydGlvbltdO1xufVxuXG4vKipcbiAqIFRoZSBSdWxlcyB0aGF0IGRlZmluZSB0ZW1wbGF0ZSBjb25zdHJhaW50cyBpbiBhbiBBV1MgU2VydmljZSBDYXRhbG9nIHBvcnRmb2xpbyBkZXNjcmliZSB3aGVuXG4gKiBlbmQgdXNlcnMgY2FuIHVzZSB0aGUgdGVtcGxhdGUgYW5kIHdoaWNoIHZhbHVlcyB0aGV5IGNhbiBzcGVjaWZ5IGZvciBwYXJhbWV0ZXJzIHRoYXQgYXJlIGRlY2xhcmVkXG4gKiBpbiB0aGUgQVdTIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIHVzZWQgdG8gY3JlYXRlIHRoZSBwcm9kdWN0IHRoZXkgYXJlIGF0dGVtcHRpbmcgdG8gdXNlLiBSdWxlc1xuICogYXJlIHVzZWZ1bCBmb3IgcHJldmVudGluZyBlbmQgdXNlcnMgZnJvbSBpbmFkdmVydGVudGx5IHNwZWNpZnlpbmcgYW4gaW5jb3JyZWN0IHZhbHVlLlxuICogRm9yIGV4YW1wbGUsIHlvdSBjYW4gYWRkIGEgcnVsZSB0byB2ZXJpZnkgd2hldGhlciBlbmQgdXNlcnMgc3BlY2lmaWVkIGEgdmFsaWQgc3VibmV0IGluIGFcbiAqIGdpdmVuIFZQQyBvciB1c2VkIG0xLnNtYWxsIGluc3RhbmNlIHR5cGVzIGZvciB0ZXN0IGVudmlyb25tZW50cy4gQVdTIENsb3VkRm9ybWF0aW9uIHVzZXNcbiAqIHJ1bGVzIHRvIHZhbGlkYXRlIHBhcmFtZXRlciB2YWx1ZXMgYmVmb3JlIGl0IGNyZWF0ZXMgdGhlIHJlc291cmNlcyBmb3IgdGhlIHByb2R1Y3QuXG4gKlxuICogQSBydWxlIGNhbiBpbmNsdWRlIGEgUnVsZUNvbmRpdGlvbiBwcm9wZXJ0eSBhbmQgbXVzdCBpbmNsdWRlIGFuIEFzc2VydGlvbnMgcHJvcGVydHkuXG4gKiBGb3IgZWFjaCBydWxlLCB5b3UgY2FuIGRlZmluZSBvbmx5IG9uZSBydWxlIGNvbmRpdGlvbjsgeW91IGNhbiBkZWZpbmUgb25lIG9yIG1vcmUgYXNzZXJ0cyB3aXRoaW4gdGhlIEFzc2VydGlvbnMgcHJvcGVydHkuXG4gKiBZb3UgZGVmaW5lIGEgcnVsZSBjb25kaXRpb24gYW5kIGFzc2VydGlvbnMgYnkgdXNpbmcgcnVsZS1zcGVjaWZpYyBpbnRyaW5zaWMgZnVuY3Rpb25zLlxuICpcbiAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlY2F0YWxvZy9sYXRlc3QvYWRtaW5ndWlkZS9yZWZlcmVuY2UtdGVtcGxhdGVfY29uc3RyYWludF9ydWxlcy5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBDZm5SdWxlIGV4dGVuZHMgQ2ZuUmVmRWxlbWVudCB7XG4gIC8qKlxuICAgKiBJZiB0aGUgcnVsZSBjb25kaXRpb24gZXZhbHVhdGVzIHRvIGZhbHNlLCB0aGUgcnVsZSBkb2Vzbid0IHRha2UgZWZmZWN0LlxuICAgKiBJZiB0aGUgZnVuY3Rpb24gaW4gdGhlIHJ1bGUgY29uZGl0aW9uIGV2YWx1YXRlcyB0byB0cnVlLCBleHByZXNzaW9ucyBpbiBlYWNoIGFzc2VydCBhcmUgZXZhbHVhdGVkIGFuZCBhcHBsaWVkLlxuICAgKi9cbiAgcHVibGljIHJ1bGVDb25kaXRpb24/OiBJQ2ZuQ29uZGl0aW9uRXhwcmVzc2lvbjtcblxuICAvKipcbiAgICogQXNzZXJ0aW9ucyB3aGljaCBkZWZpbmUgdGhlIHJ1bGUuXG4gICAqL1xuICBwdWJsaWMgYXNzZXJ0aW9ucz86IFJ1bGVBc3NlcnRpb25bXTtcblxuICAvKipcbiAgICogQ3JlYXRlcyBhbmQgYWRkcyBhIHJ1bGUuXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIHByb3BzIFRoZSBydWxlIHByb3BzLlxuICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBDZm5SdWxlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5ydWxlQ29uZGl0aW9uID0gcHJvcHMgJiYgcHJvcHMucnVsZUNvbmRpdGlvbjtcbiAgICB0aGlzLmFzc2VydGlvbnMgPSBwcm9wcyAmJiBwcm9wcy5hc3NlcnRpb25zO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gYXNzZXJ0aW9uIHRvIHRoZSBydWxlLlxuICAgKiBAcGFyYW0gY29uZGl0aW9uIFRoZSBleHByZXNzaW9uIHRvIGV2YWx1YXRpb24uXG4gICAqIEBwYXJhbSBkZXNjcmlwdGlvbiBUaGUgZGVzY3JpcHRpb24gb2YgdGhlIGFzc2VydGlvbi5cbiAgICovXG4gIHB1YmxpYyBhZGRBc3NlcnRpb24oY29uZGl0aW9uOiBJQ2ZuQ29uZGl0aW9uRXhwcmVzc2lvbiwgZGVzY3JpcHRpb246IHN0cmluZykge1xuICAgIGlmICghdGhpcy5hc3NlcnRpb25zKSB7XG4gICAgICB0aGlzLmFzc2VydGlvbnMgPSBbXTtcbiAgICB9XG5cbiAgICB0aGlzLmFzc2VydGlvbnMucHVzaCh7XG4gICAgICBhc3NlcnQ6IGNvbmRpdGlvbixcbiAgICAgIGFzc2VydERlc2NyaXB0aW9uOiBkZXNjcmlwdGlvblxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0Nsb3VkRm9ybWF0aW9uKCk6IG9iamVjdCB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFJ1bGVzOiB7XG4gICAgICAgIFt0aGlzLmxvZ2ljYWxJZF06IHtcbiAgICAgICAgICBSdWxlQ29uZGl0aW9uOiB0aGlzLnJ1bGVDb25kaXRpb24sXG4gICAgICAgICAgQXNzZXJ0aW9uczogY2FwaXRhbGl6ZVByb3BlcnR5TmFtZXModGhpcywgdGhpcy5hc3NlcnRpb25zKVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIEEgcnVsZSBhc3NlcnRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUnVsZUFzc2VydGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgYXNzZXJ0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZXJ0OiBJQ2ZuQ29uZGl0aW9uRXhwcmVzc2lvbjtcblxuICAvKipcbiAgICogVGhlIGFzc2VydGlvbiBkZXNjcmlwdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGFzc2VydERlc2NyaXB0aW9uOiBzdHJpbmc7XG59XG4iXX0=